package com.elitesland.fin.domain.service.account;


import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.account.AccountIocConvert;
import com.elitesland.fin.application.facade.dto.account.AccountIocDTO;
import com.elitesland.fin.application.facade.param.account.AccountIocParam;
import com.elitesland.fin.application.facade.vo.account.AccountIocParamVO;
import com.elitesland.fin.application.facade.vo.account.AccountIocRespVO;
import com.elitesland.fin.entity.account.AccountIoc;
import com.elitesland.fin.entity.account.AccountIocDO;
import com.elitesland.fin.repo.account.AccountIocRepo;
import com.elitesland.fin.repo.account.AccountIocRepoProc;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.AllArgsConstructor;
import lombok.val;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Service
@AllArgsConstructor
public class AccountIocDomainServiceImpl implements AccountIocDomainService {

    private final AccountIocRepo accountIocRepo;
    private final AccountIocRepoProc accountIocRepoProc;

    @Override
    @SysCodeProc
    public PagingVO<AccountIocDTO> search(AccountIocParamVO param) {
        val pagingVo = accountIocRepoProc.search(param);
        return pagingVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveAccountIoc(AccountIoc accountIoc, Optional<AccountIocDTO> optional) {
        if (optional.isPresent()) {
            AccountIocDTO accountIocDTO = optional.get();
            //修改时，不可编辑
            accountIoc.setIoCode(accountIocDTO.getIoCode());
            //修改
            accountIoc.setId(accountIocDTO.getId());
            accountIoc.setCreator(accountIocDTO.getCreator());
            accountIoc.setCreateUserId(accountIocDTO.getCreateUserId());
            accountIoc.setCreateTime(accountIocDTO.getCreateTime());
            accountIoc.setModifyTime(LocalDateTime.now());

        } else {
            accountIoc.setId(null);
            accountIoc.setCreator(null);
            accountIoc.setCreateUserId(null);
            accountIoc.setCreateTime(LocalDateTime.now());
            accountIoc.setModifyUserId(null);
            accountIoc.setUpdater(null);
            accountIoc.setModifyTime(LocalDateTime.now());
        }

        AccountIocDO accountIocDO = AccountIocConvert.INSTANCE.enToDo(accountIoc);
        Long id = accountIocRepo.save(accountIocDO).getId();

        return id;
    }

    /**
     * 根据ID查询账户事务码信息
     *
     * @param id 账户事务码ID
     * @return 账户事务码对象
     */
    @Override
    @SysCodeProc
    public Optional<AccountIocDTO> findById(Long id) {
        return accountIocRepo.findById(id).map(AccountIocConvert.INSTANCE::doToDto);
    }

    @Override
    @SysCodeProc
    public Optional<AccountIocDTO> findByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return Optional.empty();
        }
        AccountIocParamVO param = new AccountIocParamVO();
        param.setIoCode(code);
        JPAQuery<AccountIocDTO> jpaQuery = accountIocRepoProc.select(param);

        return Optional.of(jpaQuery.fetchFirst());
    }

    @Override
    @SysCodeProc
    public List<AccountIocDTO> findByIdBatch(List<Long> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.EMPTY_LIST;
        }
        List<AccountIocDTO> whAreaSettingDTOList = accountIocRepo.findAllById(idList).stream().map(AccountIocConvert.INSTANCE::doToDto).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(whAreaSettingDTOList)) {
            return Collections.EMPTY_LIST;
        }

        return whAreaSettingDTOList;
    }

    @Override
    @SysCodeProc
    public List<AccountIocDTO> findByCodeBatch(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.EMPTY_LIST;
        }
        AccountIocParamVO param = new AccountIocParamVO();
        param.setIoCodeList(codeList);
        JPAQuery<AccountIocDTO> jpaQuery = accountIocRepoProc.select(param);
        List<AccountIocDTO> accountIocDTOS = jpaQuery.fetch();
        if (CollectionUtils.isEmpty(accountIocDTOS)) {
            return Collections.EMPTY_LIST;
        }
        return accountIocDTOS;
    }

    @Override
    @SysCodeProc
    public List<AccountIocDTO> selectByParam(AccountIocParamVO param) {
        JPAQuery<AccountIocDTO> jpaQuery = accountIocRepoProc.select(param);
        List<AccountIocDTO> accountIocDTOS = jpaQuery.fetch();
        if (CollectionUtils.isEmpty(accountIocDTOS)) {
            return Collections.EMPTY_LIST;
        }
        return accountIocDTOS;
    }



    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagByIds(Integer deleteFlag, List<Long> idList) {
        accountIocRepoProc.updateDeleteFlagByIds(deleteFlag, idList).execute();
    }

    @Override
    public List<AccountIocDTO> queryByAccountIocParam(AccountIocParam accountIocParam) {
        return accountIocRepoProc.queryByAccountIocParam(accountIocParam);
    }



}
