package com.elitesland.fin.repo.accountingengine;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.FinAccountPeriodLineConvert;
import com.elitesland.fin.application.facade.dto.accountingengine.FinAccountPeriodDTO;
import com.elitesland.fin.application.facade.dto.accountingengine.FinAccountPeriodLineDTO;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountPeriodParam;
import com.elitesland.fin.domain.entity.accountingengine.FinAccountPeriodDO;
import com.elitesland.fin.domain.entity.accountingengine.QFinAccountPeriodDO;
import com.elitesland.fin.domain.entity.accountingengine.QFinAccountPeriodLineDO;
import com.elitesland.fin.utils.StringUtil;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Component
@RequiredArgsConstructor
public class FinAccountPeriodRepoProc {

    private final JPAQueryFactory jpaQueryFactory;
    private final FinAccountPeriodRepo finAccountPeriodRepo;
    private final FinAccountPeriodLineRepo finAccountPeriodLineRepo;

    private final QFinAccountPeriodDO qFinAccountPeriodDO = QFinAccountPeriodDO.finAccountPeriodDO;
    private final QFinAccountPeriodLineDO qFinAccountPeriodLineDO = QFinAccountPeriodLineDO.finAccountPeriodLineDO;

    public PagingVO<FinAccountPeriodDTO> page(FinAccountPeriodParam finAccountPeriodParam) {
        Predicate where = where(finAccountPeriodParam);
        JPAQuery<FinAccountPeriodDTO> query = select(FinAccountPeriodDTO.class).where(ExpressionUtils.allOf(where));
        finAccountPeriodParam.setPaging(query);
        finAccountPeriodParam.fillOrders(query, qFinAccountPeriodDO);

        List<FinAccountPeriodDTO> finAccountPeriodDTOList = query.fetch();

        List<Long> ids = finAccountPeriodDTOList.stream().map(FinAccountPeriodDTO::getId).collect(Collectors.toList());
        List<FinAccountPeriodLineDTO> finAccountPeriodLineDTOList = FinAccountPeriodLineConvert.INSTANCE.DOToDTO(finAccountPeriodLineRepo.findAllByMasIdIn(ids));

        for (FinAccountPeriodDTO finAccountPeriodDTO : finAccountPeriodDTOList) {
            finAccountPeriodDTO.setDetailList(finAccountPeriodLineDTOList.stream()
                    .filter(finFlexibleValueDO -> finAccountPeriodDTO.getId().equals(finFlexibleValueDO.getMasId()))
                    .collect(Collectors.toList()));
        }

        return PagingVO.<FinAccountPeriodDTO>builder()
                .total(finAccountPeriodDTOList.size())
                .records(finAccountPeriodDTOList)
                .build();
    }

    private Predicate where(FinAccountPeriodParam finAccountPeriodParam) {
        return BaseRepoProc.PredicateBuilder.builder()
                .andEq(finAccountPeriodParam.getId() != null, qFinAccountPeriodDO.id, finAccountPeriodParam.getId())
                .andIn(CollectionUtil.isNotEmpty(finAccountPeriodParam.getAccountPeriodCodeList()), qFinAccountPeriodDO.accountPeriodCode, finAccountPeriodParam.getAccountPeriodCodeList())
                .andLike(StringUtils.isNotEmpty(finAccountPeriodParam.getAccountPeriodCode()), qFinAccountPeriodDO.accountPeriodCode, StringUtil.buildLikeString(finAccountPeriodParam.getAccountPeriodCode()))
                .andLike(StringUtils.isNotEmpty(finAccountPeriodParam.getAccountPeriodName()), qFinAccountPeriodDO.accountPeriodName, StringUtil.buildLikeString(finAccountPeriodParam.getAccountPeriodName()))
                .build();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qFinAccountPeriodDO.id,
                qFinAccountPeriodDO.accountPeriodCode,
                qFinAccountPeriodDO.accountPeriodName,
                qFinAccountPeriodDO.status,

                qFinAccountPeriodDO.createTime,
                qFinAccountPeriodDO.creator,
                qFinAccountPeriodDO.modifyTime,
                qFinAccountPeriodDO.updater
        )).from(qFinAccountPeriodDO);
    }


    public FinAccountPeriodDO findByAccountPeriodCode(String accountPeriodCode) {

        List<FinAccountPeriodDO> finAccountPeriodDOList = finAccountPeriodRepo.findAllByAccountPeriodCode(accountPeriodCode);

        if (CollectionUtils.isNotEmpty(finAccountPeriodDOList)) {
            Assert.equals(finAccountPeriodDOList.size(), 1, "查询到多条会计期间");
            return finAccountPeriodDOList.get(0);
        }
        return null;
    }

    public List<FinAccountPeriodDTO> findByParam(FinAccountPeriodParam finAccountPeriodParam) {

        JPAQuery<FinAccountPeriodDTO> jpaQuery = jpaQueryFactory.select(Projections.bean(FinAccountPeriodDTO.class,
                        qFinAccountPeriodDO.id,
                        qFinAccountPeriodDO.accountPeriodCode,
                        qFinAccountPeriodDO.accountPeriodName,
                        qFinAccountPeriodDO.status,

                        qFinAccountPeriodDO.createTime,
                        qFinAccountPeriodDO.creator,
                        qFinAccountPeriodDO.modifyTime,
                        qFinAccountPeriodDO.updater,
                        qFinAccountPeriodLineDO.periodStyle))
                .from(qFinAccountPeriodDO)
                .innerJoin(qFinAccountPeriodLineDO).on(qFinAccountPeriodDO.id.eq(qFinAccountPeriodLineDO.masId));

        jpaQuery.where(ExpressionUtils.allOf(where(finAccountPeriodParam), whereDetails(finAccountPeriodParam)));
        return jpaQuery.fetch();

    }

    private Predicate whereDetails(FinAccountPeriodParam finAccountPeriodParam) {
        return BaseRepoProc.PredicateBuilder.builder()
                .andIn(CollectionUtil.isNotEmpty(finAccountPeriodParam.getPeriodStyleList()), qFinAccountPeriodLineDO.periodStyle, finAccountPeriodParam.getPeriodStyleList())
                .andBefore(ObjectUtil.isNotNull(finAccountPeriodParam.getTransactionDate()), qFinAccountPeriodLineDO.activeStartTime, finAccountPeriodParam.getTransactionDate())
                .andAfter(ObjectUtil.isNotNull(finAccountPeriodParam.getTransactionDate()), qFinAccountPeriodLineDO.activeEndTime, finAccountPeriodParam.getTransactionDate())
                .build();
    }
}
