package com.elitesland.fin.application.service.paytype;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.elitesland.fin.application.convert.paytype.PayTypeConvert;
import com.elitesland.fin.application.facade.param.paytype.PayTypeSaveParam;
import com.elitesland.fin.application.facade.vo.paytype.PayTypeVO;
import com.elitesland.fin.domain.entity.paytype.PayType;
import com.elitesland.fin.domain.param.paytype.PayTypePageParam;
import com.elitesland.fin.domain.service.payorder.PayOrderDomainService;
import com.elitesland.fin.domain.service.paytype.PayTypeDomainService;
import com.elitesland.fin.domain.service.paytype.PayTypeOuDomainService;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.infr.dto.paytype.PayTypeDTO;
import com.elitesland.fin.infr.dto.paytype.PayTypeOuDTO;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/3/16 10:41
 */
@Service
@RequiredArgsConstructor
public class PayTypeServiceImpl implements PayTypeService {

    private final PayTypeDomainService payTypeDomainService;

    private final PayTypeOuDomainService payTypeOuDomainService;

    private final PayOrderDomainService payOrderDomainService;

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> makeDefault(Long id) {
        return ApiResult.ok(payTypeDomainService.makeDefault(id));
    }

    @Override
    public ApiResult<PagingVO<PayTypeVO>> page(PayTypePageParam payTypePageParam) {
        PagingVO<PayTypeDTO> page = payTypeDomainService.page(payTypePageParam);
        return ApiResult.ok(PayTypeConvert.INSTANCE.convertPage(page));
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> enable(List<Long> ids) {
        List<PayTypeOuDTO> payTypeOuDTOS = payTypeOuDomainService.queryByPayTypeIds(ids);
        Map<Long, PayTypeDTO> dtoMap = payTypeDomainService.queryByIds(ids).stream().collect(Collectors.toMap(PayTypeDTO::getId, t -> t));
        //获取每个付款单类型下面的公司 根据付款单id分组  对应下面公司信息为空需校验
        Map<Long, List<PayTypeOuDTO>> ouMap = payTypeOuDTOS.stream().collect(Collectors.groupingBy(PayTypeOuDTO::getPayTypeId));
        check(dtoMap, ouMap);
        return ApiResult.ok(payTypeDomainService.enable(ids));
    }

    private static void check(Map<Long, PayTypeDTO> dtoMap, Map<Long, List<PayTypeOuDTO>> ouMap) {
        if (CollUtil.isEmpty(ouMap)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未分配公司信息,请分配后再启用");
        } else {
            for (Map.Entry<Long, List<PayTypeOuDTO>> entry : ouMap.entrySet()) {
                if (entry.getValue() != null && dtoMap.get(entry.getKey()) != null) {
                    if (CollectionUtil.isEmpty(ouMap.get(entry.getKey()))) {
                        throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "类型代码" + dtoMap.get(entry.getKey()).getPayTypeCode() + "启用前需要分配公司");
                    }
                }
                if (dtoMap.get(entry.getKey()) != null && dtoMap.get(entry.getKey()).getEnableFlag()) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已启用状态不允许再次启用");
                }
            }
        }
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> stopEnable(List<Long> ids) {
        return ApiResult.ok(payTypeDomainService.stopEnable(ids));
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> save(PayTypeSaveParam payTypeParam) {
        PayType payType = PayTypeConvert.INSTANCE.convert(payTypeParam);
        Long id = payTypeDomainService.save(payType);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> delete(List<Long> ids) {
        List<PayTypeOuDTO> payTypeOuDTOS = payTypeOuDomainService.queryByPayTypeIds(ids);
        Map<Long, List<PayTypeOuDTO>> typeOuDTOMap = payTypeOuDTOS.stream().collect(Collectors.groupingBy(PayTypeOuDTO::getPayTypeId));
        Map<Long, List<PayOrderDTO>> orderMap = payOrderDomainService.queryByPayTypeId(ids).stream().collect(Collectors.groupingBy(PayOrderDTO::getPayTypeId));
        for (Map.Entry<Long, List<PayTypeOuDTO>> entry : typeOuDTOMap.entrySet()) {
            if (entry.getValue() != null && CollUtil.isNotEmpty(entry.getValue())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已分配公司!");
            }
        }
        for (Map.Entry<Long, List<PayOrderDTO>> entry : orderMap.entrySet()) {
            if (entry.getValue() != null && CollUtil.isNotEmpty(entry.getValue())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已关联付款单!");
            }
        }

        return ApiResult.ok(payTypeDomainService.delete(ids));
    }

    @Override
    public ApiResult<List<PayTypeVO>> findAll() {
        List<PayTypeDTO> all = payTypeDomainService.findAll();
        List<PayTypeVO> vo = PayTypeConvert.INSTANCE.dtoToVo(all);
        return ApiResult.ok(vo);
    }
}
