package com.elitesland.fin.application.service.invoice.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.cloudt.apm.common.exception.BusinessException;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitesland.fin.application.convert.invoice.InvoiceConvert;
import com.elitesland.fin.application.convert.saleinv.SaleInvConvert;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplySaveParam;
import com.elitesland.fin.application.service.invoice.InvoiceApplyService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.domain.entity.saleinv.*;
import com.elitesland.fin.dto.invoice.InvoiceApplyRpcDTO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvDtlDTO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvdDtlDTO;
import com.elitesland.fin.infr.repo.saleinv.*;
import com.elitesland.fin.param.saleinv.InvoiceQueryParam;
import com.elitesland.fin.param.saleinv.InvoiceSaveParam;
import com.elitesland.fin.param.saleinv.XforceResult;
import com.elitesland.fin.repo.invoice.InvoiceApplyRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class InvoiceApplyServiceImpl implements InvoiceApplyService {

    //    public final InvoiceService invoiceService;
    private final SeqNumProvider sysNumberRuleService;
    private final SaleInvRepo saleInvRepo;
    private final SaleInvDtlRepoProc saleInvDtlRepoProc;
    private final SaleInvDtlRepo saleInvDtlRepo;
    private final SaleInvdDtlRepo saleInvdDtlRepo;
    private final SaleInvdDtlRepoProc saleInvdDtlRepoProc;
    private final SaleInvRepoProc saleInvRepoProc;
    private final InvoiceApplyRepoProc invoiceApplyRepoProc;

    /**
     * 批量创建发票申请信息
     *
     * @param paramList 发票申请保存参数列表
     * @return 返回创建成功的发票申请rpcDTO列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<InvoiceApplyRpcDTO> create(List<InvoiceApplySaveParam> paramList) {
        List<InvoiceApplyRpcDTO> resultList = new ArrayList<>();
        Map<String, SaleInvDO> saleInvDOMapS = new HashMap<>(); // 用于存储主表数据的临时映射
        Map<String, List<SaleInvDtlDO>> saleInvDtlDOMapS = new HashMap<>(); // 用于存储明细表数据的临时映射

        // 遍历参数列表，进行数据校验和转换，并保存到临时映射中
        paramList.forEach(param -> {
            // 转换参数到业务对象，进行业务字段校验
            SaleInv saleInv = InvoiceConvert.INSTANCE.toSaleInv(param);
            saleInv.check();
            // 检查明细字段数据
            checkDtl(saleInv.getSaleInvDtls());
            // 生成申请编号
            String applyNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.XXFP, null);
            saleInv.setApplyNo(applyNo);
            // 保存表头信息到临时映射
            SaleInvDO convertDO = SaleInvConvert.INSTANCE.convert(saleInv);
            convertDO.setSourceSysNo(param.getSourceSysNo());
            saleInvDOMapS.put(applyNo, convertDO);
            // 保存明细信息到临时映射
            List<SaleInvDtlDO> details = SaleInvConvert.INSTANCE.convertListDO(saleInv.getSaleInvDtls());
            saleInvDtlDOMapS.put(applyNo, details);
            // 更新入参申请编号
            param.setApplyNo(applyNo);
        });

        // 开始保存数据到数据库
        if (!CollectionUtils.isEmpty(saleInvDOMapS)) {
            // 保存主表数据
            List<SaleInvDO> saleInvDOList = saleInvRepo.saveAll(saleInvDOMapS.values());
            if (!CollectionUtils.isEmpty(saleInvDOList)) {
                // 组装主表和申请编号的映射关系，用于关联明细数据
                Map<String, Long> applyNoAndIdMap = new HashMap<>();
                saleInvDOList.forEach(saleInvDO -> {
                    applyNoAndIdMap.put(saleInvDO.getApplyNo(), saleInvDO.getId());
                    // 构建结果对象并添加到结果列表
                    InvoiceApplyRpcDTO result = new InvoiceApplyRpcDTO();
                    result.setApplyId(saleInvDO.getId());
                    result.setApplyNo(saleInvDO.getApplyNo());
                    resultList.add(result);
                });

                // 保存明细表数据
                if (!CollectionUtils.isEmpty(saleInvDtlDOMapS)) {
                    List<SaleInvDtlDO> saleInvDtlDOS = new ArrayList<>();
                    saleInvDtlDOMapS.forEach((key, value) -> {
                        Long masId = applyNoAndIdMap.get(key); // 获取对应的主表ID
                        value.forEach(v -> {
                            v.setMasId(masId);
                            saleInvDtlDOS.add(v);
                        });
                    });
                    // 先删除旧的明细数据，再保存新的明细数据
                    List<Long> collect = saleInvDOList.stream().map(BaseModel::getId).collect(Collectors.toList());
                    saleInvDtlRepoProc.delByMasId(collect);
                    saleInvDtlRepo.saveAll(saleInvDtlDOS);
                }
            }
        }
        return resultList;
    }


    public void checkDtl(List<SaleInvDtl> saleInvDtls) {
        // 明细
        if (CollUtil.isEmpty(saleInvDtls)) {
            throw new BusinessException("明细信息不能为空");
        }
        saleInvDtls.forEach(v -> {
            // 商品
//            Assert.notEmpty(v.getItemCode(), "itemCode is null");
            // 单位
            Assert.notEmpty(v.getUom(), "uom is null");
            // 数量
            Assert.notNull(v.getQty(), "qty is null");
            //【含税单价】
            Assert.notNull(v.getPrice(), "price is null");
            //【含税金额】*显示，计算逻辑计算，置灰；
            Assert.notNull(v.getTotalAmt(), "totalAmt is null");
            //【税率】*显示，根据商品主数据获取销项税率，可修改；
            Assert.notNull(v.getTaxRate(), "taxRate is null");
            //【税额】*显示，计算逻辑计算，置灰；
            Assert.notNull(v.getTaxAmt(), "taxAmt is null");
        });
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void saveInvoice(List<InvoiceSaveParam> param) {
        if (CollUtil.isNotEmpty(param)) {
            saleInvdDtlRepoProc.delByMasId(List.of(param.get(0).getMasId()));
            List<SaleInvdDtlDO> details = InvoiceConvert.INSTANCE.convertInvd(param);
            saleInvdDtlRepo.saveAll(details);
        }
    }

    @Override
    public InvoiceApplyRpcDTO queryIdByApplyNo(String applyNo) {
        SaleInvDO saleInvDO = saleInvRepo.findByApplyNo(applyNo);
        if (ObjectUtil.isNull(saleInvDO)) {
            throw new BusinessException("[TIMS-FIN] " + applyNo + " 订单不存在");
        }
        // 申请单的明细行数据
        InvoiceApplyRpcDTO applyRpcDTO = InvoiceConvert.INSTANCE.doToDTO(saleInvDO);
        List<SaleInvDtlDTO> details = saleInvDtlRepoProc.getList(saleInvDO.getId());
        List<String> paymentNoList = details.stream().map(SaleInvDtlDTO::getSourceNo).collect(Collectors.toList());
        List<Long> paymentIds = details.stream().map(SaleInvDtlDTO::getSourceId).distinct().collect(Collectors.toList());
        applyRpcDTO.setPaymentNoList(paymentNoList);
        applyRpcDTO.setPaymentIds(paymentIds);
        // 申请单的发票行数据
        List<SaleInvdDtlDTO> invd = saleInvdDtlRepoProc.getList(saleInvDO.getId());
        List<String> redState = invd.stream().map(SaleInvdDtlDTO::getRedState).collect(Collectors.toList());
        applyRpcDTO.setRedState(redState);
        return applyRpcDTO;
    }

    @Override
    public List<InvoiceApplyRpcDTO> queryIdBySource(InvoiceQueryParam param) {
        return invoiceApplyRepoProc.queryIdBySource(param);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void updateXforceResult(XforceResult param) {
        saleInvRepoProc.updateExternalInfo(param.getApplyNo(), param.getXforceStatus(), param.getXforceReason());
    }

//    @Override
//    public List<InvoiceRpcDTO> invQueryRpc(List<ApplyInvoiceRpcParam> applyInvoiceParam) {
//        log.info("开票入参:{}", JSON.toJSONString(applyInvoiceParam));
//        List<ApplyInvoiceParam> applySaveParamList = new ArrayList<>();
//        for (ApplyInvoiceRpcParam applyInvoiceRpcParam : applyInvoiceParam) {
//            ApplyInvoiceParam applySaveParam = new ApplyInvoiceParam();
//            BeanUtils.copyProperties(applyInvoiceRpcParam, applySaveParam);
//            applySaveParamList.add(applySaveParam);
//        }
//        var list = invoiceService.invQuery(applySaveParamList);
//        log.info("开票List:{}", JSON.toJSONString(list));
//        if (CollUtil.isNotEmpty(list)) {
//            return list;
//        }
//        return new ArrayList<>();
//    }
}
