package com.elitesland.fin.domain.service.payment;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.fin.application.convert.payment.PaymentRuleConfigConvert;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDTO;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.param.account.AccountParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountParam;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.param.payment.PaymentRuleConfigParam;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditSettingDetailVO;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.payment.PaymentRuleConfigQueryVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.creditaccount.CreditAccountService;
import com.elitesland.fin.application.service.creditaccount.CreditSettingService;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.application.service.redis.MyRedisService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.arorder.ArOrderDomainService;
import com.elitesland.fin.entity.payment.PaymentRuleConfigDO;
import com.elitesland.fin.repo.payment.PaymentRuleConfigRepo;
import com.elitesland.fin.repo.payment.PaymentRuleConfigRepoProc;
import com.elitesland.fin.rocketmq.channel.FinSalMqProcessor;
import com.elitesland.fin.rpc.sale.RmiSaleRpcService;
import com.elitesland.fin.utils.SysUtils;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Service
@AllArgsConstructor
@Transactional
@Slf4j
public class PaymentRuleConfigDomainServiceImpl implements PaymentRuleConfigDomainService {

    private final PaymentRuleConfigRepo paymentRuleConfigRepo;

    private final PaymentRuleConfigRepoProc paymentRuleConfigRepoProc;

    private final CreditAccountService creditAccountService;

    private final AccountService accountService;

    private final RmiSaleRpcService rmiSaleRpcService;

    private final CreditAccountFlowService creditAccountFlowService;

    private final AccountFlowService accountFlowService;

    private final FinSalMqProcessor finSalMqProcessor;

    private final UdcProvider udcProvider;

    private final CreditSettingService creditSettingService;

    private final ArOrderDomainService arOrderDomainService;

    private static final BigDecimal ONE_HUNDRED_BIGDECIMAL = new BigDecimal("100");

    private final UdcProvider sysUdcService;

    private final MyRedisService myRedisService;
    @Override
    @SysCodeProc
    public Optional<PaymentRuleConfigDO> findById(Long id) {
        if (StringUtils.isEmpty(id)) {
            return Optional.empty();
        }
        return paymentRuleConfigRepo.findById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteById(Long id) {
        paymentRuleConfigRepo.deleteById(id);
    }

    @Override
    public void updateDynamically(PaymentRuleConfigDO invSceneConfigDO) {
        paymentRuleConfigRepoProc.updateDynamically(invSceneConfigDO);
    }

    @Override
    public PagingVO<PaymentRuleConfigPageVO> searchPage(PaymentRuleConfigParam param) {
        return paymentRuleConfigRepoProc.searchPage(param);
    }

    @Override
    public List<PaymentRuleConfigQueryVO> findRuleConfigByRuleCode(String ruleCode) {
        List<PaymentRuleConfigDO> sceneConfigDOS = paymentRuleConfigRepo.findAllByRuleCode(ruleCode);
        return sceneConfigDOS.stream().map(PaymentRuleConfigConvert.INSTANCE::do2VO)
                .collect(Collectors.toList());
    }

    @Override
    public PaymentRuleConfigDTO findByRuleCode(String ruleCode) {
        List<PaymentRuleConfigDO> sceneConfigDOS = paymentRuleConfigRepo.findAllByRuleCode(ruleCode);
        if (!CollectionUtils.isEmpty(sceneConfigDOS)) {
            return PaymentRuleConfigConvert.INSTANCE.do2DTO(sceneConfigDOS.get(0));
        } else {
            return null;
        }
    }

    @Override
    public List<PaymentRuleConfigDtlDTO> queryAccountAmount(PaymentRuleConfigParam param) {
        log.info("财务账户查询接口，接收到参数：{}", JSON.toJSONString(param));
        checkQueryAccountAmountParam(param);

        PaymentRuleConfigParam paymentRuleConfigParam = PaymentRuleConfigConvert.INSTANCE.copyParam(param);

        buildPaymentRuleConfigParam(paymentRuleConfigParam);

        List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList = queryPaymentRuleConfigDtlWithRedis(paymentRuleConfigParam);
        Assert.notEmpty(paymentRuleConfigDtlDTOList, "未匹配到支付配置");

        Set<Long> ids = paymentRuleConfigParam.getPaymentRuleConfigDtlIds();
        if (CollectionUtils.isNotEmpty(ids)) {

            Assert.isTrue(paymentRuleConfigDtlDTOList.stream().filter(item -> ids.contains(item.getId())).count() == ids.size(),
                    "财务支付配置和入参支付规则明细id不匹配");
            //设置计算标识
            setCalculateFlag(paymentRuleConfigDtlDTOList, ids);
        }

        checkRebateAmount(paymentRuleConfigParam, paymentRuleConfigDtlDTOList);

        //获取客户组编码
        paymentRuleConfigParam.setObjectGroupCode(rmiSaleRpcService.getCustGroupCodeByCustCode(paymentRuleConfigParam.getObjectCode()));

        //获取账户及金额
        List<PaymentRuleConfigDtlDTO> result = getAccountAndAmount(paymentRuleConfigParam, paymentRuleConfigDtlDTOList);
        if (CollectionUtils.isEmpty(result)) {
            return Collections.emptyList();
        }

        //组装不同账户类型最终要扣减的金额，根据订单传的支付方式按优先级使用
        buildPayList(paymentRuleConfigParam, result);

        //设置可用支付金额和可开票金额
        setAmount(paymentRuleConfigParam, result);
        return result;
    }

    @Override
    //@Transactional(rollbackFor = {Exception.class})
    public List<PaymentRuleConfigDtlDTO> pay(PaymentRuleConfigParam param) {
        log.info("财务支付接口，接收到参数：{}", JSON.toJSONString(param));
        checkPayParam(param);
        //增加超账期检查
        CreditSettingDetailVO detail = creditSettingService.detailNotUdc();
        if(detail!=null&&detail.getPaymentPeriodFlag()!=null&&detail.getPaymentPeriodFlag()){
            if(arOrderDomainService.countOverduePeriodPay(param.getObjectCode(),param.getOuCode())>0){
                throw new BusinessException("支付失败，存在超账期未付款应收单");
            }
        }
        PaymentRuleConfigParam paymentRuleConfigParam = PaymentRuleConfigConvert.INSTANCE.copyParam(param);

        buildPaymentRuleConfigParam(paymentRuleConfigParam);
        //List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList = queryPaymentRuleConfigDtl(paymentRuleConfigParam);
        List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList =queryPaymentRuleConfigDtlWithRedis(paymentRuleConfigParam);
        Assert.notEmpty(paymentRuleConfigDtlDTOList, "未匹配到支付配置");

        Set<Long> ids = paymentRuleConfigParam.getPaymentRuleConfigDtlIds();

        Assert.isTrue(paymentRuleConfigDtlDTOList.stream().filter(item -> ids.contains(item.getId())).count() == ids.size(),
                "财务支付配置和入参支付规则明细id不匹配");

        setCalculateFlag(paymentRuleConfigDtlDTOList, ids);

        checkRebateAmount(paymentRuleConfigParam, paymentRuleConfigDtlDTOList);

        //获取客户组编码
        paymentRuleConfigParam.setObjectGroupCode(rmiSaleRpcService.getCustGroupCodeByCustCode(paymentRuleConfigParam.getObjectCode()));

        //获取账户及金额
        List<PaymentRuleConfigDtlDTO> result = getAccountAndAmount(paymentRuleConfigParam, paymentRuleConfigDtlDTOList);
        log.info("账户可用金额");

        //增加校验：同公司+主客户下  启用   的账户可用金额之和<单据金额，直接报账户余额不足
        //增加校验：各账户可支付金额之和<单据金额，直接报账户余额不足
        if(CollectionUtils.isNotEmpty(result)){
            BigDecimal docAmount = paymentRuleConfigParam.getDocAmount();
            BigDecimal totalAvailableAmount=BigDecimal.ZERO;
            BigDecimal orderAvailableAmount=BigDecimal.ZERO;
            for (PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO : result) {
                if(paymentRuleConfigDtlDTO.getAvailableAmount()!=null){
                    totalAvailableAmount=totalAvailableAmount.add(paymentRuleConfigDtlDTO.getAvailableAmount());
                }
                if(paymentRuleConfigDtlDTO.getOrderAvailableAmount()!=null){
                    orderAvailableAmount=orderAvailableAmount.add(paymentRuleConfigDtlDTO.getOrderAvailableAmount());
                }
            }
            Assert.isFalse(totalAvailableAmount.compareTo(docAmount)<0, "余额不足");
            Assert.isFalse(orderAvailableAmount.compareTo(docAmount)<0, "余额不足");
        }
        //组装不同账户类型最终要扣减的金额，根据订单传的支付方式按优先级使用
        buildPayList(paymentRuleConfigParam, result);

        BigDecimal paymentAmount = result.stream().map(PaymentRuleConfigDtlDTO::getPaymentAmount).reduce(BigDecimal::add).get();
        Assert.isTrue(NumberUtil.equals(paymentAmount, paymentRuleConfigParam.getDocAmount()), "余额不足");

        //设置可用支付金额和可开票金额
        setAmount(paymentRuleConfigParam, result);

        //生成账户流水&&账户扣款
        generateAccountFlow(param, paymentRuleConfigParam, result);

        return result;
    }

    @Override
    public void checkRebateAmount(PaymentRuleConfigParam paymentRuleConfigParam, List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList) {
        if (paymentRuleConfigParam.getRebateAmount() != null) {

            PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO = paymentRuleConfigDtlDTOList.stream()
                    .filter(item -> UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getOptAccountType()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(paymentRuleConfigDtlDTO, "查不到返利账户");
            Assert.isTrue(paymentRuleConfigDtlDTO.isCalculateFlag(), "未使用返利账户请不要指定返利金额");
        }
    }

    @Override
    public void setCalculateFlag(List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList, Set<Long> ids) {
        paymentRuleConfigDtlDTOList.forEach(item -> {
            if (!ids.contains(item.getId())) {
                item.setCalculateFlag(false);
            }
        });
    }

    @Override
    public void generateAccountFlow(PaymentRuleConfigParam param,
                                    PaymentRuleConfigParam paymentRuleConfigParam,
                                    List<PaymentRuleConfigDtlDTO> result) {
        result.stream().forEach(item -> {
            if (NumberUtil.equals(item.getPaymentAmount(), BigDecimal.ZERO)) {
                return;
            }

            if (UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode().equals(item.getAccountType())) {
                //信用账户
                item.setFlowNo(generateCreditAccountFlow(param, paymentRuleConfigParam, item));
                return;
            }
            //非信用账户
            item.setFlowNo(generateAccountFlow(param, paymentRuleConfigParam, item));
        });
    }

    private List<PaymentRuleConfigDtlDTO> getAccountAndAmount(PaymentRuleConfigParam paymentRuleConfigParam,
                                                              List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList) {
        List<PaymentRuleConfigDtlDTO> result = new ArrayList<>();
        Map<String, Set<String>> appUdcCodeQueryMap =new HashMap<>();
        Set<String> set = new HashSet<>();
        set.add(UdcEnum.ACCOUNT_TYPE_STORE.getCode());
        set.add(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode());
        appUdcCodeQueryMap.put(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getModel(),set);
        Map<String, Map<String, Map<String, String>>> udcAllValueMap = sysUdcService.getValueMapByUdcCodeBatch(appUdcCodeQueryMap);
        Map<String, String> accountTypeUdc = udcAllValueMap.get(UdcEnum.ACCOUNT_TYPE_STORE.getModel())!=null?
                udcAllValueMap.get(UdcEnum.ACCOUNT_TYPE_STORE.getModel()).get(UdcEnum.ACCOUNT_TYPE_STORE.getCode()):null;
        Map<String, String> creditAccountTypeUdc = udcAllValueMap.get(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getModel())!=null?
                udcAllValueMap.get(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getModel()).get(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode()):null;
        // 调用系统域，获取金额小数位数
        Integer amtRound = SysUtils.getAmtPlace();

        paymentRuleConfigDtlDTOList.stream().forEach(item -> {
            if (UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode().equals(item.getOptAccountType())) {
                //信用账户
                PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO = buildCreditAccountResult(item, paymentRuleConfigParam, creditAccountTypeUdc, amtRound);
                if (paymentRuleConfigDtlDTO != null) {
                    result.add(paymentRuleConfigDtlDTO);
                }
                return;
            }
            //非信用账户
            PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO = buildAccountResult(item, paymentRuleConfigParam, accountTypeUdc, amtRound);
            if (paymentRuleConfigDtlDTO != null) {
                result.add(paymentRuleConfigDtlDTO);
            }
        });
        return result;
    }

    @Override
    public void setAmount(PaymentRuleConfigParam paymentRuleConfigParam, List<PaymentRuleConfigDtlDTO> result) {

        result.stream().filter(item -> !item.isCalculateFlag()).forEach(item -> item.setOrderAvailableAmount(BigDecimal.ZERO));

        //本订单可用金额之和
        BigDecimal orderTotalAvailableAmount = result.stream()
                .map(PaymentRuleConfigDtlDTO::getOrderAvailableAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        //指定返利金额+本订单可用金额之和
        if (paymentRuleConfigParam.getRebateAmount() != null) {
            orderTotalAvailableAmount = result.stream()
                    .filter(item -> !UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getAccountType()))
                    .map(PaymentRuleConfigDtlDTO::getOrderAvailableAmount)
                    .reduce(BigDecimal.ZERO, BigDecimal::add)
                    .add(paymentRuleConfigParam.getRebateAmount());
        }

        BigDecimal invoiceAmount = result.stream()
                .filter(item -> !UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getAccountType()))
                .map(PaymentRuleConfigDtlDTO::getPaymentAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        for (PaymentRuleConfigDtlDTO item : result) {
            item.setOrderTotalAvailableAmount(orderTotalAvailableAmount);
            item.setInvoiceAmount(invoiceAmount);
        }
    }

    @Override
    public void buildPaymentRuleConfigParam(PaymentRuleConfigParam param) {
        param.setDocType(param.getDoc().concat(FinConstant.UNDERLINE).concat(param.getDocType()));
        param.setDocStatus(param.getDoc().concat(FinConstant.UNDERLINE).concat(param.getDocStatus()));
    }

    @Override
    public void buildPayList(PaymentRuleConfigParam paymentRuleConfigParam,
                             List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList) {
        BigDecimal paymentBalance = paymentRuleConfigParam.getDocAmount();
        BigDecimal rebateAmount = paymentRuleConfigParam.getRebateAmount();


        if (rebateAmount == null) {
            //不指定金额
            for (PaymentRuleConfigDtlDTO item : paymentRuleConfigDtlDTOList) {

                if (item.isCalculateFlag()) {

                    item.setPaymentAmount(item.getOrderAvailableAmount());
                    if (item.getOrderAvailableAmount().compareTo(paymentBalance) >= 0) {
                        item.setPaymentAmount(paymentBalance);
                    }

                    paymentBalance = paymentBalance.subtract(item.getOrderAvailableAmount());
                    paymentBalance = NumberUtil.isGreaterOrEqual(paymentBalance, BigDecimal.ZERO) ? paymentBalance : BigDecimal.ZERO;
                }
            }
            return;
        }

        //指定返利账户金额
        paymentBalance = paymentBalance.subtract(rebateAmount);

        for (PaymentRuleConfigDtlDTO item : paymentRuleConfigDtlDTOList) {

            if (item.isCalculateFlag()) {

                if (UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getAccountType())) {
                    item.setPaymentAmount(rebateAmount);
                    continue;
                }

                item.setPaymentAmount(item.getOrderAvailableAmount());
                if (item.getOrderAvailableAmount().compareTo(paymentBalance) >= 0) {
                    item.setPaymentAmount(paymentBalance);
                }

                paymentBalance = paymentBalance.subtract(item.getOrderAvailableAmount());
                paymentBalance = NumberUtil.isGreaterOrEqual(paymentBalance, BigDecimal.ZERO) ? paymentBalance : BigDecimal.ZERO;
            }
        }
    }

    public List<PaymentRuleConfigDtlDTO> queryPaymentRuleConfigDtl(PaymentRuleConfigParam paymentRuleConfigParam) {
        PaymentRuleConfigParam queryParam = new PaymentRuleConfigParam();
        queryParam.setOptDoc(paymentRuleConfigParam.getDoc());
        queryParam.setOptDocType(paymentRuleConfigParam.getDocType());
        queryParam.setOptDocStatus(paymentRuleConfigParam.getDocStatus());
        queryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList = paymentRuleConfigRepoProc.searchByPaymentRuleConfigParam(queryParam);
        Assert.notEmpty(paymentRuleConfigDtlDTOList, "未匹配到支付配置");

        //根据优先级排序
        paymentRuleConfigDtlDTOList.stream()
                .forEach(item -> item.setCalculateFlag(true));

        paymentRuleConfigDtlDTOList = paymentRuleConfigDtlDTOList.stream()
                .sorted(Comparator.comparing(PaymentRuleConfigDtlDTO::getPriorityNo))
                .collect(Collectors.toList());

        return paymentRuleConfigDtlDTOList;
    }

    public List<PaymentRuleConfigDtlDTO> queryPaymentRuleConfigDtlByRuleCode(String ruleCode) {
        PaymentRuleConfigParam queryParam = new PaymentRuleConfigParam();
        queryParam.setRuleCode(ruleCode);
        queryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList =
                paymentRuleConfigRepoProc.searchByPaymentRuleConfigParam(queryParam);
        if (CollUtil.isEmpty(paymentRuleConfigDtlDTOList)) {
            return Collections.emptyList();
        }
        //根据优先级排序
        paymentRuleConfigDtlDTOList.stream()
                .forEach(item -> item.setCalculateFlag(true));

        paymentRuleConfigDtlDTOList = paymentRuleConfigDtlDTOList.stream()
                .sorted(Comparator.comparing(PaymentRuleConfigDtlDTO::getPriorityNo))
                .collect(Collectors.toList());

        return paymentRuleConfigDtlDTOList;
    }

    @Override
    public List<PaymentRuleConfigDtlDTO> queryPaymentRuleConfigDtlWithRedis(PaymentRuleConfigParam paymentRuleConfigParam) {
        PaymentRuleConfigParam queryParam = new PaymentRuleConfigParam();
        queryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<PaymentRuleConfigDtlDTO> paymentRuleConfigDtlDTOList =myRedisService.getListDataFromRedis(FinRedisConstant.PAYMENT_RULE_CONFIG,PaymentRuleConfigDtlDTO.class,"支付规则配置",()->{
            return paymentRuleConfigRepoProc.searchByPaymentRuleConfigParam(queryParam);});
        if(!CollectionUtils.isEmpty(paymentRuleConfigDtlDTOList)){
            paymentRuleConfigDtlDTOList=paymentRuleConfigDtlDTOList.stream().filter(v->{
                if(org.apache.commons.lang3.StringUtils.isNotEmpty(paymentRuleConfigParam.getDoc())){
                  if(!paymentRuleConfigParam.getDoc().equals(v.getOptDoc())){
                      return false;
                  }
                }
                if(org.apache.commons.lang3.StringUtils.isNotEmpty(paymentRuleConfigParam.getDocType())){
                    if(!paymentRuleConfigParam.getDocType().equals(v.getOptDocType())){
                        return false;
                    }
                }
                if(org.apache.commons.lang3.StringUtils.isNotEmpty(paymentRuleConfigParam.getDocStatus())){
                    if(!paymentRuleConfigParam.getDocStatus().equals(v.getOptDocStatus())){
                        return false;
                    }
                }
                return true;
            }).collect(Collectors.toList());
        }
   /*     queryParam.setOptDoc(paymentRuleConfigParam.getDoc());
        queryParam.setOptDocType(paymentRuleConfigParam.getDocType());
        queryParam.setOptDocStatus(paymentRuleConfigParam.getDocStatus());
        queryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());*/
        Assert.notEmpty(paymentRuleConfigDtlDTOList, "未匹配到支付配置");
        //根据优先级排序
        paymentRuleConfigDtlDTOList.stream()
                .forEach(item -> item.setCalculateFlag(true));

        paymentRuleConfigDtlDTOList = paymentRuleConfigDtlDTOList.stream()
                .sorted(Comparator.comparing(PaymentRuleConfigDtlDTO::getPriorityNo))
                .collect(Collectors.toList());

        return paymentRuleConfigDtlDTOList;
    }

    private String generateAccountFlow(PaymentRuleConfigParam original,
                                       PaymentRuleConfigParam paymentRuleConfigParam,
                                       PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO) {

        AccountFlowParam accountFlowParam = buildAccountFlowParam(original, paymentRuleConfigParam, paymentRuleConfigDtlDTO);
        return accountFlowService.generateAccountFlow(accountFlowParam);
    }

    private AccountFlowParam buildAccountFlowParam(PaymentRuleConfigParam original,
                                                   PaymentRuleConfigParam paymentRuleConfigParam,
                                                   PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO) {

        log.info("补贴/储值账户付款信息：{}", JSONObject.toJSONString(paymentRuleConfigDtlDTO));
        AccountFlowParam accountFlowParam = new AccountFlowParam();
        accountFlowParam.setSourceDoc(paymentRuleConfigParam.getDoc());
        accountFlowParam.setSourceDocType(original.getDocType());
        accountFlowParam.setSourceDocStatus(original.getDocStatus());
        accountFlowParam.setSourceDocAmount(paymentRuleConfigDtlDTO.getPaymentAmount());
        accountFlowParam.setSourceId(paymentRuleConfigParam.getSourceId());
        accountFlowParam.setSourceNo(paymentRuleConfigParam.getSourceNo());
        accountFlowParam.setAccountCode(paymentRuleConfigDtlDTO.getAccountCode());
        accountFlowParam.setWorkflowProcInstId(original.getWorkflowProcInstId());
        accountFlowParam.setTechFee(paymentRuleConfigDtlDTO.getTechFee());
        accountFlowParam.setMarketingFee(paymentRuleConfigDtlDTO.getMarketingFee());
        accountFlowParam.setOperationFee(paymentRuleConfigDtlDTO.getOperationFee());
        accountFlowParam.setItemFee(paymentRuleConfigDtlDTO.getItemFee());
        accountFlowParam.setOuCode(original.getOuCode());
        accountFlowParam.setTOuCode(original.getTOuCode());
        accountFlowParam.setItemFeeOuCode(original.getItemFeeOuCode());
        accountFlowParam.setTechFeeOuCode(original.getTechFeeOuCode());
        accountFlowParam.setOperationFeeOuCode(original.getOperationFeeOuCode());
        accountFlowParam.setMarketingFeeOuCode(original.getMarketingFeeOuCode());
        accountFlowParam.setRootDocNo(original.getRootDocNo());
        accountFlowParam.setRootDoc(original.getRootDoc());
        return accountFlowParam;
    }

    BigDecimal getActualFee(BigDecimal amount, BigDecimal calculatePercent) {
        return amount == null ? null : amount.multiply(calculatePercent).divide(ONE_HUNDRED_BIGDECIMAL);
    }


    private String generateCreditAccountFlow(PaymentRuleConfigParam original,
                                             PaymentRuleConfigParam paymentRuleConfigParam,
                                             PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO) {

        CreditAccountFlowParam creditAccountFlowParam = buildCreditAccountFlowParam(original, paymentRuleConfigParam, paymentRuleConfigDtlDTO);
        return creditAccountFlowService.generateCreditAccountFlow(creditAccountFlowParam);
    }

    private CreditAccountFlowParam buildCreditAccountFlowParam(PaymentRuleConfigParam original,
                                                               PaymentRuleConfigParam paymentRuleConfigParam,
                                                               PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO) {

        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();

        log.info("补贴/储值账户付款信息：{}", JSONObject.toJSONString(paymentRuleConfigDtlDTO));
        creditAccountFlowParam.setSourceDoc(paymentRuleConfigParam.getDoc());
        creditAccountFlowParam.setSourceDocType(original.getDocType());
        creditAccountFlowParam.setSourceDocStatus(original.getDocStatus());
        creditAccountFlowParam.setSourceDocAmount(paymentRuleConfigDtlDTO.getPaymentAmount());
        creditAccountFlowParam.setSourceId(paymentRuleConfigParam.getSourceId());
        creditAccountFlowParam.setSourceNo(paymentRuleConfigParam.getSourceNo());
        creditAccountFlowParam.setCreditAccountCode(paymentRuleConfigDtlDTO.getAccountCode());
        creditAccountFlowParam.setWorkflowProcInstId(original.getWorkflowProcInstId());
        creditAccountFlowParam.setTechFee(paymentRuleConfigDtlDTO.getTechFee());
        creditAccountFlowParam.setMarketingFee(paymentRuleConfigDtlDTO.getMarketingFee());
        creditAccountFlowParam.setOperationFee(paymentRuleConfigDtlDTO.getOperationFee());
        creditAccountFlowParam.setItemFee(paymentRuleConfigDtlDTO.getItemFee());
        creditAccountFlowParam.setOuCode(original.getOuCode());
        creditAccountFlowParam.setTOuCode(original.getTOuCode());
        creditAccountFlowParam.setItemFeeOuCode(original.getItemFeeOuCode());
        creditAccountFlowParam.setTechFeeOuCode(original.getTechFeeOuCode());
        creditAccountFlowParam.setOperationFeeOuCode(original.getOperationFeeOuCode());
        creditAccountFlowParam.setMarketingFeeOuCode(original.getMarketingFeeOuCode());
        creditAccountFlowParam.setRootDocNo(original.getRootDocNo());
        creditAccountFlowParam.setRootDoc(original.getRootDoc());
        return creditAccountFlowParam;
    }


    @Override
    public void checkPayParam(PaymentRuleConfigParam param) {
        Assert.notEmpty(param.getDoc(), "单据不能为空");
        Assert.notEmpty(param.getDocType(), "单据类型不能为空");
        Assert.notEmpty(param.getDocStatus(), "单据状态不能为空");
        Assert.notNull(param.getDocAmount(), "来源单据金额不能为空");
//        Assert.notEmpty(param.getOuCode(), "公司编码不能为空");
        Assert.notEmpty(param.getObjectCode(), "对象编码不能为空");
        Assert.notEmpty(param.getSourceNo(), "来源单号不能为空");
        Assert.notNull(param.getSourceId(), "来源单据id不能为空");
        Assert.notEmpty(param.getPaymentRuleConfigDtlIds(), "支付规则明细id不能为空");
    }

    @Override
    public void checkQueryAccountAmountParam(PaymentRuleConfigParam param) {
        Assert.notEmpty(param.getDoc(), "单据不能为空");
        Assert.notEmpty(param.getDocType(), "单据类型不能为空");
        Assert.notEmpty(param.getDocStatus(), "单据状态不能为空");
        Assert.notNull(param.getDocAmount(), "来源单据金额不能为空");
        Assert.notEmpty(param.getObjectCode(), "对象编码不能为空");
        Assert.notEmpty(param.getOuCode(), "公司编码不能为空");
    }

    @Override
    public PaymentRuleConfigDtlDTO buildAccountResult(PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO,
                                                      PaymentRuleConfigParam paymentRuleConfigParam,
                                                      Map<String, String> accountTypeUdc,
                                                      Integer amtRound) {
        AccountParam accountParam = new AccountParam();
        accountParam.setAccountHolderCode(paymentRuleConfigParam.getObjectCode());
        accountParam.setAccountType(paymentRuleConfigDtlDTO.getOptAccountType());
        accountParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        List<AccountDTO> accountDTOList = accountService.getAccountByAccountParam(accountParam);
        if (CollectionUtils.isEmpty(accountDTOList)) {
            return null;
        }
        Assert.isTrue(accountDTOList.size() == 1, "查询到多条账户");

        AccountDTO accountDTO = accountDTOList.get(0);
        PaymentRuleConfigDtlDTO result = new PaymentRuleConfigDtlDTO();


        result.setCalculateFlag(paymentRuleConfigDtlDTO.isCalculateFlag());
        result.setAvailableAmount(BigDecimal.ZERO);
        result.setOrderAvailableAmount(BigDecimal.ZERO);
        result.setPaymentAmount(BigDecimal.ZERO);
        result.setOrderTotalAvailableAmount(BigDecimal.ZERO);
        result.setInvoiceAmount(BigDecimal.ZERO);

        result.setTime(LocalDateTime.now());
        result.setId(paymentRuleConfigDtlDTO.getId());
        result.setAccountType(accountDTO.getAccountType());
        result.setAccountTypeName(accountTypeUdc.get(accountDTO.getAccountType()));
        result.setAccountCode(accountDTO.getAccountCode());
        result.setAccountName(accountDTO.getAccountName());
        result.setAvailableAmount(accountDTO.getAccountAvailableAmount());
        result.setPriorityNo(paymentRuleConfigDtlDTO.getPriorityNo());

        //本订单可用金额
        if (paymentRuleConfigDtlDTO.getCalculatePercent() == null) {
            paymentRuleConfigDtlDTO.setCalculatePercent(ONE_HUNDRED_BIGDECIMAL);
        }
        result.setCalculatePercent(paymentRuleConfigDtlDTO.getCalculatePercent());

        BigDecimal orderAvailableAmount = paymentRuleConfigParam.getDocAmount()
                .multiply(paymentRuleConfigDtlDTO.getCalculatePercent())
                .divide(ONE_HUNDRED_BIGDECIMAL);

        result.setOrderAvailableAmount(orderAvailableAmount);
        if(accountDTO.getAccountAvailableAmount().compareTo(BigDecimal.ZERO)<=0){
            result.setOrderAvailableAmount(BigDecimal.ZERO);
        }else if (accountDTO.getAccountAvailableAmount().compareTo(orderAvailableAmount) < 0) {
            result.setOrderAvailableAmount(accountDTO.getAccountAvailableAmount());
        }

        result.setOrderAvailableAmount(result.getOrderAvailableAmount().setScale(amtRound, RoundingMode.HALF_UP));

        //指定返利账户支付金额
        if (UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(result.getAccountType()) &&
                paymentRuleConfigParam.getRebateAmount() != null) {
            Assert.isTrue(NumberUtil.isLessOrEqual(paymentRuleConfigParam.getRebateAmount(), result.getOrderAvailableAmount()),
                    "指定返利账户支付金额大于本订单可用金额");
        }

        return result;
    }

    private PaymentRuleConfigDtlDTO buildCreditAccountResult(PaymentRuleConfigDtlDTO paymentRuleConfigDtlDTO,
                                                             PaymentRuleConfigParam paymentRuleConfigParam,
                                                             Map<String, String> creditAccountTypeUdc,
                                                             Integer amtRound) {

        CreditAccountParam creditAccountParam = new CreditAccountParam();
        creditAccountParam.setObjectCode(paymentRuleConfigParam.getObjectGroupCode());
        creditAccountParam.setOuCode(paymentRuleConfigParam.getOuCode());
        creditAccountParam.setCreditAccountType(paymentRuleConfigDtlDTO.getOptAccountType());
        creditAccountParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        List<CreditAccountDTO> creditAccountDTOList = creditAccountService.getAccountByAccountParam(creditAccountParam);
        if (CollectionUtils.isEmpty(creditAccountDTOList)) {
            return null;
        }

        Assert.isTrue(creditAccountDTOList.size() == 1, "查询到多条信用账户");

        CreditAccountDTO creditAccountDTO = creditAccountDTOList.get(0);
        PaymentRuleConfigDtlDTO result = new PaymentRuleConfigDtlDTO();

        result.setAvailableAmount(BigDecimal.ZERO);
        result.setCalculateFlag(paymentRuleConfigDtlDTO.isCalculateFlag());
        result.setOrderAvailableAmount(BigDecimal.ZERO);
        result.setPaymentAmount(BigDecimal.ZERO);
        result.setOrderTotalAvailableAmount(BigDecimal.ZERO);
        result.setInvoiceAmount(BigDecimal.ZERO);

        result.setTime(LocalDateTime.now());
        result.setId(paymentRuleConfigDtlDTO.getId());
        result.setAccountType(creditAccountDTO.getCreditAccountType());
        result.setAccountTypeName(creditAccountTypeUdc.get(creditAccountDTO.getCreditAccountType()));
        result.setAccountCode(creditAccountDTO.getCreditAccountCode());
        result.setAccountName(creditAccountDTO.getCreditAccountName());
        result.setAvailableAmount(creditAccountDTO.getCreditAccountAvailableLimit());
        result.setPriorityNo(paymentRuleConfigDtlDTO.getPriorityNo());

        //本订单可用金额
        if (paymentRuleConfigDtlDTO.getCalculatePercent() == null) {
            paymentRuleConfigDtlDTO.setCalculatePercent(ONE_HUNDRED_BIGDECIMAL);
        }
        result.setCalculatePercent(paymentRuleConfigDtlDTO.getCalculatePercent());

        BigDecimal orderAvailableAmount = paymentRuleConfigParam.getDocAmount()
                .multiply(paymentRuleConfigDtlDTO.getCalculatePercent())
                .divide(ONE_HUNDRED_BIGDECIMAL);

        result.setOrderAvailableAmount(orderAvailableAmount);

        if(creditAccountDTO.getCreditAccountAvailableLimit().compareTo(BigDecimal.ZERO)<=0){
            result.setOrderAvailableAmount(BigDecimal.ZERO);
        }else if(creditAccountDTO.getCreditAccountAvailableLimit().compareTo(orderAvailableAmount) < 0) {
            result.setOrderAvailableAmount(creditAccountDTO.getCreditAccountAvailableLimit());
        }
        result.setOrderAvailableAmount(result.getOrderAvailableAmount().setScale(amtRound, RoundingMode.HALF_UP));

        return result;
    }
}