package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitesland.fin.application.convert.limitadjustorder.LimitAdjustOrderConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.excel.adjust.CreditAdjustOrderImportEntity;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountParam;
import com.elitesland.fin.application.facade.param.limitadjustorder.LimitAdjustOrderSaveParam;
import com.elitesland.fin.application.service.limitadjustorder.LimitAdjustOrderService;
import com.elitesland.fin.common.*;
import com.elitesland.fin.entity.limitadjustorder.LimitAdjustOrderDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.limitadjustorder.LimitAdjustOrderRepo;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 调整单导入
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class CreditAdjustOrderImportServiceImpl implements DataImport<CreditAdjustOrderImportEntity> {

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final LimitAdjustOrderRepo limitAdjustOrderRepo;

    private final CreditAccountRepoProc creditAccountRepoProc;

    private final UdcProvider udcProvider;

    private final SysNumberGenerator sysNumberGenerator;

    private final LimitAdjustOrderService limitAdjustOrderService;

    private final TransactionTemplate transactionTemplate;
    private final FlexFieldUtilService flexFieldUtilService;


    @Override
    public String getTmplCode() {
        return "credit_adjust_order_import";
    }

    @Override
    public List<String> executeImport(List<CreditAdjustOrderImportEntity> dataList, int rowId) {

        if (CollectionUtil.isEmpty(dataList)) {
            return List.of();
        }

        // 查询信用账户信息
        Set<String> accountCodeList = dataList.stream().map(CreditAdjustOrderImportEntity::getCreditAccountCode).filter(code -> StrUtil.isNotBlank(code)).collect(Collectors.toSet());
        Map<String, CreditAccountDTO> accountMap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(accountCodeList)) {
            CreditAccountParam creditAccountParam = new CreditAccountParam();
            creditAccountParam.setCreditAccountCodes(accountCodeList);
            List<CreditAccountDTO> creditCreditAccountDTOList = creditAccountRepoProc.getAccountByAccountParam(creditAccountParam);
            accountMap = creditCreditAccountDTOList.stream().collect(Collectors.toMap(CreditAccountDTO::getCreditAccountCode, t -> t, (t1, t2) -> t2));
        }

        // 查询UDC
        Map<String, Map<String, String>> udcCodeMap = udcProvider.getValueMapByUdcCode((UdcEnum.CREDIT_TYPE_1.getModel()),
                Set.of(UdcEnum.CREDIT_TYPE_1.getCode(), UdcEnum.ADJUST_TYPE_1.getCode(), UdcEnum.FIN_ADJUST_REASON_1.getCode()));
        Map<String, String> creditTypeMap = udcCodeMap.get(UdcEnum.CREDIT_TYPE_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));
        Map<String, String> adjustTypeMap = udcCodeMap.get(UdcEnum.ADJUST_TYPE_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));
        Map<String, String> adjustReasonMap = udcCodeMap.get(UdcEnum.FIN_ADJUST_REASON_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));


        List<String> errorResult = new ArrayList<>();
        // 储存新导入的数据
        List<LimitAdjustOrderSaveParam> saveList = new ArrayList<>();

        // 保存调出总金额
        Map<String, BigDecimal> amtMap = new HashMap<>();
        BigDecimal totalAmt = BigDecimal.ZERO;
        for (CreditAdjustOrderImportEntity entity : dataList) {
            if (StrUtil.isNotBlank(entity.getCreditAccountCode())
                    && NumberUtil.isNumber(entity.getAdjustLimit())
                    && StrUtil.equals(entity.getAdjustTypeName(), UdcEnum.ADJUST_TYPE_2.getValueCodeName())) {
                totalAmt = NumberUtil.add(totalAmt, new BigDecimal(entity.getAdjustLimit()));
                amtMap.put(entity.getCreditAccountCode(), totalAmt);
            }
        }

        // 储存已经存在的数据
        for (CreditAdjustOrderImportEntity importEntity : dataList) {

            LimitAdjustOrderSaveParam limitAdjustOrderSaveParam = new LimitAdjustOrderSaveParam();
            List<String> errorList = new ArrayList<>();

            // 必填校验
            checkImportMandatoryField(importEntity, errorList, rowId);

            // 填写的类型是否匹配UDC
            checkUdc(rowId, importEntity, limitAdjustOrderSaveParam, errorList, adjustTypeMap, adjustReasonMap, creditTypeMap);

            // 调整金额的账户信息校验
            checkAccount(rowId, importEntity, limitAdjustOrderSaveParam, errorList, accountMap, amtMap);

            // 设置信息
            limitAdjustOrderSaveParam.setRemark(importEntity.getRemark());
            limitAdjustOrderSaveParam.setDocState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());

            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            } else {
                String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());
                limitAdjustOrderSaveParam.setDocNo(docNo);
                // 不存在错误信息
                errorResult.add(null);
                limitAdjustOrderSaveParam.setExtensionInfo(importEntity.getExtensionInfo());
                saveList.add(limitAdjustOrderSaveParam);
            }
            // 行数自增
            rowId++;
        }

        if (CollectionUtils.isNotEmpty(saveList)) {
            saveData(saveList);
            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }
    }

    private void checkUdc(int rowId,
                          CreditAdjustOrderImportEntity importEntity,
                          LimitAdjustOrderSaveParam limitAdjustOrderSaveParam,
                          List<String> errorList,
                          Map<String, String> adjustTypeMap,
                          Map<String, String> adjustReasonMap,
                          Map<String, String> creditTypeMap) {

        if (adjustReasonMap.containsKey(importEntity.getAdjustReasonName())) {
            limitAdjustOrderSaveParam.setAdjustReason(adjustReasonMap.get(importEntity.getAdjustReasonName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "填写不正确"));
        }

        if (adjustTypeMap.containsKey(importEntity.getAdjustTypeName())) {
            limitAdjustOrderSaveParam.setAdjustType(adjustTypeMap.get(importEntity.getAdjustTypeName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "填写不正确"));
        }

        if (creditTypeMap.containsKey(importEntity.getCreditTypeName())) {
            limitAdjustOrderSaveParam.setCreditType(creditTypeMap.get(importEntity.getCreditTypeName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "授信类型", "填写不正确"));
        }

    }


    private void checkAccount(int rowId, CreditAdjustOrderImportEntity importEntity,
                              LimitAdjustOrderSaveParam limitAdjustOrderSaveParam, List<String> errorList,
                              Map<String, CreditAccountDTO> accountMap, Map<String, BigDecimal> amtMap) {

        // 校验金额
        BigDecimal amt = BigDecimal.ZERO;
        if (NumberUtil.isNumber(importEntity.getAdjustLimit())) {

            amt = new BigDecimal(importEntity.getAdjustLimit());
            if (NumberUtil.isLess(amt, BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "不能小于0"));
            } else {
                Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);
                if (!pattern.matcher(importEntity.getAdjustLimit()).matches()) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额，最多两位小数"));
                }
            }

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "格式不对"));
        }

        // 检验账户
        if (accountMap.containsKey(importEntity.getCreditAccountCode())) {
            CreditAccountDTO accountDTO = accountMap.get(importEntity.getCreditAccountCode());

            if (!StrUtil.equals(accountDTO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户未激活"));
            }

            // 调出金额小于账户可用金额
            if (StrUtil.equals(limitAdjustOrderSaveParam.getAdjustType(), UdcEnum.ADJUST_TYPE_2.getValueCode())
                    && amtMap.containsKey(importEntity.getCreditAccountCode())
                    && NumberUtil.isGreater(amtMap.get(importEntity.getCreditAccountCode()), accountDTO.getCreditAccountLimit())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调出金额大于信用账户额度"));
            }
            limitAdjustOrderSaveParam.setCreditAccountCode(accountDTO.getCreditAccountCode());
            limitAdjustOrderSaveParam.setCreditAccountName(accountDTO.getCreditAccountName());
            limitAdjustOrderSaveParam.setAdjustLimit(amt);
            limitAdjustOrderSaveParam.setOuName(accountDTO.getOuName());
            limitAdjustOrderSaveParam.setOuCode(accountDTO.getOuCode());
            limitAdjustOrderSaveParam.setObjectType(accountDTO.getObjectType());
            limitAdjustOrderSaveParam.setObjectName(accountDTO.getObjectName());
            limitAdjustOrderSaveParam.setProductLine(accountDTO.getProductLineCode());
            limitAdjustOrderSaveParam.setSaleUser(accountDTO.getSalesmanName());
            limitAdjustOrderSaveParam.setBuName(accountDTO.getDeptName());
            limitAdjustOrderSaveParam.setBuCode(accountDTO.getDeptCode());

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户不存在"));
        }

        try {
            if (StrUtil.isNotBlank(importEntity.getEffectiveTime())) {
                LocalDateTime effectiveTime = LocalDateTimeUtil.parse(importEntity.getEffectiveTime(), DatetimeUtil.FORMATTER_DATE);
                limitAdjustOrderSaveParam.setEffectiveTime(effectiveTime);
            }
        } catch (Exception e) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "效期起", "效期起格式不正确，需为yyyy-MM-dd"));
        }

        try {
            if (StrUtil.isNotBlank(importEntity.getExpireTime())) {
                LocalDateTime expireTime = LocalDateTimeUtil.parse(importEntity.getExpireTime(), DatetimeUtil.FORMATTER_DATE);
                LocalDateTime localDateTime = expireTime.withHour(23).withMinute(59).withSecond(59);
                limitAdjustOrderSaveParam.setExpireTime(localDateTime);
            }
        } catch (Exception e) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "效期至", "效期至格式不正确，需为yyyy-MM-dd"));
        }

        if (ObjUtil.isNotNull(limitAdjustOrderSaveParam.getEffectiveTime())
                && ObjUtil.isNotNull(limitAdjustOrderSaveParam.getExpireTime())
                && limitAdjustOrderSaveParam.getEffectiveTime().isAfter(limitAdjustOrderSaveParam.getExpireTime())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "有效期", "效期至不能早于效期起"));
        }

    }


    private void checkImportMandatoryField(CreditAdjustOrderImportEntity importEntity, List<String> errorList, int rowId) {


        if (StringUtils.isBlank(importEntity.getCreditTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "授信类型", "授信类型必填"));
        }

        if (StringUtils.isBlank(importEntity.getAdjustTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "调整类型必填"));
        }

        if (StringUtils.isBlank(importEntity.getEffectiveTime())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "效期起", "效期起必填"));
        }

        if (StringUtils.isBlank(importEntity.getExpireTime())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "效期至", "效期至必填"));
        }

        if (StringUtils.isBlank(importEntity.getCreditAccountCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码必填"));
        }

        if (StringUtils.isBlank(importEntity.getAdjustLimit())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整额度", "调整额度必填"));
        }

        if (StringUtils.isBlank(importEntity.getAdjustReasonName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "调整原因必填"));
        }


    }

    public void saveData(List<LimitAdjustOrderSaveParam> saveList) {
        List<LimitAdjustOrderDO> doList = LimitAdjustOrderConvert.INSTANCE.limitAdjustOrderParams2LimitAdjustOrderDOs(saveList);
        flexFieldUtilService.handFlexFieldValueFeferenceList(FinFlexFieldCodeConstant.LIMIT_ADJUST_ORDER, doList);
        List<LimitAdjustOrderDO> adjustOrderDOS = limitAdjustOrderRepo.saveAll(doList);
        submit(adjustOrderDOS);
    }

    public void submit(List<LimitAdjustOrderDO> doList) {
        doList.stream().forEach(adjustOrderDO -> {
            limitAdjustOrderService.submit(adjustOrderDO.getId());
        });

    }

}
