package com.elitesland.fin.rpc.sale;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.json.JSONUtil;
import com.cloudt.apm.common.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.sale.api.vo.param.crm.AllowShipCustGroupDParam;
import com.elitesland.sale.api.vo.param.crm.CustCode2BaseParam;
import com.elitesland.sale.api.vo.resp.crm.AllowShipCustGroupDVO;
import com.elitesland.sale.api.vo.resp.crm.CustBaseDTO;
import com.elitesland.sale.api.vo.resp.crm.CustBelongOuDTO;
import com.elitesland.sale.api.vo.resp.crm.LmSaveCustRespVO;
import com.elitesland.sale.api.vo.resp.sal.RmiOrgBankAccRpcVO;
import com.elitesland.sale.constant.AllowShipCustGroupEnum;
import com.elitesland.sale.constant.AllowShipStatus;
import com.elitesland.sale.dto.CrmCustBaseDTO;
import com.elitesland.sale.dto.CustBaseInfoDTO;
import com.elitesland.sale.dto.param.CustBaseRpcParam;
import com.elitesland.sale.service.AllowShipRpcService;
import com.elitesland.sale.service.CrmCustRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class RmiSaleRpcServiceImpl implements RmiSaleRpcService {

    private final CrmCustRpcService crmCustRpcService;

    private final AllowShipRpcService allowShipRpcService;

    @Override
    public List<LmSaveCustRespVO> findInvCust(List<String> custCodeList) {
        ApiResult<List<LmSaveCustRespVO>> apiResult = crmCustRpcService.findInvCust(custCodeList);
        Assert.notNull(apiResult, "调用2B销售域查询主客户信息为空,客户编码{}", custCodeList);
        Assert.isTrue(apiResult.isSuccess(), "调用2B销售域查询主客户信息失败，客户编码{}", custCodeList);
        return apiResult.getData();
    }

    @Override
    public List<AllowShipCustGroupDVO> getCustGroupInfo(AllowShipCustGroupDParam param) {
        List<AllowShipCustGroupDVO> custGroupInfo;
        try {
            custGroupInfo = allowShipRpcService.getCustGroupInfo(param);
        } catch (Exception e) {
            log.error("调用销售域接口查询客户组信息失败, {}", e.getMessage());
            throw new BusinessException("调用销售域接口查询客户组信息失败");
        }
        return custGroupInfo;
    }

    @Override
    public String getCustGroupCodeByCustCode(String custCode) {
        AllowShipCustGroupDParam allowShipCustGroupDParam = new AllowShipCustGroupDParam();
        List<String> custCodes = new ArrayList<>();
        custCodes.add(custCode);
        allowShipCustGroupDParam.setCustCodes(custCodes);
        allowShipCustGroupDParam.setStatus(AllowShipStatus.ACTIVE.getType());
        allowShipCustGroupDParam.setType(AllowShipCustGroupEnum.CREIDT_ACCOUNT_GROUP.getType());

        List<AllowShipCustGroupDVO> allowShipCustGroupDVOList = allowShipRpcService.getCustGroupInfo(allowShipCustGroupDParam);

        if (CollectionUtils.isNotEmpty(allowShipCustGroupDVOList)) {
            String custGroupCode = allowShipCustGroupDVOList.get(0).getCode();
            Assert.notEmpty(custGroupCode, "查询客户组信息失败");
            return custGroupCode;
        }
        return custCode;
    }

    @Override
    public List<RmiOrgBankAccRpcVO> findBankAccByCustCode(String custCode) {
        List<RmiOrgBankAccRpcVO> bankAccRpcDtoByCustCode = null;
        try {
            bankAccRpcDtoByCustCode = crmCustRpcService.findBankAccRpcDtoByCustCode(custCode);
        } catch (Exception e) {
            log.error("调用销售域查询客户银行信息失败, {}", e.getMessage());
            throw new BusinessException("调用销售域查询客户银行信息失败");
        }

        return bankAccRpcDtoByCustCode;
    }

    public List<CrmCustBaseDTO> getCustBaseInfo(List<String> custCodes) {
        log.info("调用2B销售域-获取客户基本信息,参数:{}", custCodes);
        List<CrmCustBaseDTO> crmCustBaseDTOS = crmCustRpcService.getBaseCustByParam(custCodes).computeData();
        log.info("调用2B销售域-获取客户基本信息,结果:{}", crmCustBaseDTOS);
        return crmCustBaseDTOS;
    }



    @Override
    public List<CustBaseDTO> findBaseCustByCodes(List<String> custCodes) {
        //log.info("调用销售域(findBaseCustByCodes)查询客户基础信息接口，时间：{}，入参：{}", LocalDateTime.now(), JSON.toJSONString(custCodes));
        if (CollectionUtil.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        try {
            CustBaseRpcParam custBaseRpcParam = new CustBaseRpcParam();
            custBaseRpcParam.setCustCodeList(custCodes);
            ApiResult<List<CustBaseDTO>> apiResult = crmCustRpcService.findBaseByParam(custBaseRpcParam);
            //log.info("调用销售域-findBaseCustByCodes，时间：{}，出参：{}", LocalDateTime.now(), JSON.toJSONString(apiResult));
            if (apiResult.isSuccess()) {
                List<CustBaseDTO> rpcDTOList = apiResult.getData();
                if (CollectionUtil.isNotEmpty(rpcDTOList)) {
                    return rpcDTOList;
                }
            } else {
                throw new com.elitescloud.boot.exception.BusinessException(ApiCode.FAIL, "调用销售域(findBaseCustByCodes)查询客户失败：" + apiResult.getErrorMsg());
            }
        } catch (Exception e) {
            log.error("findBaseCustByCodes error:", e);
            throw new com.elitescloud.boot.exception.BusinessException(ApiCode.FAIL, "调用销售域中心dubbo服务异常：" + e);
        }

        return Collections.EMPTY_LIST;
    }

    @Override
    public List<CustBaseDTO> findBaseCustByParam(CustBaseRpcParam custBaseRpcParam) {
        log.info("客户基本信息查询，参数:{}", JSONUtil.toJsonStr(custBaseRpcParam));
        List<CustBaseDTO> data = new ArrayList<>();
        try {
            data = crmCustRpcService.findBaseByParam(custBaseRpcParam).computeData();
        } catch (Exception e) {
            throw new com.elitescloud.boot.exception.BusinessException("客户基本信息查询异常", e);
        }
        log.info("客户基本信息查询,结果:{}", JSONUtil.toJsonStr(data));
        return data;
    }

    @Override
    public Map<String, CustBaseDTO> findBaseCustMapByCodes(List<String> custCodes) {

        List<CustBaseDTO> rpcDTOList = this.findBaseCustByCodes(custCodes);
        if (CollectionUtil.isEmpty(rpcDTOList)) {
            return new HashMap<>();
        }
        Map<String, CustBaseDTO> custMap = rpcDTOList.stream().collect(Collectors.toMap(CustBaseDTO::getCustCode, t -> t, (t1, t2) -> t1));

        return custMap;

    }



    @Override
    public List<CustBaseInfoDTO> getCustBaseInfoByCode(List<String> custCodes) {
        if (CollectionUtil.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        try {
            CustCode2BaseParam custBaseRpcParam = new CustCode2BaseParam();
            custBaseRpcParam.setCustCode(custCodes);
            ApiResult<List<CustBaseInfoDTO>> apiResult = crmCustRpcService.getCustBaseInfoByCode(custBaseRpcParam);
            if (apiResult.isSuccess()) {
                List<CustBaseInfoDTO> rpcDTOList = apiResult.getData();
                if (CollectionUtil.isNotEmpty(rpcDTOList)) {
                    return rpcDTOList;
                }
            } else {
                throw new com.elitescloud.boot.exception.BusinessException(ApiCode.FAIL, "调用销售域(getCustBaseInfoByCode)查询客户失败：" + apiResult.getErrorMsg());
            }
        } catch (Exception e) {
            log.error("getCustBaseInfoByCode error:", e);
            throw new com.elitescloud.boot.exception.BusinessException(ApiCode.FAIL, "调用销售域中心dubbo服务异常：" + e);
        }

        return Collections.EMPTY_LIST;
    }

    @Override
    public List<CustBelongOuDTO> findCustBelongOuByCustCode(String custCode) {
        List<CustBelongOuDTO> custBelongOuByCustCode = new ArrayList<>();
        try {
            custBelongOuByCustCode = crmCustRpcService.findCustBelongOuByCustCode(custCode);
        } catch (Exception e) {
            throw new com.elitescloud.boot.exception.BusinessException("客户归属公司信息查询异常", e);
        }
        log.info("客户归属公司信息:{}", JSONUtil.toJsonStr(custBelongOuByCustCode));
        return custBelongOuByCustCode;
    }

}
