package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.fin.application.convert.adjustorder.AdjustOrderConvert;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.excel.adjust.AdjustOrderImportEntity;
import com.elitesland.fin.application.facade.param.account.AccountParam;
import com.elitesland.fin.application.facade.param.adjustorder.AdjustOrderSaveParam;
import com.elitesland.fin.application.service.adjustorder.AdjustOrderServiceImpl;
import com.elitesland.fin.common.*;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.support.provider.org.dto.OrgOuBankRpcDTO;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 调整单导入
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class AdjustOrderImportServiceImpl implements DataImport<AdjustOrderImportEntity> {

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final AccountRepoProc accountRepoProc;

    private final AdjustOrderRepo adjustOrderRepo;

    private final UdcProvider udcProvider;

    private final SysNumberGenerator sysNumberGenerator;

    private final AdjustOrderServiceImpl adjustOrderService;
    private final FlexFieldUtilService flexFieldUtilService;
    private final OrgStoreRpcService orgStoreRpcService;
    private final OrgOuRpcService orgOuRpcService;
    private final AdjustOrderSubmitJob adjustOrderSubmitJob;

    @Override
    public String getTmplCode() {
        return "yst_fin_adjust_order_import";
    }

    @Override
    public Integer stepSize() {
        return 50000;
    }

    @Override
    public List<String> executeImport(List<AdjustOrderImportEntity> dataList, int rowId) {
        if (CollectionUtil.isEmpty(dataList)) {
            return List.of();
        }

        // 查询账户信息
        Set<String> accountCodeList = dataList.stream().map(AdjustOrderImportEntity::getAccCode).filter(StrUtil::isNotBlank).collect(Collectors.toSet());
        Map<String, AccountDTO> accountMap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(accountCodeList)) {
            AccountParam accountParam = new AccountParam();
            accountParam.setAccountCodes(accountCodeList);
            accountParam.setState("ACTIVE");
            List<AccountDTO> accountDTOList = accountRepoProc.getAccountByAccountParam(accountParam);
            accountMap = accountDTOList.stream().collect(Collectors.toMap(AccountDTO::getAccountCode, t -> t, (t1, t2) -> t1));
        }

        // 查询UDC
        Map<String, Map<String, String>> udcCodeMap = udcProvider.getValueMapByUdcCode((UdcEnum.ADJUST_TYPE_1.getModel()),
                Set.of(UdcEnum.ADJUST_TYPE_1.getCode(), UdcEnum.FIN_ADJUST_REASON_1.getCode()));
        Map<String, String> adjustTypeMap = udcCodeMap.get(UdcEnum.ADJUST_TYPE_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));
        Map<String, String> adjustReasonMap = udcCodeMap.get(UdcEnum.FIN_ADJUST_REASON_1.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));

        List<String> errorResult = new ArrayList<>();
        // 储存新导入的数据
        List<AdjustOrderSaveParam> saveList = new ArrayList<>();

        // 保存调出总金额
        Map<String, BigDecimal> amtMap = new HashMap<>();
        for (AdjustOrderImportEntity entity : dataList) {
            if (StrUtil.isNotBlank(entity.getAccCode())
                    && NumberUtil.isNumber(entity.getTotalAmt())
                    && StrUtil.equals(entity.getAdjTypeName(), UdcEnum.ADJUST_TYPE_2.getValueCodeName())) {
                // 按账号，累加调出总金额
                BigDecimal totalAmt = amtMap.getOrDefault(entity.getAccCode(), BigDecimal.ZERO);
                totalAmt = NumberUtil.add(totalAmt, new BigDecimal(entity.getTotalAmt()));
                amtMap.put(entity.getAccCode(), totalAmt);
            }
        }

        Map<String, OrgStoreBaseRpcDTO> storeMap = new HashMap<>();
        Map<String, String> storeBankAccMap = new HashMap<>();
        if (MapUtils.isNotEmpty(accountMap)) {
            OrgStoreBaseRpcParam storeBaseRpcParam = new OrgStoreBaseRpcParam();
            storeBaseRpcParam.setStoreCodeList(accountMap.values().stream().map(AccountDTO::getAccountHolderCode).filter(Objects::nonNull).distinct().toList());
            ApiResult<List<OrgStoreBaseRpcDTO>> storeResult = orgStoreRpcService.findSimpleStoreByParam(storeBaseRpcParam);
            if (storeResult.isFailed() || CollectionUtil.isEmpty(storeResult.getData())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账号所属门店信息查询失败");
            }
            storeMap = storeResult.getData().stream().collect(Collectors.toMap(OrgStoreBaseRpcDTO::getStoreCode, c -> c));
            Set<String> storedOuCodes = storeResult.getData().stream().map(OrgStoreBaseRpcDTO::getStoredOuCode).filter(StrUtil::isNotBlank).collect(Collectors.toSet());
            log.info("储值公司编码: {}", JSONUtil.toJsonString(storedOuCodes));
            List<OrgOuRpcSimpleDTO> storedOuList = storedOuCodes.isEmpty() ? Collections.emptyList() : orgOuRpcService.findSimpleByOuCodes(new ArrayList<>(storedOuCodes));
            Set<Long> storedOuIds = CollUtil.isEmpty(storedOuList) ? Collections.emptySet() : storedOuList.stream().map(OrgOuRpcSimpleDTO::getId).filter(Objects::nonNull).collect(Collectors.toSet());
            log.info("储值公司ID: {}", JSONUtil.toJsonString(storedOuIds));
            List<OrgOuBankRpcDTO> ouBankList = storedOuIds.isEmpty() ? Collections.emptyList() : orgOuRpcService.findOuBankDetail(UdcEnum.COM_OU_TYPE_10.getValueCode(), new ArrayList<>(storedOuIds));
            log.info("储值公司银行信息: {}", JSONUtil.toJsonString(ouBankList));
            for (OrgOuBankRpcDTO orgOuBankRpcDTO : ouBankList) {
                if ("IN".equals(orgOuBankRpcDTO.getAccType())) {
                    storeBankAccMap.put(orgOuBankRpcDTO.getOuCode(), orgOuBankRpcDTO.getBankAcc());
                }
            }
        }

        // 储存已经存在的数据
        for (AdjustOrderImportEntity importEntity : dataList) {
            AdjustOrderSaveParam adjustOrderSaveParam = new AdjustOrderSaveParam();
            List<String> errorList = new ArrayList<>();

            String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());

            // 必填校验
            checkImportMandatoryField(importEntity, errorList, rowId);

            // 填写的类型是否匹配UDC
            checkUdc(rowId, importEntity, adjustOrderSaveParam, errorList, adjustTypeMap, adjustReasonMap);

            // 调整金额的账户信息校验
            checkAccount(rowId, importEntity, adjustOrderSaveParam, errorList, accountMap, amtMap, storeMap, storeBankAccMap);

            // 设置信息
            adjustOrderSaveParam.setRemark(importEntity.getRemark());
            adjustOrderSaveParam.setType(UdcEnum.ADJUST_ORDER_TYPE_YETZ.getValueCode());
            adjustOrderSaveParam.setDocNo(docNo);
            adjustOrderSaveParam.setState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
            adjustOrderSaveParam.setExtensionInfo(importEntity.getExtensionInfo());

            // 数据来源是导入
            adjustOrderSaveParam.setDataSource("IMPORT");
            // 自动处理状态为，待处理
            adjustOrderSaveParam.setProcessStatus("TODO");

            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            } else {
                // 不存在错误信息
                errorResult.add(null);
                saveList.add(adjustOrderSaveParam);
            }
            // 行数自增
            rowId++;
        }

        long count = errorResult.stream().filter(Objects::nonNull).count();
        if (count > 0) {
            List<String> newErrorResult = new ArrayList<>();
            for (String s : errorResult) {
                if (s == null) {
                    s = "解析成功";
                }
                newErrorResult.add(s);
            }
            log.info("存调整单导入数据错误数据: {}", JSONUtil.toJsonString(newErrorResult));
            return newErrorResult;
        } else {
            if (CollectionUtils.isNotEmpty(saveList)) {
                log.info("保存调整单导入数据: {}", JSONUtil.toJsonString(saveList));
                saveData(saveList);
            }
        }
        return null;
    }

    @Override
    public void afterImport(boolean sync) {
        // 触发定时任务,处理后续逻辑
        log.info("触发定时任务,处理后续逻辑");
        adjustOrderSubmitJob.submitDraftAdjustOrders(null);
    }

    private void checkUdc(int rowId, AdjustOrderImportEntity importEntity,
                                 AdjustOrderSaveParam adjustOrderSaveParam, List<String> errorList,
                                 Map<String, String> adjustTypeMap,
                                 Map<String, String> adjustReasonMap) {

        if (adjustReasonMap.containsKey(importEntity.getAdjReasonName())) {
            adjustOrderSaveParam.setAdjReason(adjustReasonMap.get(importEntity.getAdjReasonName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "填写不正确"));
        }

        if (adjustTypeMap.containsKey(importEntity.getAdjTypeName())) {
            adjustOrderSaveParam.setAdjType(adjustTypeMap.get(importEntity.getAdjTypeName()));
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "填写不正确"));
        }
    }


    private void checkAccount(int rowId, AdjustOrderImportEntity importEntity,
                                     AdjustOrderSaveParam adjustOrderSaveParam, List<String> errorList,
                                     Map<String, AccountDTO> accountMap, Map<String, BigDecimal> amtMap, Map<String, OrgStoreBaseRpcDTO> storeMap,
                              Map<String, String> storeBankAccMap) {

        // 校验金额
        BigDecimal amt = BigDecimal.ZERO;
        if (NumberUtil.isNumber(importEntity.getTotalAmt())) {

            Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);
            if (!pattern.matcher(importEntity.getTotalAmt()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额，最多两位小数"));
            }

            amt = new BigDecimal(importEntity.getTotalAmt());
            if (NumberUtil.isLess(amt, BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额，不能小于0"));
            }

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额格式不对"));
        }

        // 检验账户
        if (accountMap.containsKey(importEntity.getAccCode())) {
            AccountDTO accountDTO = accountMap.get(importEntity.getAccCode());

            if (!StrUtil.equals(accountDTO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户未激活"));
            }

            // 调出金额小于账户可用金额
            if (StrUtil.equals(adjustOrderSaveParam.getAdjType(), UdcEnum.ADJUST_TYPE_2.getValueCode())
                    && amtMap.containsKey(importEntity.getAccCode())
                    && NumberUtil.isGreater(amtMap.get(importEntity.getAccCode()), accountDTO.getAccountAvailableAmount())) {
                log.warn("{}, 调出金额({})大于账户可用金额({})", importEntity.getAccCode(), amtMap.get(importEntity.getAccCode()), accountDTO.getAccountAvailableAmount());
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调出金额大于账户可用金额"));
            }
            adjustOrderSaveParam.setAccCode(importEntity.getAccCode());
            adjustOrderSaveParam.setAccName(accountDTO.getAccountName());
            adjustOrderSaveParam.setTotalAmt(amt);
            adjustOrderSaveParam.setAccType(accountDTO.getAccountType());
            adjustOrderSaveParam.setAccountHolderName(accountDTO.getAccountHolderName());
            adjustOrderSaveParam.setAccountHolderCode(accountDTO.getAccountHolderCode());

            if (storeMap.containsKey(accountDTO.getAccountHolderCode())) {
                OrgStoreBaseRpcDTO orgStore = storeMap.get(accountDTO.getAccountHolderCode());
                if ("CLOSED".equals(orgStore.getStoreStatus())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "该账户的门店已闭店,无法调整"));
                }
                adjustOrderSaveParam.setStoredBankAcc(storeBankAccMap.get(orgStore.getStoredOuCode()));
            }

        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户不存在"));
        }
    }


    private void checkImportMandatoryField(AdjustOrderImportEntity importEntity, List<String> errorList, int rowId) {

        if (StringUtils.isBlank(importEntity.getAccCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码必填"));
        }
        if (StringUtils.isBlank(importEntity.getAdjTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整类型", "调整类型必填"));
        }
        if (StringUtils.isBlank(importEntity.getTotalAmt())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整金额", "调整金额必填"));
        }
        if (StringUtils.isBlank(importEntity.getAdjReasonName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "调整原因", "调整原因必填"));
        }
    }

    public void saveData(List<AdjustOrderSaveParam> saveList) {
        List<AdjustOrderDO> doList = AdjustOrderConvert.INSTANCE.paramsToDOs(saveList);
        flexFieldUtilService.handFlexFieldValueFeferenceList(FinFlexFieldCodeConstant.ADJUST_ORDER,doList);
        adjustOrderRepo.saveAll(doList);
    }

    public void submit(List<AdjustOrderDO> doList) {
        doList.forEach(adjustOrderService::workflowAndGenerateAccountFlow);
    }

}
