package com.elitesland.fin.application.service.excel.imp;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitesland.fin.application.facade.dto.mq.AdjustOrderSubmitProcessMqMessageDTO;
import com.elitesland.fin.application.service.adjustorder.AdjustOrderService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.redisson.RedissonRedLock;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

@Component
@Slf4j
@RequiredArgsConstructor
public class AdjustOrderSubmitProcessListener implements MessageQueueListener<AdjustOrderSubmitProcessMqMessageDTO> {

    private final AdjustOrderService adjustOrderService;
    private final RedLockUtils redLockUtils;
    private final AdjustOrderRepo adjustOrderRepo;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{AdjustOrderSubmitProcessMqMessageDTO.SUBMIT_CHANNEL};
    }

    @Override
    public void onConsume(@NotBlank String s, @NotNull AdjustOrderSubmitProcessMqMessageDTO messageDTO) {
        log.info("调整单确认处理提交：{}", JSON.toJSONString(messageDTO));
        RedissonRedLock redLock = null;
        AdjustOrderDO draftOrder = null;
        try {
            // 分布式锁
            redLock = redLockUtils.getRedLock("adjust-order-submit-process" + messageDTO.getBusinessKey());
            boolean lockFlag = redLock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("调整单已在处理中,获取锁失败");
                return;
            }
            // 幂等，判断单据状态
            Optional<AdjustOrderDO> byId = adjustOrderRepo.findById(messageDTO.getId());
            if (byId.isEmpty()) {
                log.error("未找到调整单");
                return;
            }
            draftOrder = byId.get();
            if (!draftOrder.getState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                log.error("调整单不处于草稿状态，跳过处理");
                return;
            }
            if (!draftOrder.getProcessStatus().equals("TODO")) {
                log.error("调整单不处于待处理状态，跳过处理");
                return;
            }
            adjustOrderService.workflowAndGenerateAccountFlow(draftOrder);
        } catch (Exception e) {
            log.error("提交调整单时发生异常,调整单号:{}", messageDTO.getBusinessKey(), e);
            if (draftOrder != null) {
                draftOrder.setProcessResult(e.getMessage());
                draftOrder.setState("REJECTED");
                draftOrder.setProcessStatus("FAIL");
            }
        } finally {
            if (redLock != null) {
                redLock.unlock();
                log.info("调整单确认完成，释放锁成功");
            }
        }
        if (draftOrder != null && Strings.isNotBlank(draftOrder.getProcessResult())) {
            adjustOrderRepo.save(draftOrder);
        }
    }
}
