package com.elitesland.fin.application.service.flow;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.account.AccountConvert;
import com.elitesland.fin.application.convert.flow.AccountFlowConvert;
import com.elitesland.fin.application.facade.dto.account.AccountIocDTO;
import com.elitesland.fin.application.facade.param.account.AccountIocParam;
import com.elitesland.fin.application.facade.param.account.AccountRuleConfigQueryParam;
import com.elitesland.fin.application.facade.param.flow.*;
import com.elitesland.fin.application.facade.vo.account.AccountRuleConfigPageVO;
import com.elitesland.fin.application.facade.vo.account.AccountVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowDealerVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.account.AccountIocService;
import com.elitesland.fin.application.service.account.AccountRuleConfigService;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.redis.MyRedisService;
import com.elitesland.fin.application.service.saobei.SaobeiAccountService;
import com.elitesland.fin.application.service.unionpay.entity.enums.StoredOrderTypeEnum;
import com.elitesland.fin.common.*;
import com.elitesland.fin.domain.service.AccountFlow.AccountFlowDomainService;
import com.elitesland.fin.domain.service.account.AccountDomainService;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.account.AccountRuleConfigDO;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.entity.flow.AccountFlowDO;
import com.elitesland.fin.provider.bip.BipFinService;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.account.AccountRuleConfigRepo;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepo;
import com.elitesland.fin.repo.flow.AccountFlowRepoProc;
import com.elitesland.fin.repo.push.FinPushResultRepoProc;
import com.elitesland.fin.rocketmq.channel.FinSalMqProcessor;
import com.elitesland.fin.rpc.order.RmiOrderRpcService;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.fin.utils.BeanUtils;
import com.elitesland.fin.utils.SysUtils;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.support.provider.org.dto.OrgOuBankAccRpcDTO;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgBankAccQueryRpcParam;
import com.elitesland.support.provider.org.service.OrgAddrRpcService;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import com.google.common.collect.Lists;
import com.saobei.open.sdk.model.requst.allocate.SaobeiDoTransRequest;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.redisson.RedissonRedLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;


/**
 * <p>
 * 功能说明:账户流水
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/25
 * @Version 1.0
 * @Content:
 */
@Service
@Slf4j
public class AccountFlowServiceImpl implements AccountFlowService {
    @Autowired
    private AccountFlowRepo accountFlowRepo;
    @Autowired
    private AccountFlowRepoProc accountFlowRepoProc;

    @Autowired
    private SysNumberGenerator sysNumberGenerator;

    @Autowired
    private AccountFlowZcService accountFlowZcService;
    @Autowired
    private AccountFlowPjService accountFlowPjService;
    @Autowired
    private AccountFlowFjbService accountFlowFjbService;
    @Autowired
    private AccountFlowTcService accountFlowTcService;
    @Autowired
    private AccountFlowCommonService accountFlowCommonService;

    @Autowired
    private AccountRepoProc accountRepoProc;

    @Autowired
    private AccountRepo accountRepo;

    @Autowired
    private RedissonClient redissonClient;


    @Autowired
    private RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;

    @Autowired
    private AccountRuleConfigRepo accountRuleConfigRepo;

    @Autowired
    private RmiOrderRpcService rmiOrderRpcService;

    @Autowired
    private AccountFlowService accountFlowService;

    @Autowired
    private AccountIocService accountIocService;

    @Autowired
    private SeqNumProvider seqNumProvider;

    @Autowired
    private AccountRuleConfigService accountRuleConfigService;

    @Autowired
    private AccountService accountService;

    @Autowired
    private FinSalMqProcessor finSalMqProcessor;
    @Autowired
    private RedLockUtils redLockUtils;
    @Autowired
    private FlexFieldUtilService flexFieldUtilService;
    @Autowired
    private MyRedisService myRedisService;
    @Autowired
    private AccountDomainService accountDomainService;
    @Autowired
    private AccountFlowDomainService accountFlowDomainService;
    @Autowired
    private SaobeiAccountService saobeiAccountService;
    @Autowired
    private OrgAddrRpcService orgAddrRpcService;
    @Autowired
    private OrgStoreRpcService orgStoreRpcService;
    @Autowired
    private FinPushResultRepoProc finPushResultRepoProc;

    private static final BigDecimal ONE_HUNDRED_BIGDECIMAL = new BigDecimal("100");
    @Autowired
    private AdjustOrderRepo adjustOrderRepo;


    /**
     * 账户流水分页查询：中台端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> page(AccountFlowPageParam pageParam) {

        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        //根据公司编码、加盟商编码、规则编码获取名称
        List<AccountFlowVO> accountFlowVOList = pagingVO.getRecords();

        List<String> companyCodes = accountFlowVOList.stream().map(AccountFlowVO::getFromOuCode).filter(Objects::nonNull).collect(Collectors.toList());
        companyCodes.addAll(accountFlowVOList.stream().map(AccountFlowVO::getToOuCode).filter(Objects::nonNull).collect(Collectors.toList()));
        companyCodes.addAll(accountFlowVOList.stream().map(AccountFlowVO::getCompanyCode).filter(Objects::nonNull).collect(Collectors.toList()));
        Map<String, String> ouMap = rmiOrgOuRpcServiceService.findBaseOuByCodes(companyCodes).stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getOuCode, OrgOuRpcSimpleDTO::getOuName));

        List<AccountRuleConfigDO> accountRuleConfigDOList = accountRuleConfigRepo.findAll();

        Set<Long> ids = accountFlowVOList.stream().map(AccountFlowVO::getId).collect(Collectors.toSet());
        var bipPushResultMap = finPushResultRepoProc.queryByDocId(ids, FinPushInter.BIP_FIN_PAY_ACCOUNT_FLOW_STORAGE_SAVE);

        accountFlowVOList.stream().forEach(item -> {

            if (StringUtils.isNotBlank(item.getCompanyCode())) {
                item.setFromOuName(ouMap.get(item.getFromOuCode()));
            }
            if (StringUtils.isNotBlank(item.getTouCode())) {
                item.setToOuName(ouMap.get(item.getToOuCode()));
            }
            if (StringUtils.isNotBlank(item.getCompanyCode())) {
                item.setCompanyCodeName(ouMap.get(item.getCompanyCode()));
            }

            AccountRuleConfigDO accountRuleConfigDOResult = accountRuleConfigDOList.stream().filter(accountRuleConfigDO -> StringUtils.equals(item.getRuleCode(), accountRuleConfigDO.getRuleCode()))
                    .findFirst()
                    .orElse(null);
            if (accountRuleConfigDOResult != null) {
                item.setRuleName(accountRuleConfigDOResult.getRuleName());
            }
            item.setBipPushResult(bipPushResultMap.get(item.getId()));

            item.setOptType(SysUtils.getOptType(item.getSourceDocStatus()));
        });
        if (!CollectionUtils.isEmpty(pagingVO.getRecords())) {
            flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.ACCOUNT_FLOW, pagingVO.getRecords());
        }
        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 根据数据来源+来源单号查询流水的信息
     * <p>
     * 使用场景1：校验是否有重复数据已经存在
     *
     * @param dataSource 数据来源
     * @param sourceNo   来源单号
     * @return 账户流水信息
     */
    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectBySourceAndNo(String dataSource, String sourceNo) {
        if (StringUtils.isEmpty(dataSource)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据来源为空!");
        }
        if (StringUtils.isEmpty(sourceNo)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "来源单号为空!");
        }
        AccountFlowPageParam accountFlowPageParam = new AccountFlowPageParam();
        accountFlowPageParam.setDataSource(dataSource);
        accountFlowPageParam.setSourceNoEq(sourceNo);
        List<AccountFlowVO> accountFlowVOList = accountFlowRepoProc.selectListByParam(accountFlowPageParam);
        if (CollectionUtils.isEmpty(accountFlowVOList)) {
            return Collections.emptyList();
        }
        return accountFlowVOList;
    }

    /**
     * 账户流水分页查询：经销商端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> dealerPage(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        if (StringUtils.isNotBlank(pageParam.getYearMonthStr())) {
            //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
            LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
            //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
            accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);
        }
        //PC端只展示实际扣减和增加的流水
        pageParam.setAccIoTypeList(Lists.newArrayList(FinConstant.ACC_IO_TYPE_ADD, FinConstant.ACC_IO_TYPE_SUB));
        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.dealerPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVO.getRecords())
                .build();
    }

    /**
     * 经销商端查询相关统计金额：入账金额、使用金额
     *
     * @param pageParam 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAmt(AccountFlowPageParam pageParam) {
        //经销商下全部整车储值账户/配件储值账户/附件包储值账户/整车返利账户/配件返利账户/统筹账户
        // 当月实际增加金额之和     当月实际减少金额之和
        //经销商账户金额是 +交易金额  的    就是入账金额    经销商账户金额是-交易金额  的    就是使用金额
        //可以这么理解   不过经销商账户金额是 +-交易金额  的  也是根据交易类型来做的
        //本质是根据交易类型来判断的
        //占用和释放的这种  不需要计算
        //入账和使用  是实际的扣减和增加
        //那么标产里就是对应账户金额+-的发生金额，通过交易类型来分区发生金额是扣减还是增加，然后求的发生金额之和

        //校验必填项02:年月日期、账户编码
        checkMandatoryField02(pageParam);
        //把年月格式yyyy-MM字符串转换拼接成年月日格式的日期
        LocalDate transactionDate = accountFlowCommonService.transitionYearMonthStr(pageParam.getYearMonthStr());
        //根据年月日格式的日期拼装入参里的交易起始时间、交易截止时间
        accountFlowCommonService.splicTransactionTimeSection(transactionDate, pageParam);

        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setUseAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setRecordAmt(BigDecimal.ZERO);
        //等产品确定根据新的维度进行判断扣减或增加统计
        //获取发生金额为增加的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.ADD_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_ADD);
        List<AccountFlowVO> addAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(addAmtList)) {
            //入账金额
            BigDecimal recordAmt = addAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setRecordAmt(recordAmt);
        }
        //获取发生金额为扣减的交易类型编码的集合，并查询数据对发生金额进行相加求和
        //pageParam.setTransactionTypeList(FinConstant.SUB_TRANSACTION_TYPE);
        pageParam.setAccIoType(FinConstant.ACC_IO_TYPE_SUB);
        List<AccountFlowVO> subAmtList = accountFlowRepoProc.dealerSelectAmt(pageParam);
        if (!CollectionUtils.isEmpty(subAmtList)) {
            //使用金额
            BigDecimal useAmt = subAmtList.stream().map(AccountFlowVO::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
            accountFlowDealerVO.setUseAmt(useAmt);
        }

        return accountFlowDealerVO;
    }


    /**
     * 账户流水保存
     *
     * @param paramList 入参
     * @return 账户流水返回对象
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public AccountFlowVO save(List<AccountFlowParam> paramList) {
        log.info("获取流水入参{}", JSON.toJSONString(paramList));
        //校验必填项
        this.checkMandatoryFieldList(paramList);
        //以防万一，正常不需要这种校验处理
        this.checkRepeatability(paramList);
        //默认赋值，放到校验逻辑之后,目前下放到每个业务里了
        //this.defaultAssignment(paramList);

        this.distinguishAccType(paramList);

        //TODO 占用前都会有这个   校验的接口逻辑
        // 则朋 就是这块不是先释放  再占用吗     就是释放后   还有一个校验的接口检查完了之后看看是否满足  才能占用
        // 校验逻辑：整车下单、整车订单修改需要进行校验
        // 订单的时候需要返回结果和值给他们
        // 金额扣减为负的情况，在校验接口里处理
        // 流水第二条是占用的情况，需要再次反向调用校验接口，只有外部接口里需要


        return new AccountFlowVO();
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public AccountFlowDO saveAccountFlow(AccountFlowParam accountFlowParam) {

        checkMandatoryField(accountFlowParam);
        AccountFlowDO accountFlowDO = AccountFlowConvert.INSTANCE.paramToDo(accountFlowParam);
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.ACCOUNT_FLOW, accountFlowDO);
        return accountFlowRepo.save(accountFlowDO);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void fixAccountFlow(List<AccountFlowFixParam> paramList) {
        // 获取当前用户信息
        GeneralUserDetails userDetails = com.elitescloud.boot.auth.util.SecurityContextUtil.currentUser();
        if (userDetails == null) {
            throw new com.el.coordinator.core.common.exception.BusinessException("当前登陆人信息获取为空!");
        }
        SysUserDTO user = userDetails.getUser();
        for (AccountFlowFixParam accountFlowFixParam : paramList) {
            // 通过这个时间点，获取上一条最近的流水
            AccountFlowDO accountFlowDO = accountFlowRepoProc.getLatestFlowBeforeTime(accountFlowFixParam.getAccountCode(),
                    accountFlowFixParam.getAccountType(),
                    accountFlowFixParam.getTransactionTime());
            if (accountFlowDO == null) {
                throw new com.cloudt.apm.common.exception.BusinessException("未找到上一条流水" + accountFlowFixParam.getAccountCode() + "," +
                        accountFlowFixParam.getAccountType() + "," +
                        accountFlowFixParam.getTransactionTime());
            }

            List<AccountDO> allByAccountCode = accountRepo.findAllByAccountCode(accountFlowDO.getAccountCode());
            if (CollectionUtils.isEmpty(allByAccountCode)) {
                throw new com.cloudt.apm.common.exception.BusinessException("未找到账户" + accountFlowDO.getAccountCode());
            }
            AccountDO account = allByAccountCode.stream().filter(accountDO -> accountDO.getAccountType().equals(accountFlowDO.getAccountType())).findFirst()
                    .orElseThrow(() -> new com.cloudt.apm.common.exception.BusinessException("未找到账户" + accountFlowDO.getAccountCode() + "," + accountFlowDO.getAccountType()));

            // 生成调整单
            AdjustOrderDO adjustOrderDO = new AdjustOrderDO();
            adjustOrderDO.setRemark(accountFlowFixParam.getRemark());
            String generate = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());
            adjustOrderDO.setDocNo(generate);
            adjustOrderDO.setState("COMPLETE");
            adjustOrderDO.setType("YETZ");
            adjustOrderDO.setRecAccTime(accountFlowFixParam.getTransactionTime());
            adjustOrderDO.setAccName(accountFlowDO.getAccountName());
            adjustOrderDO.setAccCode(accountFlowDO.getAccountCode());
            adjustOrderDO.setAccType(accountFlowFixParam.getAccountType());
            adjustOrderDO.setTotalAmt(accountFlowFixParam.getAmount());
            adjustOrderDO.setAuditUser(user.getLastName());
            adjustOrderDO.setAuditUserId(user.getId());
            adjustOrderDO.setAuditTime(accountFlowFixParam.getTransactionTime());
            adjustOrderDO.setAdjReason("2");
            adjustOrderDO.setRemark(accountFlowFixParam.getRemark());

            if (Objects.equals(accountFlowFixParam.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
                adjustOrderDO.setAdjType("2");
            } else if (Objects.equals(accountFlowFixParam.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
                adjustOrderDO.setAdjType("1");
            }

            adjustOrderDO.setAccountHolderName(account.getAccountHolderName());
            adjustOrderDO.setAccountHolderCode(account.getAccountHolderCode());

            AdjustOrderDO savedAdjustOrder = adjustOrderRepo.save(adjustOrderDO);

            AccountFlowDO newAccountFlowDO = new AccountFlowDO();
            BeanUtils.copyProperties(accountFlowDO, newAccountFlowDO);

            String flowNo = seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList());
            newAccountFlowDO.setId(null);
            newAccountFlowDO.setFlowNo(flowNo);
            newAccountFlowDO.setTransactionTime(accountFlowFixParam.getTransactionTime());
            newAccountFlowDO.setAmount(accountFlowFixParam.getAmount());
            newAccountFlowDO.setTransactionType(accountFlowFixParam.getTransactionType());
            newAccountFlowDO.setRemark(accountFlowFixParam.getRemark());
            newAccountFlowDO.setCompanyCode(null);
            newAccountFlowDO.setSourceDoc("AO");
            newAccountFlowDO.setSourceNo(savedAdjustOrder.getDocNo());
            newAccountFlowDO.setSourceId(savedAdjustOrder.getId());
            newAccountFlowDO.setSourceDocStatus("AO_COMPLETE");
            newAccountFlowDO.setRuleCode("TZ001");
            newAccountFlowDO.setPriorityNo(8);
            newAccountFlowDO.setBusinessType(null);
            newAccountFlowDO.setTOuCode(null);
            newAccountFlowDO.setToOuCode(null);
            newAccountFlowDO.setFromOuCode(null);
            newAccountFlowDO.setRootDoc(null);
            newAccountFlowDO.setRootDocNo(null);

            newAccountFlowDO.setAuditUserId(user.getId());
            newAccountFlowDO.setAuditUserName(user.getLastName());
            newAccountFlowDO.setAuditDate(accountFlowFixParam.getTransactionTime());

            if (Objects.equals(accountFlowFixParam.getTransactionType(), UdcEnum.IO_TYPE_O.getValueCode())) {
                newAccountFlowDO.setAccountAmount(accountFlowDO.getAccountAmount().subtract(accountFlowFixParam.getAmount()));
                newAccountFlowDO.setAccountAvailableAmount(accountFlowDO.getAccountAvailableAmount().subtract(accountFlowFixParam.getAmount()));
            } else if (Objects.equals(accountFlowFixParam.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode())) {
                newAccountFlowDO.setAccountAmount(accountFlowDO.getAccountAmount().add(accountFlowFixParam.getAmount()));
                newAccountFlowDO.setAccountAvailableAmount(accountFlowDO.getAccountAvailableAmount().add(accountFlowFixParam.getAmount()));
            }
            accountFlowRepo.save(newAccountFlowDO);
        }
    }

    @Override
    public AccountFlowDO saveAccountFlowWithOutTx(AccountFlowParam accountFlowParam) {
        checkMandatoryField(accountFlowParam);
        AccountFlowDO accountFlowDO = AccountFlowConvert.INSTANCE.paramToDo(accountFlowParam);
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.ACCOUNT_FLOW, accountFlowDO);
        return accountFlowRepo.save(accountFlowDO);

    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void saveAccountFlowList(List<AccountFlowParam> accountFlowParams) {
        accountFlowParams.stream().forEach(v -> {
            AccountFlowDO accountFlowDO = saveAccountFlowWithOutTx(v);
            v.setId(accountFlowDO.getId());
        });
    }

    /**
     * 按照账户类型区分
     *
     * @param paramList 入参
     */
    private void distinguishAccType(List<AccountFlowParam> paramList) {

        //避免批量穿过来会交叉多种账户类型模块的数据，故每个大模块账户类型都做了过滤区分
        List<AccountFlowParam> accountFlowZcList = paramList.stream().filter(param -> accountFlowCommonService.judgeZcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowZcList)) {
            accountFlowZcService.saveZc(accountFlowZcList);
        }
        List<AccountFlowParam> accountFlowPjList = paramList.stream().filter(param -> accountFlowCommonService.judgePjAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowPjList)) {
            accountFlowPjService.savePj(accountFlowPjList);
        }
        List<AccountFlowParam> accountFlowFjbList = paramList.stream().filter(param -> accountFlowCommonService.judgeFjbAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowFjbList)) {
            accountFlowFjbService.saveFjb(accountFlowFjbList);
        }
        List<AccountFlowParam> accountFlowTcList = paramList.stream().filter(param -> accountFlowCommonService.judgeTcAccType(param.getAccType())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(accountFlowTcList)) {
            accountFlowTcService.saveTc(accountFlowTcList);
        }


        //1.按照账户类型拆分成4个大模块
        //2.在每个模块里根据数据来源区分不同的方法，方便后续变动或扩展
        //3.把设计到账户金额、账户占用金额(子账户金额、子账户占用金额、经销商账户金额、经销商占用金额)
        // 查询上一条记录的能共用的抽离出单独的方法块，即使立马项目里无法引用，但是代码可以复用

    }


    /**
     * 默认赋值
     *
     * @param paramList 入参
     */
    /*private void defaultAssignment(List<AccountFlowParam> paramList) {
        paramList.forEach(accountFlowParam -> {
            if (StringUtils.isEmpty(accountFlowParam.getSourcePlatform())) {
                accountFlowParam.setSourcePlatform(UdcEnum.FIN_SOURCE_PLATFORM_TYPE_OTHER.getValueCode());
            }
            String flowNo = sysNumberGenerator.generate(SysNumEnum.FIN_FLOW_NO.getCode());
            accountFlowParam.setFlowNo(flowNo);
        });
    }*/

    /**
     * 校验必填项
     *
     * @param paramList 入参
     */
    @Override
    public void checkMandatoryFieldList(List<AccountFlowParam> paramList) {

        Assert.notEmpty(paramList, "参数不能为空");
        paramList.stream().forEach(accountFlowParam -> checkMandatoryField(accountFlowParam));
    }

    /**
     * 校验必填项
     *
     * @param param 入参
     */
    @Override
    public void checkMandatoryField(AccountFlowParam param) {
        if (StringUtils.isBlank(param.getAccountType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户类型为空!");
        }
        if (StringUtils.isBlank(param.getSourceDoc())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单据为空!");
        }
        if (StringUtils.isBlank(param.getSourceNo())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "来源单号为空!");
        }
        if (StringUtils.isBlank(param.getTransactionType())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易类型为空!");
        }
        if (Objects.isNull(param.getAmount())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "发生金额为空!");
        }
        if (Objects.isNull(param.getTransactionTime())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "交易日期为空!");
        }
        if (StringUtils.isBlank(param.getAccountCode())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户编码为空!");
        }
        if (StringUtils.isBlank(param.getAccountName())) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "账户名称为空!");
        }

    }

    /**
     * 校验是否有重复数据或脏数据、数据结构是否正确：
     * 必须每次只有一种数据来源，要不数据融合在一起无法判断谁先谁后，会涉及到上一条数据问题。
     * 同种数据来源，也应该是单次的（一个单号），就是不是批量的，会涉及到上一条数据问题。
     * 可以先判断数据来源是否是多种，再判断账户类型+数据来源+交易类型是否是唯一
     *
     * @param paramList 入参
     */
    @Override
    public void checkRepeatability(List<AccountFlowParam> paramList) {
        val dataSourceList = paramList.stream().map(AccountFlowParam::getDataSource).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(dataSourceList) && dataSourceList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "存在多种数据来源!");
        }
        Map<String, List<AccountFlowParam>> accountFlowListMap = paramList.stream().collect(Collectors.groupingBy(
                i -> i.getAccType() + FinConstant.LINE_SPLIT + i.getDataSource() + FinConstant.LINE_SPLIT + i.getTransactionType()));
        for (String accountFlowKey : accountFlowListMap.keySet()) {
            val accountFlowList = accountFlowListMap.get(accountFlowKey);

            String[] keyParams = accountFlowKey.split(FinConstant.LINE_SPLIT);
            String dataSource = keyParams[FinConstant.ARRAY_INDEX_1];
            //数据来源是调剂单审核通过时，交易类型可能是相同的，所以使用拆分类型来区分
            if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TJD03.getValueCode())) {
                List<String> splitTypeList = accountFlowList.stream().map(AccountFlowParam::getSplitType).distinct().collect(Collectors.toList());
                if (!Objects.equals(accountFlowList.size(), splitTypeList.size())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "调剂单数据不唯一，无法处理!");
                }
            } else {
                if (!CollectionUtils.isEmpty(accountFlowList) && accountFlowList.size() > 1) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据不唯一，无法处理!");
                }
            }

        }
    }


    /**
     * 校验必填项02:年月日期、账户编码、账户类型
     *
     * @param pageParam 查询入参
     */
    @Override
    public void checkMandatoryField02(AccountFlowPageParam pageParam) {
        if (StringUtils.isBlank(pageParam.getYearMonthStr())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "年月日期为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户编码为空!");
        }
        if (StringUtils.isBlank(pageParam.getAccType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户类型为空!");
        }
    }


    /**
     * 经销商端查询相关统计金额：储值账户金额、返利账户金额
     *
     * @param param 入参
     * @return 相关统计金额出参对象
     */
    @Override
    public AccountFlowDealerVO dealerSelectAccAmt(AccountFlowDealerParam param) {
        AccountFlowDealerVO accountFlowDealerVO = new AccountFlowDealerVO();
        accountFlowDealerVO.setCzAccAmt(BigDecimal.ZERO);
        accountFlowDealerVO.setFlAccAmt(BigDecimal.ZERO);

        //整车/配件储值账户金额
        Optional<AccountFlowVO> czOptional = accountFlowCommonService.selectByAccCode(param.getCzAccCode());
        czOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setCzAccAmt(accountFlowVO.getAccAmt());
        });

        //整车/配件返利账户金额
        Optional<AccountFlowVO> flOptional = accountFlowCommonService.selectByAccCode(param.getFlAccCode());
        flOptional.ifPresent(accountFlowVO -> {
            accountFlowDealerVO.setFlAccAmt(accountFlowVO.getAccAmt());
        });

        return accountFlowDealerVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void approve(AccountFlowParam accountFlowParam) {
        checkApproveParam(accountFlowParam);
        approve(accountFlowParam, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode(), true);
    }

    @Override
    public void autoApprove(AccountFlowParam accountFlowParam) {
        updateAccountAmountAndApprove(accountFlowParam, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void reject(AccountFlowParam accountFlowParam) {
        checkApproveParam(accountFlowParam);
        approve(accountFlowParam, UdcEnum.APPLY_STATUS_REJECTED.getValueCode(), false);
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> queryByAccountFlowQueryParam(AccountFlowQueryParam accountFlowQueryParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.queryByAccountFlowQueryParam(accountFlowQueryParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    public void accountFlowCallBack(String doc, AccountFlowDO accountFlowDO, boolean isApprove) {

        if (UdcEnum.DOC_CLS_SVO.getValueCode().equals(doc) || UdcEnum.DOC_CLS_AO.getValueCode().equals(doc)) {
            return;
        }
//        OrderAccountFlowParam orderAccountFlowParam = new OrderAccountFlowParam();
//        //销售订单id
//        orderAccountFlowParam.setSoId(accountFlowDO.getSourceId());
//        //订单类别
//        orderAccountFlowParam.setDocCls(accountFlowDO.getSourceDoc());
//        //流水号
//        orderAccountFlowParam.setReceiptSerial(accountFlowDO.getFlowNo());
//        //交易类型
//        orderAccountFlowParam.setTransType(accountFlowDO.getTransactionType());
//        //发生金额
//        orderAccountFlowParam.setReceiptAmt(accountFlowDO.getAmount());
//        //交易日期
//        orderAccountFlowParam.setReceiptDate(accountFlowDO.getTransactionTime());
//        //账户编码
//        orderAccountFlowParam.setPayAccNo(accountFlowDO.getAccountCode());
//        //账户名称
//        orderAccountFlowParam.setPayAccName(accountFlowDO.getAccountName());
//        //账户名称
//        orderAccountFlowParam.setIsApprove(isApprove);
//
//        rmiOrderRpcService.orderAccountFlow(orderAccountFlowParam);
    }


    private void checkApproveParam(AccountFlowParam accountFlowParam) {
        Assert.notNull(accountFlowParam, "参数不能为空");
        Assert.notNull(accountFlowParam.getId(), "id不能为空");
        Assert.notEmpty(accountFlowParam.getAccountCode(), "账户编码不能为空");
        Assert.notEmpty(accountFlowParam.getTransactionType(), "交易类型不能为空");
        Assert.notNull(accountFlowParam.getAmount(), "发生金额不能为空");
    }

    private void approve(AccountFlowParam accountFlowParam, String state, boolean isApprove) {


        Optional<AccountFlowDO> accountFlowDOOptional = accountFlowRepo.findById(accountFlowParam.getId());
        AccountFlowDO accountFlowDO = accountFlowDOOptional.get();
        Assert.equals(UdcEnum.APPLY_STATUS_DOING.getValueCode(), accountFlowDO.getOrderState(), "只有审批中状态单据才可审批");

        //账户占用特殊处理
       /* if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
            updateAccountAmountFor_IO_TYPE_T(accountFlowParam, state);
        }else{
            updateAccountAmountOther(accountFlowParam, state);
        }*/
        updateAccountAmountAndApprove(accountFlowParam, state);
        //更新流水单状态
       /* SysUserDTO user = getUser();

        accountFlowParam.setOrderState(state);
        accountFlowParam.setAuditDate(LocalDateTime.now());
        accountFlowParam.setAuditUserName(FinConstant.SYSTEM);

        if (user != null) {
            accountFlowParam.setAuditUserId(user.getId());
            accountFlowParam.setAuditUserName(user.getLastName());
        }

        accountFlowRepoProc.approveAccountFlow(accountFlowParam);*/
    }

    private void updateAccountAmount(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {

            AccountDO accountDO = accountRepoProc.findByAccountCode(accountFlowParam.getAccountCode());

            BigDecimal accountAmount = accountDO.getAccountAmount();
            BigDecimal accountOccupancyAmount = accountDO.getAccountOccupancyAmount();

            Assert.equals(accountDO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");

            if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountAmount = accountAmount.add(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountAmount = accountAmount.subtract(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountOccupancyAmount = accountOccupancyAmount.add(accountFlowParam.getAmount());
            }

            if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                accountOccupancyAmount = accountOccupancyAmount.subtract(accountFlowParam.getAmount());
            }
            accountDO.setAccountAmount(accountAmount);
            accountDO.setAccountOccupancyAmount(accountOccupancyAmount);
            accountDO.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(accountAmount);
            accountFlowParam.setAccountOccupancyAmount(accountOccupancyAmount);
            accountFlowParam.setAccountAvailableAmount(accountAmount.subtract(accountOccupancyAmount));
            accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            accountFlowRepoProc.updateAccountFlowAmount(accountFlowParam);

        }
    }

    private void updateAccountAmountFor_IO_TYPE_T(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {
            AccountVO accountVOOld = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(accountVOOld.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
            int res = accountRepo.updateAccountOccupancyAmountAndAccountAvailableAmountById(accountVOOld.getId(), accountFlowParam.getAmount());
            log.info("账户流水生成，更新记录数：{}", res);
            if (res < 1) {
                log.error("账户流水生成,更新账户金额失败,可用金额不足");
                throw new BusinessException("账户流水生成,更新账户金额失败,可用金额不足");
            }
            AccountVO accountVO = accountRepoProc.get(accountVOOld.getId());
            if (accountVO == null) {
                log.error("账户流水生成,查询账户信息失败");
                throw new BusinessException("账户流水生成,查询账户信息失败");
            }
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(accountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(accountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(accountVO.getAccountAvailableAmount());
            //accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            //更新流水单状态
            SysUserDTO user = getUser();
            accountFlowParam.setOrderState(state);
            accountFlowParam.setAuditDate(LocalDateTime.now());
            accountFlowParam.setAuditUserName(FinConstant.SYSTEM);
            if (user != null) {
                accountFlowParam.setAuditUserId(user.getId());
                accountFlowParam.setAuditUserName(user.getLastName());
            }
            accountFlowRepoProc.updateAccountFlowAmountById(accountFlowParam);
        }
    }

    private void updateAccountAmountOther(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {

            AccountVO oldAccountVO = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(oldAccountVO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");

            if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateAddAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户增加流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户增加流水生成,更新账户金额失败!");
                    throw new BusinessException("账户增加流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户扣减流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户扣减流水生成,更新账户金额失败!");
                    throw new BusinessException("账户扣减流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户占用释放流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户占用释放流水生成,更新账户金额失败!");
                    throw new BusinessException("账户占用释放流水生成,更新账户金额失败!");
                }
            } else {
                throw new BusinessException("账户未匹配到交易类型!");
            }

            AccountVO newAccountVO = accountRepoProc.get(oldAccountVO.getId());
            if (newAccountVO == null) {
                log.error("账户流水生成,查询更新后账户信息失败");
                throw new BusinessException("账户流水生成,查询更新后账户信息失败");
            }
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(newAccountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(newAccountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(newAccountVO.getAccountAvailableAmount());
            //accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            //更新流水单状态
            SysUserDTO user = getUser();
            accountFlowParam.setOrderState(state);
            accountFlowParam.setAuditDate(LocalDateTime.now());
            accountFlowParam.setAuditUserName(FinConstant.SYSTEM);
            if (user != null) {
                accountFlowParam.setAuditUserId(user.getId());
                accountFlowParam.setAuditUserName(user.getLastName());
            }
            accountFlowRepoProc.updateAccountFlowAmountById(accountFlowParam);
        }
    }

    private void updateAccountAmountAndApprove(AccountFlowParam accountFlowParam, String state) {
        if (UdcEnum.APPLY_STATUS_COMPLETE.getValueCode().equals(state)) {
            AccountVO oldAccountVO = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(oldAccountVO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户流水生成,更新账户金额失败,可用金额不足");
                    throw new BusinessException("账户流水生成,更新账户金额失败,可用金额不足");
                }
            } else if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateAddAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户增加流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户增加流水生成,更新账户金额失败!");
                    throw new BusinessException("账户增加流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户扣减流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户扣减流水生成,更新账户金额失败!");
                    throw new BusinessException("账户扣减流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountRepo.updateSubtractAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户占用释放流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户占用释放流水生成,更新账户金额失败!");
                    throw new BusinessException("账户占用释放流水生成,更新账户金额失败!");
                }
            } else {
                throw new BusinessException("账户未匹配到交易类型!");
            }

            AccountVO newAccountVO = accountRepoProc.get(oldAccountVO.getId());
            if (newAccountVO == null) {
                log.error("账户流水生成,查询更新后账户信息失败");
                throw new BusinessException("账户流水生成,查询更新后账户信息失败");
            }
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(newAccountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(newAccountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(newAccountVO.getAccountAvailableAmount());
            //accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            //更新流水单状态
            SysUserDTO user = getUser();
            accountFlowParam.setOrderState(state);
            accountFlowParam.setAuditDate(LocalDateTime.now());
            accountFlowParam.setAuditUserName(FinConstant.SYSTEM);
            if (user != null) {
                accountFlowParam.setAuditUserId(user.getId());
                accountFlowParam.setAuditUserName(user.getLastName());
            }
            accountFlowRepoProc.updateAccountFlowAmountById(accountFlowParam);
        }
    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            return null;
        }
        return userDetails.getUser();
    }

    @Override
    //@Transactional(rollbackFor = {Exception.class})
    public String generateAccountFlow(AccountFlowParam param) {
        log.info("生成账户流水，接收到参数{}", JSONUtil.toJsonStr(param));
        //参数检查
        checkGenerateAccountFlowParam(param);

        AccountFlowParam accountFlowParam = AccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildAccountFlowParam(accountFlowParam);

        //获取账户信息
        AccountDO accountDO = getAccountMsg(accountFlowParam);
        log.info("账户信息:{}", JSONUtil.toJsonStr(accountDO));
        if (param.getIgnore() != null && param.getIgnore()) {
            log.info("允许忽略存在流水的校验");
            boolean flowExist = checkIfSameFlow(accountDO, accountFlowParam);
            if (flowExist) {
                log.info("存在流水,直接返回,账户信息:{},参数:{}", JSONUtil.toJsonStr(accountDO),
                        JSONUtil.toJsonStr(accountFlowParam));
                return "Flow9999";
            }
        } else {
            //幂等
            // 账户流水校验重复：增加是否启用工作流判断，如果单据已有工作流，就允许重复；如果单据没有工作流，就不允许重复
            if (StrUtil.isEmpty(param.getWorkflowProcInstId())) {
                checkIdempotent(accountDO, accountFlowParam);
            }
        }

        //获取账户规则
        //List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = getAccountRuleMsg(accountDO, accountFlowParam);
        List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = getAccountRuleMsgWithRedis(accountDO, accountFlowParam);

        //获取账户事务码
        //List<AccountIocDTO> accountIocDTOList = getAccountIocMsg(accountRuleConfigPageVOList);
        List<AccountIocDTO> accountIocDTOList = getAccountIocMsgWithRedis(accountRuleConfigPageVOList);
        //根据优先级排序
        accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(AccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
        /*  checkAccountAmount(accountRuleConfigPageVOList, accountDO, accountFlowParam, accountIocDTOList);*/

        //生成账户流水
      /*  List<AccountFlowParam> accountFlowParamList = generateAccountFlow(accountRuleConfigPageVOList,
                accountDO,
                accountIocDTOList,
                accountFlowParam);*/

        //流水审批
        /*  accountFlowReview(accountFlowParamList);*/
        //加锁begin
        RedissonRedLock redLock = null;
        try {
            String lockKey = "ACCOUNT_FLOW_GENERATE:" + accountDO.getAccountCode();
            log.info("账户流水生成，锁:{}", lockKey);
            redLock = redLockUtils.getRedLock(lockKey);
            boolean lockFlag = redLock.tryLock(3, 300, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("账户流水生成，获取锁失败");
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户流水生成并发过大,请稍后再试！");
            }
            log.info("账户流水生成，获取锁成功");
            //再次获取账户信息
            AccountVO accountVO = accountRepoProc.getVOByCode(accountDO.getAccountCode());
            AccountDO accountDO1 = AccountConvert.INSTANCE.VO2DO(accountVO);
            //校验账户余额
            checkAccountAmount(accountRuleConfigPageVOList, accountDO1, accountFlowParam, accountIocDTOList);
            //生成流水单
       /*          List<AccountFlowParam> accountFlowParamList = generateAccountFlow(accountRuleConfigPageVOList,
                    accountDO1,
                    accountIocDTOList,
                    accountFlowParam);*/
            //构造待保存流水参数
            log.info("支付配置{}，流水参数{}", JSONUtil.toJsonStr(accountRuleConfigPageVOList), JSONUtil.toJsonStr(accountFlowParam));
            List<AccountFlowParam> accountFlowParamList = buildAccountFlow(accountRuleConfigPageVOList,
                    accountDO1,
                    accountIocDTOList,
                    accountFlowParam);

            log.info("账户类型{}，流水参数{}", JSONUtil.toJsonStr(accountVO.getAccountType()), JSONUtil.toJsonStr(accountFlowParamList));
            // 非储值账户出入账公司处理
            boolean out_flag = FinConstant.FLOW_OPT_DOC_STATUS_OUT.contains(param.getSourceDocStatus());
            accountFlowParamList.stream().filter(p -> UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(p.getAccountType())).forEach(p -> {
                if (out_flag) {
                    p.setFromOuCode(p.getTOuCode());
                    p.setToOuCode(p.getOuCode());
                } else {
                    p.setFromOuCode(p.getOuCode());
                    p.setToOuCode(p.getTOuCode());
                }
                p.setCompanyCode(p.getTOuCode());
            });

            // 储值账户扫呗分账
            List<AccountFlowParam> storeAccountFlowParamList = accountFlowParamList.stream()
                    .filter(p -> UdcEnum.ACCOUNT_TYPE_STORE.getValueCode().equals(p.getAccountType()))
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(storeAccountFlowParamList) && !"AO".equals(param.getSourceDoc())) {
                saoBeiTrans(storeAccountFlowParamList, out_flag);
            }

            List<AccountFlowParam> autoApproveAccountFlowParamList = new ArrayList<>();
            List<AccountFlowParam> notAutoApproveAccountFlowParamList = new ArrayList<>();
            accountFlowParamList.stream().forEach(item -> {
                if (item.isAutomaticReview()) {
                    autoApproveAccountFlowParamList.add(item);
                } else {
                    notAutoApproveAccountFlowParamList.add(item);
                }
            });
            //自动审批流水生成
            if (!CollectionUtils.isEmpty(autoApproveAccountFlowParamList)) {
                accountFlowDomainService.updateAccountBalanceAndGenerateAccountFlow(autoApproveAccountFlowParamList);
            }
            //非自动审批流水生成,只保存流水数据
            if (!CollectionUtils.isEmpty(notAutoApproveAccountFlowParamList)) {
                accountFlowDomainService.saveAccountFlowList(notAutoApproveAccountFlowParamList);
            }

            // 推送BIP
            for (AccountFlowParam v : accountFlowParamList) {
                SpringContextHolder.getBean(BipFinService.class).pushReceiptOfStorageAccountFlow(v.getId());
            }
            return accountFlowParamList.get(0).getFlowNo();
        } catch (InterruptedException e) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户流水生成，获取redis锁异常,请稍后再试！");
        } finally {
            redLock.unlock();
            log.info("账户流水生成，释放锁成功");
        }
        //加锁end

    }


    private void saoBeiTrans(List<AccountFlowParam> accountFlowParamList, Boolean out_flag) {
        log.info("开始处理账户流水{}", JSONUtil.toJsonStr(accountFlowParamList));
        String storeOuCode = accountFlowParamList.get(0).getOuCode();
        List<String> ouCodes = accountFlowParamList.stream().map(AccountFlowParam::getTOuCode).filter(StringUtils::isNotEmpty).distinct().collect(Collectors.toList());

        // 查询门店信息获取门店储值公司
        ApiResult<OrgStoreBaseRpcDTO> storeApiResult = orgStoreRpcService.findSimpleByCode(storeOuCode);
        if (storeApiResult.isFailed() || storeApiResult.getData() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "门店" + storeOuCode + "储值公司查询失败");
        }
        OrgStoreBaseRpcDTO storeBaseRpcDTO = storeApiResult.getData();
        if (StringUtils.isEmpty(storeBaseRpcDTO.getStoredOuCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "门店" + storeOuCode + "储值公司不能为空");
        }
        // 甲指乙采、门店账号类型为 施工方： "CONTRACTOR"
        if("CONTRACTOR".equals(storeBaseRpcDTO.getStoreType())){
            handleSaobeiTransactionsDbo(accountFlowParamList,storeBaseRpcDTO, ouCodes);
        }else {
            handleSaobeiTransactions(accountFlowParamList, out_flag, storeBaseRpcDTO, ouCodes);
        }
    }

    /**
     *  非甲指乙采 扫呗支付
     * @param accountFlowParamList
     * @param out_flag
     * @param storeBaseRpcDTO
     * @param ouCodes
     */
    private void handleSaobeiTransactions(List<AccountFlowParam> accountFlowParamList, Boolean out_flag, OrgStoreBaseRpcDTO storeBaseRpcDTO, List<String> ouCodes) {
        String storeStoredOuCode = storeBaseRpcDTO.getStoredOuCode();

        ouCodes.add(storeStoredOuCode);
        OrgBankAccQueryRpcParam bankAccQueryParam = new OrgBankAccQueryRpcParam();
        bankAccQueryParam.setAccType(UdcEnum.ORG_BANK_TYPE_SAO_BEI.getValueCode());
        bankAccQueryParam.setStatus(UdcEnum.ORG_BANK_STATUS_ACTIVE.getValueCode());
        bankAccQueryParam.setOuCodes(ouCodes);
        List<OrgOuBankAccRpcDTO> bankList = orgAddrRpcService.findOuBankByParam(bankAccQueryParam);
        Map<String, String> ouAccountMap = bankList.stream().collect(Collectors.toMap(OrgOuBankAccRpcDTO::getOuCode, OrgOuBankAccRpcDTO::getBankAcc));

        // 校验公司是否绑定扫呗账号，同时过滤掉施工方类型(CONTRACTOR)的门店
        List<String> noAccountOuCodes = ouCodes.stream()
                .filter(ouCode -> !ouAccountMap.containsKey(ouCode))
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(noAccountOuCodes)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "公司未绑定扫呗账户：" + String.join(",", noAccountOuCodes));

        }

        accountFlowParamList.forEach(flow -> {
            SaobeiDoTransRequest request = new SaobeiDoTransRequest();
            if (out_flag) {
                request.setAccount_out(ouAccountMap.get(flow.getTOuCode()));
                request.setAccount_in(ouAccountMap.get(storeStoredOuCode));
                flow.setFromOuCode(flow.getTOuCode());
                flow.setToOuCode(storeStoredOuCode);
            } else {
                request.setAccount_out(ouAccountMap.get(storeStoredOuCode));
                request.setAccount_in(ouAccountMap.get(flow.getTOuCode()));
                flow.setFromOuCode(storeStoredOuCode);
                flow.setToOuCode(flow.getTOuCode());
            }

            flow.setCompanyCode(flow.getTOuCode());
            flow.setFromAccountNo(request.getAccount_out());
            flow.setToAccountNo(request.getAccount_in());

            // 出入账公司不同时才调扫呗
            if (flow.getFromAccountNo().equals(flow.getToAccountNo())) {
                flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_4.getValueCode());
            } else {
                flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_0.getValueCode());
//
//                request.setAmt(flow.getAmount().multiply(ONE_HUNDRED_BIGDECIMAL).stripTrailingZeros().toPlainString());
//                request.setTrade_no(flow.getFlowNo());
//                request.setTrans_type("1");
//                request.setOrder_body("储值账户分账");
//                try {
//                    var response = saobeiAccountService.doAlltrans(request);
//                    if (SaoBeiConstant.SUCCESS_CODE.equals(response.getReturn_code()) && SaoBeiConstant.SUCCESS_CODE.equals(response.getResult_code())) {
//                        flow.setRelateFlowNo(StringUtils.isBlank(response.getOut_trade_no()) ? response.getRev_trade_no() : response.getOut_trade_no() );
//                        flow.setSyncStatus(response.getTrade_status());
//                        if ("3".equals(response.getTrade_status())) {
//                        } else if ("1".equals(response.getTrade_status())) {
//                            // 分账成功
//                        } else {
//                            // 分账失败
//                            String errorMsg = StrUtil.isNotBlank(response.getReturn_msg()) ? response.getReturn_msg() : "CBK余额分账交易失败";
//                            flow.setSyncLog("CBK余额分账交易失败:" + errorMsg);
//                            log.error("扫呗:{},CBK余额分账交易失败，原因：{}", request.getTrade_no(), errorMsg);
//                        }
//                    } else {
//                        // 分账接口调用失败 todo
//                        String errorMsg = response.getReturn_msg();
//                        flow.setSyncLog("CBK余额分账失败:" + errorMsg);
//                        flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_2.getValueCode());
//                        log.error("扫呗:{},CBK余额分账失败，原因：{}", request.getTrade_no(), errorMsg);
//                    }
//                } catch (Exception e) {
//                    log.info("扫呗分账接口调用失败，参数{}，原因：{}", JSONObject.toJSONString(request), e.getMessage());
//                    flow.setSyncLog("扫呗分账接口调用失败:" + e.getMessage());
//                    flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_2.getValueCode());
////                    throw new com.elitescloud.boot.exception.BusinessException(ApiCode.BUSINESS_EXCEPTION, "扫呗分账失败:" + e.getMessage());
//                }
            }
        });
    }

    /**
     * 甲指乙采 扫呗支付
     *
     * @param accountFlowParamList
     *
     */
    private void handleSaobeiTransactionsDbo(List<AccountFlowParam> accountFlowParamList, OrgStoreBaseRpcDTO storeBaseRpcDTO, List<String> ouCodes) {

        String storeStoredOuCode = storeBaseRpcDTO.getStoredOuCode();

        ouCodes.add(storeStoredOuCode);
        OrgBankAccQueryRpcParam bankAccQueryParam = new OrgBankAccQueryRpcParam();
        bankAccQueryParam.setAccType(UdcEnum.ORG_BANK_TYPE_SAO_BEI.getValueCode());
        bankAccQueryParam.setStatus(UdcEnum.ORG_BANK_STATUS_ACTIVE.getValueCode());
        bankAccQueryParam.setOuCodes(ouCodes);
        List<OrgOuBankAccRpcDTO> bankList = orgAddrRpcService.findOuBankByParam(bankAccQueryParam);
        Map<String, String> ouAccountMap = bankList.stream().collect(Collectors.toMap(OrgOuBankAccRpcDTO::getOuCode, OrgOuBankAccRpcDTO::getBankAcc));
        log.info("ouAccountMap:{}", ouAccountMap);
        // 校验公司是否绑定扫呗账号，同时过滤掉施工方类型(CONTRACTOR)的门店
        List<String> noAccountOuCodes = ouCodes.stream()
                .filter(ouCode -> !ouAccountMap.containsKey(ouCode) &&
                        !"CONTRACTOR".equals(storeBaseRpcDTO.getStoreType()))
                .collect(Collectors.toList());

        if (!CollectionUtils.isEmpty(noAccountOuCodes)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "公司未绑定扫呗账户：" + String.join(",", noAccountOuCodes));

        }

        accountFlowParamList.forEach(flow -> {
            SaobeiDoTransRequest request = new SaobeiDoTransRequest();

            request.setAccount_out(ouAccountMap.get(flow.getTOuCode()));
            request.setAccount_in(ouAccountMap.get(storeStoredOuCode));
            flow.setFromOuCode(flow.getTOuCode());
            flow.setToOuCode(storeStoredOuCode);
            flow.setCompanyCode(flow.getTOuCode());
            flow.setFromAccountNo(request.getAccount_out());
            flow.setToAccountNo(request.getAccount_in());

            // 出入账公司不同时才调扫呗
            if (flow.getFromAccountNo().equals(flow.getToAccountNo())) {
                flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_4.getValueCode());
            } else {
                flow.setSyncStatus(UdcEnum.FLOW_SYNC_STATUS_0.getValueCode());
            }
        });
    }

    private void accountFlowReview(List<AccountFlowParam> accountFlowParamList) {
        accountFlowParamList.stream().forEach(item -> {
            if (item.isAutomaticReview()) {
                accountFlowService.approve(item);
            }
        });
    }

    //@Transactional(rollbackFor = {Exception.class},propagation = Propagation.REQUIRES_NEW)
    @Override
    public void updateAccountBalanceAndGenerateAccountFlow(List<AccountFlowParam> accountFlowParamList) {
        accountFlowParamList.stream().forEach(item -> {
            AccountFlowParam accountFlowParam = item;
            AccountVO oldAccountVO = accountRepoProc.getVOByCode(accountFlowParam.getAccountCode());
            Assert.equals(oldAccountVO.getState(), UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), "账户已经被禁用");
            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountDomainService.updateAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户流水生成,更新账户金额失败,可用金额不足");
                    throw new BusinessException("账户流水生成,更新账户金额失败,可用金额不足");
                }
            } else if (UdcEnum.IO_TYPE_I.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountDomainService.updateAddAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户增加流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户增加流水生成,更新账户金额失败!");
                    throw new BusinessException("账户增加流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_O.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountDomainService.updateSubtractAccountAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户扣减流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户扣减流水生成,更新账户金额失败!");
                    throw new BusinessException("账户扣减流水生成,更新账户金额失败!");
                }
            } else if (UdcEnum.IO_TYPE_R.getValueCode().equals(accountFlowParam.getTransactionType())) {
                int res = accountDomainService.updateSubtractAccountOccupancyAmountAndAccountAvailableAmountById(oldAccountVO.getId(), accountFlowParam.getAmount());
                log.info("账户占用释放流水生成，更新记录数：{}", res);
                if (res < 1) {
                    log.error("账户占用释放流水生成,更新账户金额失败!");
                    throw new BusinessException("账户占用释放流水生成,更新账户金额失败!");
                }
            } else {
                throw new BusinessException("账户未匹配到交易类型!");
            }

            AccountVO newAccountVO = accountRepoProc.get(oldAccountVO.getId());
            if (newAccountVO == null) {
                log.error("账户流水生成,查询更新后账户信息失败");
                throw new BusinessException("账户流水生成,查询更新后账户信息失败");
            }
            //更新审批中的流水的状态
            accountFlowParam.setAccountAmount(newAccountVO.getAccountAmount());
            accountFlowParam.setAccountOccupancyAmount(newAccountVO.getAccountOccupancyAmount());
            accountFlowParam.setAccountAvailableAmount(newAccountVO.getAccountAvailableAmount());
            //accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
            //更新流水单状态
            SysUserDTO user = getUser();
            accountFlowParam.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
            accountFlowParam.setAuditDate(LocalDateTime.now());
            accountFlowParam.setAuditUserName(FinConstant.SYSTEM);
            if (user != null) {
                accountFlowParam.setAuditUserId(user.getId());
                accountFlowParam.setAuditUserName(user.getLastName());
            }
            TransactionStatus transactionStatus1 = TransactionAspectSupport.currentTransactionStatus();
            //保存流水
            //创建账户流水
            AccountFlowDO accountFlowDO = accountFlowService.saveAccountFlowWithOutTx(accountFlowParam);
            TransactionStatus transactionStatus2 = TransactionAspectSupport.currentTransactionStatus();
            accountFlowParam.setId(accountFlowDO.getId());
        });
    }

//    private void sendAccountFlowReviewMsg(List<AccountFlowParam> accountFlowParamList) {
//        accountFlowParamList.stream().forEach(item -> {
//            if (item.isAutomaticReview()) {
//                //发送流水审批消息
//                try {
//                    boolean result = finSalMqProcessor.approveAccountFlowOutput().send(MessageBuilder.withPayload(item).build());
//                    if (!result) {
//                        log.error("流水自动审批消息发送失败");
//                    }
//                } catch (Exception e) {
//                    log.error("流水自动审批消息发送失败，失败原因：" + e.getMessage(), e);
//                }
//            }
//        });
//    }

    private List<AccountFlowParam> generateAccountFlow(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                                       AccountDO accountDO,
                                                       List<AccountIocDTO> accountIocDTOList,
                                                       AccountFlowParam accountFlowParam) {

        List<AccountRuleConfigPageVO> finalAccountRuleConfigPageVOList = accountRuleConfigPageVOList;
        List<AccountFlowParam> accountFlowParamList = new ArrayList<>();

        Integer amtRound = SysUtils.getAmtPlace();

        //生成流水单
        finalAccountRuleConfigPageVOList.stream().forEach(accountRuleConfigPageVO -> {

            AccountFlowParam param = buildAccountFlowParam(accountFlowParam,
                    accountDO,
                    accountIocDTOList,
                    accountRuleConfigPageVO, amtRound);

            //创建账户流水
            AccountFlowDO accountFlowDO = accountFlowService.saveAccountFlow(param);

            param.setId(accountFlowDO.getId());
            param.setFlowNo(accountFlowDO.getFlowNo());

            accountFlowParamList.add(param);
        });

        return accountFlowParamList;
    }

    private List<AccountFlowParam> buildAccountFlow(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                                    AccountDO accountDO,
                                                    List<AccountIocDTO> accountIocDTOList,
                                                    AccountFlowParam originalAccountFlowParam) {

        List<AccountRuleConfigPageVO> finalAccountRuleConfigPageVOList = accountRuleConfigPageVOList;
        List<AccountFlowParam> accountFlowParamList = new ArrayList<>();

        Integer amtRound = SysUtils.getAmtPlace();

        //生成流水单
        finalAccountRuleConfigPageVOList.stream().forEach(accountRuleConfigPageVO -> {

            AccountFlowParam accountParam = buildAccountFlowParam(originalAccountFlowParam,
                    accountDO,
                    accountIocDTOList,
                    accountRuleConfigPageVO, amtRound);

            boolean addFlag = addAccountFlowParam(accountFlowParamList, accountRuleConfigPageVO, originalAccountFlowParam, accountParam, amtRound);

            //创建账户流水
           /* AccountFlowDO accountFlowDO = accountFlowService.saveAccountFlow(param);

            param.setId(accountFlowDO.getId());
            param.setFlowNo(accountFlowDO.getFlowNo());*/

            if (!addFlag) {
                accountFlowParamList.add(accountParam);
                accountParam.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
            }
        });

        return accountFlowParamList;
    }


    public boolean addAccountFlowParam(List<AccountFlowParam> accountFlowParamList, AccountRuleConfigPageVO accountRuleConfigPageVO, AccountFlowParam originalAccountFlowParam, AccountFlowParam accountParam, Integer amtRound) {
        // 货款
        boolean addFlag = false;
        if (originalAccountFlowParam.getItemFee() != null && originalAccountFlowParam.getItemFee().compareTo(BigDecimal.ZERO) > 0) {
            AccountFlowParam param = AccountFlowConvert.INSTANCE.copyParam(accountParam);
            //发生金额
            param.setAmount(originalAccountFlowParam.getItemFee()
                    .multiply(accountRuleConfigPageVO.getCalculatePercent())
                    .divide(ONE_HUNDRED_BIGDECIMAL, amtRound, RoundingMode.HALF_UP));
            param.setBusinessType(UdcEnum.ACCOUNT_FLOW_BUSINESS_TYPE_ITEM.getValueCode());
            param.setTOuCode(accountParam.getItemFeeOuCode());
            param.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));

            accountFlowParamList.add(param);
            addFlag = true;
        }

        // 营销费
        if (originalAccountFlowParam.getMarketingFee() != null && originalAccountFlowParam.getMarketingFee().compareTo(BigDecimal.ZERO) > 0) {
            AccountFlowParam param = AccountFlowConvert.INSTANCE.copyParam(accountParam);
            //发生金额
            param.setAmount(originalAccountFlowParam.getMarketingFee()
                    .multiply(accountRuleConfigPageVO.getCalculatePercent())
                    .divide(ONE_HUNDRED_BIGDECIMAL, amtRound, RoundingMode.HALF_UP));
            param.setBusinessType(UdcEnum.ACCOUNT_FLOW_BUSINESS_TYPE_MARKETING.getValueCode());
            param.setTOuCode(accountParam.getMarketingFeeOuCode());
            param.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
            accountFlowParamList.add(param);
            addFlag = true;
        }

        // 科技费
        if (originalAccountFlowParam.getTechFee() != null && originalAccountFlowParam.getTechFee().compareTo(BigDecimal.ZERO) > 0) {
            AccountFlowParam param = AccountFlowConvert.INSTANCE.copyParam(accountParam);
            //发生金额
            param.setAmount(originalAccountFlowParam.getTechFee()
                    .multiply(accountRuleConfigPageVO.getCalculatePercent())
                    .divide(ONE_HUNDRED_BIGDECIMAL, amtRound, RoundingMode.HALF_UP));
            param.setBusinessType(UdcEnum.ACCOUNT_FLOW_BUSINESS_TYPE_TECH.getValueCode());
            param.setTOuCode(accountParam.getTechFeeOuCode());
            param.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
            accountFlowParamList.add(param);
            addFlag = true;
        }

        // 科技费
        if (originalAccountFlowParam.getOperationFee() != null && originalAccountFlowParam.getOperationFee().compareTo(BigDecimal.ZERO) > 0) {
            AccountFlowParam param = AccountFlowConvert.INSTANCE.copyParam(accountParam);
            //发生金额
            param.setAmount(originalAccountFlowParam.getOperationFee()
                    .multiply(accountRuleConfigPageVO.getCalculatePercent())
                    .divide(ONE_HUNDRED_BIGDECIMAL, amtRound, RoundingMode.HALF_UP));
            param.setBusinessType(UdcEnum.ACCOUNT_FLOW_BUSINESS_TYPE_OPERATION.getValueCode());
            param.setTOuCode(accountParam.getOperationFeeOuCode());
            param.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
            accountFlowParamList.add(param);
            addFlag = true;
        }
        return addFlag;
    }

    @Override
    public Boolean checkAccountBalance(AccountFlowParam param) {
        log.info("财务校验账户余额接口，参数：{}", JSON.toJSONString(param));
        //参数检查
        checkGenerateAccountFlowParam(param);

        AccountFlowParam accountFlowParam = AccountFlowConvert.INSTANCE.copyParam(param);

        //组装财务域udc
        buildAccountFlowParam(accountFlowParam);

        //获取账户信息
        AccountDO accountDO = getAccountMsg(accountFlowParam);

        //幂等
        checkIdempotent(accountDO, accountFlowParam);

        //获取账户规则
        List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = getAccountRuleMsg(accountDO, accountFlowParam);

        //获取账户事务码
        List<AccountIocDTO> accountIocDTOList = getAccountIocMsg(accountRuleConfigPageVOList);

        //根据优先级排序
        accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream()
                .sorted(Comparator.comparing(AccountRuleConfigPageVO::getPriorityNo))
                .collect(Collectors.toList());

        //校验账户余额
        checkAccountAmount(accountRuleConfigPageVOList, accountDO, accountFlowParam, accountIocDTOList);

        return true;
    }


    private void buildAccountFlowParam(AccountFlowParam accountFlowParam) {
        accountFlowParam.setSourceDocType(accountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(accountFlowParam.getSourceDocType()));

        accountFlowParam.setSourceDocStatus(accountFlowParam.getSourceDoc()
                .concat(FinConstant.UNDERLINE)
                .concat(accountFlowParam.getSourceDocStatus()));
    }

    private void checkAccountAmount(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList,
                                    AccountDO accountDO,
                                    AccountFlowParam accountFlowParam,
                                    List<AccountIocDTO> accountIocDTOList) {

        accountRuleConfigPageVOList.stream().forEach(item -> {
            //交易类型
            AccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                            .equals(item.getIoCode()))
                    .findFirst()
                    .orElse(null);

            Assert.notNull(accountIocDTO, "没有查询到启用的账户事务");

            if (UdcEnum.IO_TYPE_T.getValueCode().equals(accountIocDTO.getIoType())) {
                Assert.isTrue(accountDO.getAccountAvailableAmount().compareTo(accountFlowParam.getSourceDocAmount()
                        .multiply(item.getCalculatePercent())
                        .divide(ONE_HUNDRED_BIGDECIMAL)) >= 0, "账户可用余额不足");
            }
        });
    }

    private void checkIdempotent(AccountDO accountDO, AccountFlowParam accountFlowParam) {

        AccountFlowQueryParam accountFlowQueryParam = new AccountFlowQueryParam();
        accountFlowQueryParam.setSourceId(accountFlowParam.getSourceId());
        accountFlowQueryParam.setSourceDoc(accountFlowParam.getSourceDoc());
        accountFlowQueryParam.setSourceDocStatus(accountFlowParam.getSourceDocStatus());
        accountFlowQueryParam.setSourceNo(accountFlowParam.getSourceNo());
        accountFlowQueryParam.setAccountType(accountDO.getAccountType());
        if (ObjUtil.equals(accountFlowParam.getOverOneFlow(), Boolean.TRUE)) {
            accountFlowQueryParam.setAccountCode(accountFlowParam.getAccountCode());
        }

        Assert.isTrue(CollectionUtils.isEmpty(accountFlowService.queryByAccountFlowQueryParam(accountFlowQueryParam)), "订单已经生成过流水");
    }

    /**
     * 是否已存在流水
     *
     * @param accountDO
     * @param accountFlowParam
     * @return true 已存在流水
     */
    private boolean checkIfSameFlow(AccountDO accountDO, AccountFlowParam accountFlowParam) {

        AccountFlowQueryParam accountFlowQueryParam = new AccountFlowQueryParam();
        accountFlowQueryParam.setSourceId(accountFlowParam.getSourceId());
        accountFlowQueryParam.setSourceDoc(accountFlowParam.getSourceDoc());
        accountFlowQueryParam.setSourceDocStatus(accountFlowParam.getSourceDocStatus());
        accountFlowQueryParam.setSourceNo(accountFlowParam.getSourceNo());
        accountFlowQueryParam.setAccountType(accountDO.getAccountType());
        if (ObjUtil.equals(accountFlowParam.getOverOneFlow(), Boolean.TRUE)) {
            accountFlowQueryParam.setAccountCode(accountFlowParam.getAccountCode());
        }
        return CollUtil.isNotEmpty(accountFlowService.queryByAccountFlowQueryParam(accountFlowQueryParam));
    }

    private List<AccountIocDTO> getAccountIocMsg(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList) {
        Set<String> ioCodeSet = new HashSet<>();
        accountRuleConfigPageVOList.stream().forEach(item -> ioCodeSet.add(item.getIoCode()));
        AccountIocParam accountIocParam = new AccountIocParam();
        accountIocParam.setIoCodeSet(ioCodeSet);
        accountIocParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<AccountIocDTO> accountIocDTOList = accountIocService.queryByAccountIocParam(accountIocParam);
        Assert.notEmpty(accountIocDTOList, "查询账户事务码失败");
        return accountIocDTOList;
    }

    private List<AccountIocDTO> getAccountIocMsgWithRedis(List<AccountRuleConfigPageVO> accountRuleConfigPageVOList) {
        List<AccountIocDTO> accountIocDTOList =
                myRedisService.getListDataFromRedis(FinRedisConstant.ACCOUNT_IOC, AccountIocDTO.class, "事务码配置", () -> {
                    AccountIocParam accountIocParam = new AccountIocParam();
                    //accountIocParam.setIoCodeSet(ioCodeSet);
                    accountIocParam.setSize(Integer.MAX_VALUE);
                    accountIocParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
                    return accountIocService.queryByAccountIocParam(accountIocParam);
                });
        Set<String> ioCodeSet = new HashSet<>();
        accountRuleConfigPageVOList.stream().forEach(item -> ioCodeSet.add(item.getIoCode()));
        if (!CollectionUtils.isEmpty(accountIocDTOList)) {
            accountIocDTOList = accountIocDTOList.stream().filter(v -> ioCodeSet.contains(v.getIoCode())).collect(Collectors.toList());
        }
        Assert.notEmpty(accountIocDTOList, "查询账户事务码失败");
        return accountIocDTOList;
    }

    public AccountFlowParam buildAccountFlowParam(AccountFlowParam accountFlowParam,
                                                  AccountDO accountDO,
                                                  List<AccountIocDTO> accountIocDTOList,
                                                  AccountRuleConfigPageVO accountRuleConfigPageVO, Integer amtRound) {

        AccountFlowParam result = new AccountFlowParam();
        //公司
        result.setCompanyCode(accountFlowParam.getOuCode());
        result.setOuCode(accountFlowParam.getOuCode());
        result.setTOuCode(accountFlowParam.getTOuCode());
        //归属加盟商
        result.setSecFranchiseeCode(accountDO.getSecFranchiseeCode());
        //流水号
//        result.setFlowNo(seqNumProvider.generateCode(Application.NAME, FinConstant.ACCOUNT_FLOW_NO, Collections.emptyList()));
        //交易类型
        AccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(accountIocRespVO -> accountIocRespVO.getIoCode()
                        .equals(accountRuleConfigPageVO.getIoCode()))
                .findFirst()
                .orElse(null);

        Assert.notNull(accountIocDTO, "没有查询到启用的账户事务");

        result.setTransactionType(accountIocDTO.getIoType());

        //发生金额
        result.setAmount(accountFlowParam.getSourceDocAmount()
                .multiply(accountRuleConfigPageVO.getCalculatePercent())
                .divide(ONE_HUNDRED_BIGDECIMAL, amtRound, RoundingMode.HALF_UP));

        //交易日期
        result.setTransactionTime(LocalDateTime.now());
        //开户主体名称
        result.setAccountHolderName(accountDO.getAccountHolderName());
        //账户类型
        result.setAccountType(accountDO.getAccountType());
        //账户编码
        result.setAccountCode(accountDO.getAccountCode());
        //账户名称
        result.setAccountName(accountDO.getAccountName());
        //来源单据
        result.setSourceDoc(accountFlowParam.getSourceDoc());
        //来源单号
        result.setSourceNo(accountFlowParam.getSourceNo());
        //来源id
        result.setSourceId(accountFlowParam.getSourceId());

        //规则编码
        result.setRuleCode(accountRuleConfigPageVO.getRuleCode());
        //规则优先级
        result.setPriorityNo(accountRuleConfigPageVO.getPriorityNo());

        //单据状态
        result.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());

        //适用单据状态
        result.setSourceDocStatus(accountFlowParam.getSourceDocStatus());

        result.setAutomaticReview(accountRuleConfigPageVO.isAutomaticReview());

        result.setAccountAmount(accountDO.getAccountAmount());

        result.setAccountOccupancyAmount(accountDO.getAccountOccupancyAmount());

        result.setAccountAvailableAmount(accountDO.getAccountAvailableAmount());
        result.setExtensionInfo(accountFlowParam.getExtensionInfo());
        result.setItemFeeOuCode(accountFlowParam.getItemFeeOuCode());
        result.setMarketingFeeOuCode(accountFlowParam.getMarketingFeeOuCode());
        result.setTechFeeOuCode(accountFlowParam.getTechFeeOuCode());
        result.setOperationFeeOuCode(accountFlowParam.getOperationFeeOuCode());
        result.setRootDoc(accountFlowParam.getRootDoc());
        result.setRootDocNo(accountFlowParam.getRootDocNo());
        result.setRemark(accountFlowParam.getRemark());
        return result;
    }

    private List<AccountRuleConfigPageVO> getAccountRuleMsg(AccountDO accountDO, AccountFlowParam accountFlowParam) {
        AccountRuleConfigQueryParam accountRuleConfigQueryParam = new AccountRuleConfigQueryParam();

        //适用单据
        accountRuleConfigQueryParam.setOptDoc(accountFlowParam.getSourceDoc());
        //适用单据类型
        accountRuleConfigQueryParam.setOptDocType(accountFlowParam.getSourceDocType());
        //适用单据状态
        accountRuleConfigQueryParam.setOptDocStatus(accountFlowParam.getSourceDocStatus());
        //适用账户类型
        accountRuleConfigQueryParam.setOptAccountType(accountDO.getAccountType());
        //指定账户
        accountRuleConfigQueryParam.setSpecificAccount(accountDO.getAccountCode());
        accountRuleConfigQueryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());

        PagingVO<AccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO = accountRuleConfigService.searchPageWithOutName(accountRuleConfigQueryParam);
        Assert.notNull(accountRuleConfigPageVOPagingVO, "查询订单能匹配到的规则失败");
        Assert.notEmpty(accountRuleConfigPageVOPagingVO.getRecords(), "没有匹配到账户规则配置");

        return accountRuleConfigPageVOPagingVO.getRecords();

    }

    private List<AccountRuleConfigPageVO> getAccountRuleMsgWithRedis(AccountDO accountDO, AccountFlowParam accountFlowParam) {

        AccountRuleConfigQueryParam accountRuleConfigQueryParam = new AccountRuleConfigQueryParam();

        /*//适用单据
        accountRuleConfigQueryParam.setOptDoc(accountFlowParam.getSourceDoc());
        //适用单据类型
        accountRuleConfigQueryParam.setOptDocType(accountFlowParam.getSourceDocType());
        //适用单据状态
        accountRuleConfigQueryParam.setOptDocStatus(accountFlowParam.getSourceDocStatus());
        //适用账户类型
        accountRuleConfigQueryParam.setOptAccountType(accountDO.getAccountType());
        //指定账户
        accountRuleConfigQueryParam.setSpecificAccount(accountDO.getAccountCode());*/
        accountRuleConfigQueryParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<AccountRuleConfigPageVO> accountRuleConfigPageVOList = myRedisService.getListDataFromRedis(FinRedisConstant.ACCOUNT_RULE_CONFIG, AccountRuleConfigPageVO.class, "账户规则配置", () -> {
            accountRuleConfigQueryParam.setSize(Integer.MAX_VALUE);
            PagingVO<AccountRuleConfigPageVO> accountRuleConfigPageVOPagingVO =
                    accountRuleConfigService.searchPageWithOutName(accountRuleConfigQueryParam);
            if (accountRuleConfigPageVOPagingVO == null || CollectionUtils.isEmpty(accountRuleConfigPageVOPagingVO.getRecords())) {
                return null;
            }
            return accountRuleConfigPageVOPagingVO.getRecords();
        });
        if (!CollectionUtils.isEmpty(accountRuleConfigPageVOList)) {
            accountRuleConfigPageVOList = accountRuleConfigPageVOList.stream().filter(
                    v -> {
                        if (StringUtils.isNotEmpty(accountFlowParam.getSourceDoc())) {
                            if (!accountFlowParam.getSourceDoc().equals(v.getOptDoc())) {
                                return false;
                            }
                        }
                        if (StringUtils.isNotEmpty(accountFlowParam.getSourceDocType())) {
                            if (!accountFlowParam.getSourceDocType().equals(v.getOptDocType())) {
                                return false;
                            }
                        }
                        if (StringUtils.isNotEmpty(accountFlowParam.getSourceDocStatus())) {
                            if (!accountFlowParam.getSourceDocStatus().equals(v.getOptDocStatus())) {
                                return false;
                            }
                        }
                        if (StringUtils.isNotEmpty(accountDO.getAccountType())) {
                            if (!accountDO.getAccountType().equals(v.getOptAccountType())) {
                                return false;
                            }
                        }
                        if (StringUtils.isNotEmpty(accountDO.getAccountCode())) {
                            if (v.getSpecificAccount() == null) {
                                return true;
                            }
                            if (!accountDO.getAccountCode().equals(v.getSpecificAccount())) {
                                return false;
                            }
                        }
                        return true;
                    }).collect(Collectors.toList());
        }
        Assert.notEmpty(accountRuleConfigPageVOList, "没有匹配到账户规则配置");
        return accountRuleConfigPageVOList;
    }

    private AccountDO getAccountMsg(AccountFlowParam accountFlowParam) {
        return accountRepoProc.findByAccountCode(accountFlowParam.getAccountCode());
    }

    private void checkGenerateAccountFlowParam(AccountFlowParam accountFlowParam) {
        Assert.notNull(accountFlowParam, "参数不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDoc(), "来源单据不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDocType(), "来源单据类型不能为空");
        Assert.notEmpty(accountFlowParam.getSourceDocStatus(), "来源单据状态不能为空");
//        Assert.notNull(accountFlowParam.getSourceDocAmount(), "来源单据金额不能为空");
        Assert.notNull(accountFlowParam.getSourceId(), "来源单据id不能为空");
        Assert.notEmpty(accountFlowParam.getSourceNo(), "来源单号不能为空");
        Assert.notEmpty(accountFlowParam.getAccountCode(), "账户编码不能为空");
    }


    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectListByQueryParam(AccountFlowQueryParam queryParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectListByQueryParam(queryParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectRepairAfterPage(AccountFlowPageParam pageParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectRepairAfterPage(pageParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> selectRepairAfter(AccountFlowPageParam pageParam) {
        List<AccountFlowVO> flowVOList = accountFlowRepoProc.selectRepairAfter(pageParam);
        if (CollectionUtils.isEmpty(flowVOList)) {
            return Collections.EMPTY_LIST;
        }
        return flowVOList;
    }


    /**
     * 账户流水分页查询：APP端
     *
     * @param pageParam 入参
     * @return 账户流水信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountFlowVO> appPage(AccountFlowPageParam pageParam) {

        PagingVO<AccountFlowVO> pagingVO = accountFlowRepoProc.appPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        //根据公司编码、加盟商编码、规则编码获取名称
        List<AccountFlowVO> accountFlowVOList = pagingVO.getRecords();

        return PagingVO.<AccountFlowVO>builder()
                .total(pagingVO.getTotal())
                .records(accountFlowVOList)
                .build();
    }

    @Override
    @SysCodeProc
    public List<AccountFlowVO> appList(AccountFlowPageParam pageParam) {
        return accountFlowRepoProc.appList(pageParam);
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagBatch(List<Long> ids) {
        //发生金额为0的增加删除功能
        List<AccountFlowDO> accountFlowDOList = accountFlowRepo.findAllById(ids);
        if (CollectionUtil.isNotEmpty(accountFlowDOList)) {
            List<AccountFlowDO> accountFlowFilterList = accountFlowDOList.stream().filter(accountFlowDO -> deleteCheckAmount(accountFlowDO.getAmount())).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(accountFlowFilterList)) {
                String checkResult = accountFlowFilterList.stream().map(vo ->
                        "流水号:" + vo.getFlowNo() + ""
                ).collect(Collectors.joining(";", "只有发生金额为0的可删除,[", "], 请检查"));
                throw new com.elitescloud.boot.exception.BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);
            }
        }
       /* accountFlowDOList.forEach(accountFlowDO -> {
            accountFlowDO.setDeleteFlag(1);
        });
        accountFlowRepo.saveAllAndFlush(accountFlowDOList);*/

        accountFlowRepoProc.updateDeleteFlagBatch(1, ids);
    }

    private Boolean deleteCheckAmount(BigDecimal amount) {
        if (Objects.isNull(amount) || amount.compareTo(BigDecimal.ZERO) == 0) {
            return false;
        }

        return true;
    }

}
