package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountConvert;
import com.elitesland.fin.application.facade.excel.creditaccount.CreditAccountImportEntity;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.rpc.sale.RmiSaleRpcService;
import com.elitesland.fin.rpc.system.SystemRpcService;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.sale.api.vo.param.crm.AllowShipCustGroupDParam;
import com.elitesland.sale.api.vo.resp.crm.AllowShipCustGroupDVO;
import com.elitesland.sale.api.vo.resp.crm.LmSaveCustRespVO;
import com.elitesland.sale.constant.AllowShipCustGroupEnum;
import com.elitesland.sale.constant.AllowShipStatus;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;


@Component
@Slf4j
@RequiredArgsConstructor
public class CreditAccountImportServiceImpl implements DataImport<CreditAccountImportEntity> {

    private final CreditAccountRepo creditAccountRepo;
    private final RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;
    private final RmiSaleRpcService rmiSaleRpcService;
    private final SystemRpcService systemRpcService;
    private final UdcProvider udcProvider;
    private final CreditAccountRepoProc creditAccountRepoProc;

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    @Override
    public String getTmplCode() {
        return "yst_fin_credit_account_import";
    }


    @Override
    public List<String> executeImport(List<CreditAccountImportEntity> dataList, int rowId) {

        if (CollectionUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }

        Map<String, Map<String, String>> udcCodeMap = udcProvider.getValueMapByUdcCode((UdcEnum.OBJECT_TYPE_CUST_GROUP.getModel()),
                Set.of(UdcEnum.OBJECT_TYPE_CUST_GROUP.getCode(), UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode()));
        Map<String, String> objectNameTypeMap = udcCodeMap.get(UdcEnum.OBJECT_TYPE_CUST_GROUP.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));
        Map<String, String> accountNameTypeMap = udcCodeMap.get(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getCode()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey, (v1, v2) -> v1));

        // 账户名称列表
        List<String> ouCodeList = dataList.stream().map(CreditAccountImportEntity::getOuCode).collect(Collectors.toList());

        // 查询公司信息
        OrgOuRpcDtoParam ouParam = new OrgOuRpcDtoParam();
        ouParam.setOuCodes(ouCodeList);
        List<OrgOuRpcDTO> ouRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(ouParam);
        Map<String, OrgOuRpcDTO> ouMap = ouRpcDTOList.stream().collect(Collectors.toMap(OrgOuRpcDTO::getOuCode, t -> t, (t1, t2) -> t1));

        // 查询客户信息
        AllowShipCustGroupDParam custParam = new AllowShipCustGroupDParam();
        List<String> custCodeList = dataList.stream().filter(item -> StrUtil.equals(item.getObjectTypeName(), UdcEnum.OBJECT_TYPE_CUST.getValueCodeName())).map(CreditAccountImportEntity::getObjectCode).collect(Collectors.toList());
        custParam.setCustCodes(custCodeList);
        List<AllowShipCustGroupDVO> custInfo = getAllowShipCustGroupInfo(custParam);
        Map<String, String> custGroupMap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(custInfo)) {
            custGroupMap = custInfo.stream().collect(Collectors.toMap(AllowShipCustGroupDVO::getCustCode, AllowShipCustGroupDVO::getCode));
        }
        List<LmSaveCustRespVO> invCust = rmiSaleRpcService.findInvCust(custCodeList);
        Map<String, String> custMap = Map.of();
        if (CollectionUtil.isNotEmpty(invCust)) {
            custMap = invCust.stream().collect(Collectors.toMap(LmSaveCustRespVO::getCustCode, LmSaveCustRespVO::getCustName, (t1, t2) -> t1));
        }


        // 客户组信息
        AllowShipCustGroupDParam groupDParam = new AllowShipCustGroupDParam();
        List<String> custGroupCodeList = dataList.stream().filter(item -> StrUtil.equals(item.getObjectTypeName(), UdcEnum.OBJECT_TYPE_CUST_GROUP.getValueCodeName())).map(CreditAccountImportEntity::getObjectCode).collect(Collectors.toList());
        groupDParam.setCodes(custGroupCodeList);
        List<AllowShipCustGroupDVO> custGroupInfo = getAllowShipCustGroupInfo(groupDParam);
        Map<String, List<String>> groupMap = new HashMap<>();
        Map<String, List<AllowShipCustGroupDVO>> map = new HashMap<>();
        if (CollectionUtil.isNotEmpty(custGroupInfo)) {
            map = custGroupInfo.stream().collect(Collectors.groupingBy(AllowShipCustGroupDVO::getCode));
            map.forEach((key, values) -> {
                List<String> custCodes = values.stream().map(item -> item.getCustCode()).collect(Collectors.toList());
                groupMap.put(key, custCodes);
            });
        }

        // 储存新导入的数据
        List<CreditAccountImportEntity> saveList = new ArrayList<>();
        // 错误集合
        List<String> errorResult = new ArrayList<>();
        // 存放已经遍历过的数据
        List<String> nameList = new ArrayList<>();
        List<String> uniList = new ArrayList<>();

        for (CreditAccountImportEntity data : dataList) {
            List<String> errorList = new ArrayList<>();

            // 必填校验
            checkImportMandatoryField(data, errorList, rowId);

            // 账户类型
            if (StrUtil.isNotBlank(data.getCreditAccountTypeName()) && accountNameTypeMap.containsKey(data.getCreditAccountTypeName())) {
                data.setCreditAccountType(accountNameTypeMap.get(data.getCreditAccountTypeName()));
            } else {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "信用账户类型", "信用账户类型，不正确"));
            }

            // 对象类型
            if (StrUtil.isNotBlank(data.getObjectTypeName()) && objectNameTypeMap.containsKey(data.getObjectTypeName())) {
                data.setObjectType(objectNameTypeMap.get(data.getObjectTypeName()));
            } else {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象类型", "对象类型，不正确"));
            }


            // 公司信息
            if (CollectionUtil.isNotEmpty(ouMap) && ouMap.containsKey(data.getOuCode())) {
                data.setOuId(ouMap.get(data.getOuCode()).getId());
                data.setOuName(ouMap.get(data.getOuCode()).getOuName());
            } else {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "授信组织(公司编码)", "没有查询到公司信息"));
            }

            // 账户名称
            if (StrUtil.isNotBlank(data.getCreditAccountName()) && creditAccountRepo.existsByCreditAccountName(data.getCreditAccountName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称已存在"));
            }
            if (nameList.contains(data.getCreditAccountName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称重复"));
            }
            nameList.add(data.getCreditAccountName());

            // 账户编码
            String code = systemRpcService.sysNumberRuleGenerateCode("yst-fin", "XYZH", new ArrayList<>());
            data.setCreditAccountCode(code);

            if (StrUtil.isNotBlank(data.getOuCode()) && StrUtil.isNotBlank(data.getObjectCode())) {
                if (creditAccountRepo.existsByObjectCodeAndOuCode(data.getObjectCode(), data.getOuCode())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "", "该对象编码和公司下已有账户存在"));
                }
            }

            String key = data.getOuCode() + data.getObjectTypeName() + data.getObjectCode();
            if (uniList.contains(key)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "", "公司/客户类型/客户编码 重复"));
            }
            uniList.add(key);


            // 账户类型属于客户
            if (StrUtil.equals(data.getObjectTypeName(), UdcEnum.OBJECT_TYPE_CUST.getValueCodeName()) && StrUtil.isNotBlank(data.getObjectCode())) {
                if (custGroupMap.containsKey(data.getObjectCode())) {
                    String groupCode = custGroupMap.get(data.getObjectCode());
                    if (creditAccountRepo.existsByObjectCodeAndOuCode(groupCode,data.getObjectCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象编码", "客户所属客户组已存在账户"));
                    }
                }
                if (custMap.containsKey(data.getObjectCode())) {
                    data.setObjectName(custMap.get(data.getObjectCode()));
                }
            }

            // 账户类型属于客户组
            if (StrUtil.equals(data.getObjectTypeName(), UdcEnum.OBJECT_TYPE_CUST_GROUP.getValueCodeName()) && StrUtil.isNotBlank(data.getObjectCode())) {
                // 2.校验客户是否有对应客户组已经被创建
                if (groupMap.containsKey(data.getObjectCode())) {
                    List<String> custCodes = groupMap.get(data.getObjectCode());
                    if (creditAccountRepo.existsByObjectCodeInAndOuCode(custCodes, data.getOuCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象编码", "该客户组已有客户创建账户，请调整"));
                    }
                    List<AllowShipCustGroupDVO> group = map.get(data.getObjectCode());
                    data.setObjectName(group.get(0).getName());
                } else {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象编码", "没有查询到客户组信息"));
                }
            }

            // 如果该条数据存在错误 添加错误信息
            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            } else {
                // 不存在错误信息
                errorResult.add(null);
                saveList.add(data);
            }
            // 行数自增
            rowId++;
        }

        if (CollectionUtils.isNotEmpty(saveList)) {
            saveData(saveList);
            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }

    }

    private void checkImportMandatoryField(CreditAccountImportEntity data, List<String> errorList, int rowId) {

        // 账户类型
        if (StrUtil.isBlank(data.getCreditAccountTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "信用账户类型", "信用账户类型，不能为空"));
        }

        // 对象类型
        if (StrUtil.isBlank(data.getObjectTypeName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象类型", "对象类型，不能为空"));
        }

        // 公司信息
        if (StrUtil.isBlank(data.getOuCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "授信组织(公司编码)", "授信组织(公司编码)，不能为空"));
        }

        // 账户名称
        if (StrUtil.isBlank(data.getCreditAccountName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称不能为空"));
        }

        // 账户编码
        if (StrUtil.isBlank(data.getObjectCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "对象编码", "对象编码不能为空"));
        }

    }

    public void saveData(List<CreditAccountImportEntity> saveList) {
        List<CreditAccountDO> doList = CreditAccountConvert.INSTANCE.importEntitys2Dos(saveList);
        creditAccountRepo.saveAll(doList);
    }

    private List<AllowShipCustGroupDVO> getAllowShipCustGroupInfo(AllowShipCustGroupDParam groupDParam) {
        groupDParam.setType(AllowShipCustGroupEnum.CREIDT_ACCOUNT_GROUP.getType());
        groupDParam.setStatus(AllowShipStatus.ACTIVE.getType());
        List<AllowShipCustGroupDVO> custGroupInfo = rmiSaleRpcService.getCustGroupInfo(groupDParam);
        return custGroupInfo;
    }
}
