package com.elitesland.fin.application.web.unionpay;


import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitesland.fin.application.service.unionpay.UnionPayService;
import com.elitesland.fin.application.service.unionpay.entity.req.SendPayReq;
import com.elitesland.fin.application.service.unionpay.util.UnionPayUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

/**
 * @author sunxw
 * @description 银联商务回调api
 * @Date 2023/5/18
 */
@Slf4j
@Api(tags = "tims银商支付接口")
@RestController
@RequestMapping(value = "/unionpay")
@AllArgsConstructor
public class UnionPayController {

    private final UnionPayService unionPayService;
    private final TaskExecutor taskExecutor;

    @GetMapping("/test/getCurrentTenant")
    @ApiModelProperty("银商接口下的当前租户")
    public ApiResult<?> getCurrentTenant(){
        SysTenantDTO tenant = TenantSession.getCurrentTenant();
        return ApiResult.ok(tenant);
    }

    @GetMapping("/test/taskExecutorTenant")
    @ApiModelProperty("线程中的当前租户")
    public ApiResult<?> taskExecutorTenant() {
        // 上下文中的租户
        SysTenantDTO tenant = TenantClient.getCurrentTenant();
        log.info("主线程租户：{}", tenant);
        CompletableFuture.runAsync(() -> {
            log.info("异步线程租户：{}", TenantClient.getCurrentTenant());
        }, taskExecutor);
        return ApiResult.ok(tenant);
    }

    @PostMapping("/sendPay")
    @ApiModelProperty("发起支付")
    public ApiResult<Map<String, Object>> sendPay(@RequestBody SendPayReq payReq){
        if (ObjectUtils.isEmpty(payReq.getRemitterAmt())){
            log.warn("账户储值金额有误:{}",payReq.getRemitterAmt());
            return ApiResult.fail("账户储值金额有误");
        }
        //发起支付请求时，设置单据状态为支付中
        Map<String, Object> respData = unionPayService.sendPay(payReq);
        return ApiResult.ok(respData);
    }

    /**
     * 银联支付接口异步回调
     * @return
     */
    @PostMapping("/payCallback")
    public ApiResult<Void> payCallback(HttpServletRequest request){
        Map<String, String[]> parameterMap = request.getParameterMap();
        log.info("4.1支付异步回调响应信息 is {}",parameterMap);
        if (CollectionUtils.isEmpty(parameterMap)){
            return ApiResult.fail("4.1支付异步回调响应信息为空");
        }
        Map<String, String> data = UnionPayUtil.valueArray2String(parameterMap);
        unionPayService.payCallBack(data);
        return ApiResult.ok();
    }

    /**
     * 银联支付接口异步回调
     * @return
     */
    @PostMapping("/followPayCallback")
    public ApiResult<Void> followPayCallback(HttpServletRequest request){
        Map<String, String[]> parameterMap = request.getParameterMap();
        log.info("4.15后续交易异步回调响应信息 is {}",parameterMap);
        if (CollectionUtils.isEmpty(parameterMap)){
            return ApiResult.fail("4.15后续交易异步回调响应信息为空");
        }
        Map<String, String> data = UnionPayUtil.valueArray2String(parameterMap);
        unionPayService.followPayCallback(data);
        return ApiResult.ok();
    }

}
