package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.FinEventTableConditionConvert;
import com.elitesland.fin.application.convert.accountingengine.FinEventTableConvert;
import com.elitesland.fin.application.convert.accountingengine.FinEventTableLineConvert;
import com.elitesland.fin.application.facade.dto.accountingengine.FinEventTableLineDTO;
import com.elitesland.fin.application.facade.param.accountingengine.FinEventTableConditionParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinEventTableLineParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinEventTableParam;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableConditionVO;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableLineVO;
import com.elitesland.fin.application.facade.vo.accountingengine.FinEventTableVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.accountingengine.FinEventTableDO;
import com.elitesland.fin.repo.accountingengine.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.sql.*;
import java.util.*;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinEventTableServiceImpl implements FinEventTableService {


    private final FinEventTableRepo finEventTableRepo;

    private final FinEventTableRepoProc finEventTableRepoProc;

    private final FinEventTableConditionRepo finEventTableConditionRepo;

    private final FinEventTableLineRepo finEventTableLineRepo;

    private final UdcProvider udcProvider;

    private final FinEventTableLineRepoProc finEventTableLineRepoProc;

    @SysCodeProc
    @Override
    public PagingVO<FinEventTableVO> page(FinEventTableParam finEventTableParam) {

        Map<String, String> map = udcProvider.getValueMapByUdcCode(UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getModel(), UdcEnum.EVENT_TABLE_CONDITION_EQUAL.getCode());
        PagingVO<FinEventTableVO> pagingVO = FinEventTableConvert.INSTANCE.DTOToVO(finEventTableRepoProc.page(finEventTableParam));
        List<FinEventTableVO> records = pagingVO.getRecords();
        if (CollectionUtils.isNotEmpty(records)) {
            records.stream().forEach(item -> {
                List<FinEventTableConditionVO> finEventTableConditionDetailList = item.getFinEventTableConditionDetailList();
                if (CollectionUtils.isNotEmpty(finEventTableConditionDetailList)) {
                    finEventTableConditionDetailList.stream().forEach(e -> e.setConditionTypeName(map.get(e.getConditionType())));
                }
                List<FinEventTableConditionVO> finEventTableProposedConditionList = item.getFinEventTableProposedConditionList();
                if (CollectionUtils.isNotEmpty(finEventTableProposedConditionList)) {
                    finEventTableProposedConditionList.stream().forEach(e -> e.setConditionTypeName(map.get(e.getConditionType())));
                }
            });
        }

        return pagingVO;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinEventTableParam finEventTableParam) {

        checkEnableOrDisableParam(finEventTableParam);

        List<FinEventTableDO> finEventTableDOList = finEventTableRepo.findAllById(finEventTableParam.getIds());
        finEventTableDOList.stream().forEach(item -> Assert.isFalse(finEventTableParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finEventTableDOList.stream().forEach(item -> item.setStatus(finEventTableParam.getStatus()));
    }

    @Override
    public List<FinEventTableLineVO> loadDataTemplate(FinEventTableParam finEventTableParam) {

        checkLoadDataTemplateParam(finEventTableParam);

        List<FinEventTableLineVO> finEventTableLineVOList = queryTableField(finEventTableParam, finEventTableParam.getMasTable());

        if (StringUtils.isNotEmpty(finEventTableParam.getTableName())) {
            finEventTableLineVOList.addAll(queryTableField(finEventTableParam, finEventTableParam.getTableName()));
        }

        return finEventTableLineVOList;

    }

    private List<FinEventTableLineVO> queryTableField(FinEventTableParam finEventTableParam, String tableName) {

        Connection connection = null;
        ResultSet columnResultSet = null;

        try {
            Class.forName(FinConstant.DRIVEN_NAME);

            connection = DriverManager.getConnection(String.format(FinConstant.CONNECTION_NAME,
                    finEventTableParam.getHost(),
                    finEventTableParam.getPort(),
                    finEventTableParam.getDatabaseName()),
                    finEventTableParam.getUserName(),
                    finEventTableParam.getPassword());

            DatabaseMetaData metaData = connection.getMetaData();

        /*    columnResultSet = metaData.getColumns(null,
                    finEventTableParam.getDatabaseName(),
                    tableName,
                    null);*/
            columnResultSet = metaData.getColumns(finEventTableParam.getDatabaseName(),
                    null,
                    tableName,
                    null);

            List<FinEventTableLineVO> finEventTableLineVOList = new ArrayList<>();
            while (columnResultSet.next()) {

                FinEventTableLineVO finEventTableLineVO = new FinEventTableLineVO();
                finEventTableLineVO.setTableName(tableName);
                finEventTableLineVO.setColumnName(columnResultSet.getString(FinConstant.COLUMN_NAME));
                finEventTableLineVO.setColumnType(columnResultSet.getString(FinConstant.TYPE_NAME));
                finEventTableLineVO.setColumnComment(columnResultSet.getString(FinConstant.REMARKS));

                finEventTableLineVOList.add(finEventTableLineVO);
            }

            return finEventTableLineVOList;

        } catch (Exception e) {
            throw new BusinessException("查询表结果异常", e);
        } finally {
            if (connection != null) {
                try {
                    connection.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }

            if (columnResultSet != null) {
                try {
                    columnResultSet.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
    }


    private void checkLoadDataTemplateParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getHost(), "主机必填");
        Assert.notEmpty(finEventTableParam.getPort(), "端口必填");
        Assert.notEmpty(finEventTableParam.getUserName(), "用户名必填");
        Assert.notEmpty(finEventTableParam.getPassword(), "密码必填");
        Assert.notEmpty(finEventTableParam.getDatabaseName(), "数据库名称必填");
        Assert.notEmpty(finEventTableParam.getMasTable(), "主表必填");
        Assert.notEmpty(finEventTableParam.getMasTableColumn(), "主表条件字段必填");
    }

    private void checkEnableOrDisableParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getIds(), "id必填");
        Assert.notEmpty(finEventTableParam.getStatus(), "状态必填");
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinEventTableParam finEventTableParam) {

        //防止npe
        finEventTableParam.setFinEventTableLineDetailList(Optional.ofNullable(finEventTableParam.getFinEventTableLineDetailList()).orElse(Collections.emptyList()));
        finEventTableParam.setFinEventTableConditionDetailList(Optional.ofNullable(finEventTableParam.getFinEventTableConditionDetailList()).orElse(Collections.emptyList()));
        finEventTableParam.setFinEventTableProposedConditionList(Optional.ofNullable(finEventTableParam.getFinEventTableProposedConditionList()).orElse(Collections.emptyList()));

        checkSaveOrUpdateParam(finEventTableParam);

        checkIdempotent(finEventTableParam);

        FinEventTableDO finEventTableDO = finEventTableRepo.save(FinEventTableConvert.INSTANCE.paramToDO(finEventTableParam));

        List<FinEventTableLineParam> finEventTableLineDetailList = finEventTableParam.getFinEventTableLineDetailList();
        List<FinEventTableConditionParam> finEventTableConditionDetailList = finEventTableParam.getFinEventTableConditionDetailList();
        List<FinEventTableConditionParam> finEventTableProposedConditionList = finEventTableParam.getFinEventTableProposedConditionList();

        if (CollectionUtils.isNotEmpty(finEventTableLineDetailList)) {
            finEventTableLineRepo.deleteAllByMasId(finEventTableDO.getId());
            finEventTableLineDetailList.stream().forEach(item -> item.setMasId(finEventTableDO.getId()));
            finEventTableLineRepo.saveAll(FinEventTableLineConvert.INSTANCE.paramToDO(finEventTableLineDetailList));
        }
        List<FinEventTableConditionParam> conditionParamsAll = new ArrayList<>();
        conditionParamsAll.addAll(finEventTableConditionDetailList);
        conditionParamsAll.addAll(finEventTableProposedConditionList);

        finEventTableConditionRepo.deleteAllByMasId(finEventTableDO.getId());
        if (CollectionUtils.isNotEmpty(conditionParamsAll)) {
            conditionParamsAll.stream().forEach(item -> item.setMasId(finEventTableDO.getId()));
            finEventTableConditionRepo.saveAll(FinEventTableConditionConvert.INSTANCE.paramToDO(conditionParamsAll));
        }
        return finEventTableDO.getId();
    }

    private void checkSaveOrUpdateParam(FinEventTableParam finEventTableParam) {
        Assert.notEmpty(finEventTableParam.getEventTable(), "事件表单名称必填");
        Assert.notEmpty(finEventTableParam.getHost(), "主机必填");
        Assert.notEmpty(finEventTableParam.getPort(), "端口必填");
        Assert.notEmpty(finEventTableParam.getUserName(), "用户名必填");
        Assert.notEmpty(finEventTableParam.getPassword(), "密码必填");
        Assert.notEmpty(finEventTableParam.getDatabaseName(), "数据库名称必填");
        Assert.notEmpty(finEventTableParam.getMasTable(), "主表必填");
        Assert.notEmpty(finEventTableParam.getMasTableColumn(), "主表条件字段必填");

        List<FinEventTableLineParam> finEventTableLineDetailList = finEventTableParam.getFinEventTableLineDetailList();
        List<FinEventTableConditionParam> finEventTableConditionDetailList = finEventTableParam.getFinEventTableConditionDetailList();
        List<FinEventTableConditionParam> finEventTableProposedConditionList = finEventTableParam.getFinEventTableProposedConditionList();

        Assert.isTrue(CollectionUtils.isNotEmpty(finEventTableLineDetailList) || CollectionUtils.isNotEmpty(finEventTableConditionDetailList), "明细不能为空");
        Set<String> fieldSet = new HashSet<>();
        finEventTableLineDetailList.stream().forEach(item -> {
            Assert.notBlank(item.getTableName(), "表单名称必填");
            Assert.notBlank(item.getColumnName(), "字段名称必填");
            Assert.notBlank(item.getColumnType(), "字段类型必填");
            Assert.notBlank(item.getColumnComment(), "字段含义必填");
            if(fieldSet.contains(item.getTableName().trim().concat(item.getColumnName().trim()))){
                throw new BusinessException("表单字段重复");
            }else{
                fieldSet.add(item.getTableName().trim().concat(item.getColumnName().trim()));
            }
        });

        finEventTableConditionDetailList.stream().forEach(item -> {
            Assert.notBlank(item.getTableName(), "表单名称必填");
            Assert.notBlank(item.getColumnName(), "字段名称必填");
            Assert.notBlank(item.getConditionType(), "条件必填");
            Assert.notBlank(item.getCategory(), "条件分类必填");
            Assert.isTrue(FinConstant.EVENT_TABLE_CONDITION_CATEGORY_DATA_SELECT.equals(item.getCategory()),"条件分类值不正确");
        });
        finEventTableProposedConditionList.stream().forEach(item -> {
            Assert.notBlank(item.getTableName(), "表单名称必填");
            Assert.notBlank(item.getColumnName(), "字段名称必填");
            Assert.notBlank(item.getConditionType(), "条件必填");
            Assert.notBlank(item.getCategory(), "条件分类必填");
            Assert.isTrue(FinConstant.EVENT_TABLE_CONDITION_CATEGORY_PROPOSED.equals(item.getCategory()),"条件分类值不正确");
        });
    }

    private void checkIdempotent(FinEventTableParam finEventTableParam) {

        FinEventTableDO finEventTableDO = finEventTableRepoProc.findByEventTable(finEventTableParam.getEventTable());

        if (finEventTableDO != null) {
            Assert.equals(finEventTableParam.getId(), finEventTableDO.getId(), "事件表单名称已经存在");
        }
    }
    @Override
    public PagingVO<FinEventTableLineVO> queryEventTableLinesById(FinEventTableLineParam finEventTableLineParam){
        PagingVO<FinEventTableLineDTO> pageLineDTO= finEventTableLineRepoProc.page(finEventTableLineParam);
        if(pageLineDTO.isEmpty()){
            return null;
        }
        List<FinEventTableLineVO> finEventTableLineVOS = FinEventTableConvert.INSTANCE.LineDTOToVO(pageLineDTO.getRecords());
        PagingVO<FinEventTableLineVO> pagingVO = new PagingVO<FinEventTableLineVO>();
        pagingVO.total( pageLineDTO.getTotal() );
        pagingVO.records(finEventTableLineVOS );
        return pagingVO;
    }
}
