package com.elitesland.fin.application.service.invoice.impl;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitesland.fin.application.facade.dto.mq.InvoiceSaveProcessMqMessageDTO;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplyParam;
import com.elitesland.fin.application.service.invoice.InvoiceAwaitService;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.invoice.InvoiceAwaitDO;
import com.elitesland.fin.repo.invoice.InvoiceAwaitRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.redisson.RedissonRedLock;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

@Component
@Slf4j
@RequiredArgsConstructor
public class InvoiceSaveProcessListener implements MessageQueueListener<InvoiceSaveProcessMqMessageDTO> {

    private final RedLockUtils redLockUtils;
    private final InvoiceAwaitService invoiceAwaitService;
    private final InvoiceAwaitRepo invoiceAwaitRepo;
    private final AbstractInvoiceSaveServiceImpl abstractInvoiceSaveService;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{InvoiceSaveProcessMqMessageDTO.SAVE_CHANNEL};
    }

    @Override
    public void onConsume(@NotBlank String s, @NotNull InvoiceSaveProcessMqMessageDTO messageDTO) {
        log.info("待开票数据处理提交：{}", JSON.toJSONString(messageDTO));
        RedissonRedLock redLock = null;
        try {
            // 分布式锁
            redLock = redLockUtils.getRedLock("invoice_save_process" + messageDTO.getBusinessKey());
            boolean lockFlag = redLock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("待开票数据已在处理中,获取锁失败");
                return;
            }
            // 幂等，判断发票状态
            Long id = messageDTO.getId();

            Optional<InvoiceAwaitDO> byId = invoiceAwaitRepo.findById(id);
            if (byId.isEmpty()) {
                log.error("未找到待开票数据");
                return;
            }
            InvoiceAwaitDO save = byId.get();
            if (!Objects.equals(save.getState(), UdcEnum.INVOICE_AWAIT_STATUS_WAIT.getValueCode())) {
                log.error("待开票数据不处于待开票状态，跳过处理");
                return;
            }

            InvoiceApplyParam applyParam = invoiceAwaitService.getInvoiceApplyParamByAwaitOrder(save);
            log.info("生成开票申请：{}", JSONUtil.toJsonStr(applyParam));
            abstractInvoiceSaveService.save(applyParam);
        } catch (Exception e) {
            log.error("开票时发生异常,待开票数据单号:{}", messageDTO.getBusinessKey(), e);
        } finally {
            if (redLock != null) {
                redLock.unlock();
                log.info("开票完成，释放锁成功");
            }
        }
    }
}
