package com.elitesland.fin.repo.invoice;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.facade.param.invoice.InvoiceRuleConfigQueryParam;
import com.elitesland.fin.entity.invoice.InvoiceRuleConfigDO;
import com.elitesland.fin.entity.invoice.QInvoiceRuleConfigDO;
import com.elitesland.fin.infr.dto.invoice.InvoiceRuleConfigDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * @author sunxw
 * @description
 * @Date 2023/7/11
 */
@Component
@RequiredArgsConstructor
public class InvoiceRuleConfigRepoProc {

    private static final QInvoiceRuleConfigDO qdo = QInvoiceRuleConfigDO.invoiceRuleConfigDO;
    private final JPAQueryFactory jpaQueryFactory;

    /**
     * 分页查询
     *
     * @param param 查询参数
     * @return
     */
    public PagingVO<InvoiceRuleConfigDTO> queryPaging(InvoiceRuleConfigQueryParam param) {
        List<Predicate> where = where(param);
        JPAQuery<InvoiceRuleConfigDTO> query = select(InvoiceRuleConfigDTO.class)
                .where(ExpressionUtils.allOf(where));
        param.setPaging(query);
        param.fillOrders(query, qdo);
        return PagingVO.<InvoiceRuleConfigDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    public List<InvoiceRuleConfigDTO> getList(List<InvoiceRuleConfigQueryParam> param) {

        List<Predicate> predicateList = new ArrayList<>();
        for (InvoiceRuleConfigQueryParam invoiceRuleConfigQueryParam : param) {
            Predicate predicate = whereConfig(invoiceRuleConfigQueryParam);
            predicateList.add(predicate);
        }
        JPAQuery<InvoiceRuleConfigDTO> jpaQuery = select(InvoiceRuleConfigDTO.class)
                .where(ExpressionUtils.anyOf(predicateList)).orderBy(qdo.createTime.desc());
        return jpaQuery.fetch();
    }

    /**
     * 获取启用状态的开票设置列表
     * @return
     */
    public List<InvoiceRuleConfigDTO> getListByParam(InvoiceRuleConfigQueryParam param) {
        Predicate predicate = whereConfig(param);
        JPAQuery<InvoiceRuleConfigDTO> jpaQuery = select(InvoiceRuleConfigDTO.class)
                .where(predicate).orderBy(qdo.createTime.desc());
        return jpaQuery.fetch();
    }

    private Predicate whereConfig(InvoiceRuleConfigQueryParam param) {
        return BaseRepoProc.PredicateBuilder.builder()
                .andEq(StrUtil.isNotBlank(param.getStatus()), qdo.status, param.getStatus())
                .andEq(StrUtil.isNotBlank(param.getOptDocCls()), qdo.optDocCls, param.getOptDocCls())
                .andEq(StrUtil.isNotBlank(param.getOptDocType()), qdo.optDocType, param.getOptDocType())
                .andIn(CollUtil.isNotEmpty(param.getOptDocTypes()), qdo.optDocType, param.getOptDocTypes())
                .andIn(CollUtil.isNotEmpty(param.getOptDocClss()), qdo.optDocCls, param.getOptDocClss())
                .andEq(StrUtil.isNotBlank(param.getOptDocStatus()),qdo.optDocStatus,param.getOptDocStatus())
                .andEq(StrUtil.isNotBlank(param.getOuCode()),qdo.ouCode,param.getOuCode())
                .andLike(StrUtil.isNotBlank(param.getOuName()),qdo.ouName,param.getOuName())
                .build();
    }

    /**
     * 拼装查询条件
     *
     * @param param 查询参数
     * @return 条件
     */
    private List<Predicate> where(InvoiceRuleConfigQueryParam param) {
        List<Predicate> predicates = new ArrayList<>();
        if (StrUtil.isNotBlank(param.getStatus())) {
            predicates.add(qdo.status.eq(param.getStatus()));
        }
        if (StrUtil.isNotBlank(param.getOptDocCls())) {
            predicates.add(qdo.optDocCls.eq(param.getOptDocCls()));
        }
        if (StrUtil.isNotBlank(param.getOptDocType())) {
            predicates.add(qdo.optDocType.eq(param.getOptDocType()));
        }
        if (StrUtil.isNotBlank(param.getOptDocStatus())) {
            predicates.add(qdo.optDocStatus.eq(param.getOptDocStatus()));
        }
        if (StrUtil.isNotBlank(param.getOuName())) {
            predicates.add(qdo.ouName.like("%" + param.getOuName() + "%"));
        }
        if (StrUtil.isNotBlank(param.getOuCode())) {
            predicates.add(qdo.ouCode.eq(param.getOuCode()));
        }
        return predicates;
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qdo.id,
                qdo.optDocCls,
                qdo.optDocType,
                qdo.optDocStatus,
                qdo.invoiceLimit,
                qdo.autoInvoice,
                qdo.autoReview,
                qdo.tenantId,
                qdo.remark,
                qdo.creator,
                qdo.createUserId,
                qdo.modifyUserId,
                qdo.createTime,
                qdo.modifyTime,
                qdo.deleteFlag,
                qdo.ouCode,
                qdo.ouName,
                qdo.needRedraft,
                qdo.editInvTitle
        )).from(qdo);
    }

    public void update(InvoiceRuleConfigDO invoiceRuleConfigDO) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        update.where(qdo.id.eq(invoiceRuleConfigDO.getId()));
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getStatus())) {
            update.set(qdo.status, invoiceRuleConfigDO.getStatus());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getOptDocCls())) {
            update.set(qdo.optDocCls, invoiceRuleConfigDO.getOptDocCls());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getOptDocType())) {
            update.set(qdo.optDocType, invoiceRuleConfigDO.getOptDocType());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getOptDocStatus())) {
            update.set(qdo.optDocStatus, invoiceRuleConfigDO.getOptDocStatus());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getInvoiceLimit())) {
            update.set(qdo.invoiceLimit, invoiceRuleConfigDO.getInvoiceLimit());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getAutoReview())) {
            update.set(qdo.autoReview, invoiceRuleConfigDO.getAutoReview());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getAutoInvoice())) {
            update.set(qdo.autoInvoice, invoiceRuleConfigDO.getAutoInvoice());
        }
        if (!ObjectUtils.isEmpty(invoiceRuleConfigDO.getEditInvTitle())) {
            update.set(qdo.editInvTitle, invoiceRuleConfigDO.getEditInvTitle());
        }
        update.execute();
    }
}
