package com.elitesland.fin.infinity.aisino.service;

import cn.hutool.core.lang.Assert;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplyParam;
import com.elitesland.fin.application.facade.vo.invoice.InvoiceSaveVO;
import com.elitesland.fin.application.service.invoice.InvoiceApplyService;
import com.elitesland.fin.application.service.invoice.InvoiceAwaitService;
import com.elitesland.fin.domain.entity.invoiceredraft.InvoiceRedraftDO;
import com.elitesland.fin.domain.entity.invoiceredraft.QInvoiceRedraftDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDO;
import com.elitesland.fin.dto.invoice.InvoiceApplyRpcDTO;
import com.elitesland.fin.infinity.aisino.entity.AisinoLogDO;
import com.elitesland.fin.infinity.aisino.enums.RedStateEnum;
import com.elitesland.fin.infinity.aisino.repo.AisinoLogRepo;
import com.elitesland.fin.infinity.aisino.vo.param.AisinoCoverParamPayload;
import com.elitesland.fin.infinity.aisino.vo.param.invoice.InvoiceMain;
import com.elitesland.fin.infinity.aisino.vo.param.notice.InvoiceParam;
import com.elitesland.fin.infinity.aisino.vo.resp.*;
import com.elitesland.fin.infr.dto.saleinv.SaleInvDtlDTO;
import com.elitesland.fin.infr.repo.invoiceredraft.InvoiceRedraftRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvDtlRepoProc;
import com.elitesland.fin.infr.repo.saleinv.SaleInvRepo;
import com.elitesland.fin.param.saleinv.InvoiceSaveParam;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import com.saobei.open.sdk.util.HttpClientUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.elitesland.fin.infinity.aisino.domain.AisinoRouterCode.*;
import static java.time.LocalDateTime.parse;

/**
 * @author eric.hao
 * @since 2023/05/16
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AisinoServiceImpl implements AisinoService {

    private final AisinoLogRepo aisinoLogRepo;
    private final SaleInvRepo saleInvRepo;
    private final SaleInvDtlRepoProc saleInvDtlRepoProc;
    private final JPAQueryFactory jpaQueryFactory;
    private final QInvoiceRedraftDO qdo = QInvoiceRedraftDO.invoiceRedraftDO;
    private final TaskExecutor taskExecutor;
    @Autowired
    @Lazy
    private InvoiceAwaitService invoiceAwaitService;
    @Autowired
    private InvoiceRedraftRepo invoiceRedraftRepo;
    private final InvoiceApplyService invoiceApplyService;

    private final AisinoPayloadService aisinoPayloadService;

    @Value("${aisino.api.url:http://221.234.42.184:9236}")
    private String baseUrl;

    @Override
    public void addLog(String type, String name, String param, String param2) {
        AisinoLogDO xforceLog = new AisinoLogDO();
        xforceLog.setRequestType(type);
        xforceLog.setRequestTypeName(name);
        xforceLog.setRequestParam(param);
        xforceLog.setRequestParam2(param2);
        aisinoLogRepo.save(xforceLog);
    }

    private AisinoCoverRespVO getAisinoApplyRespVO(String url, String jsonString) {
        String responseStr;
        try {
            responseStr = HttpClientUtil.post(url, jsonString, 10000, "application/json", "utf-8");
            if (Strings.isBlank(responseStr)) {
                throw new BusinessException("航信接口响应为空");
            }
            log.info("航信接口调用返回：{}", responseStr);
        } catch (Exception e) {
            log.error("航信接口调用异常: {}", e.getMessage());
            throw new BusinessException("航信接口调用异常:" + e.getMessage());
        }
        return JSON.parseObject(responseStr, AisinoCoverRespVO.class);
    }

    @Override
    public BlueInvoiceApplyRespVO blueInvoiceApply(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Kp_Fpkj";
        String finalParam = JSON.toJSONString(payload);
        log.info("蓝票申请地址：{},入参: {}", url, finalParam);
        addLog(AISINO_BLUE_INVOICE_APPLY, "航信开票上传接口：参数", finalParam, originParam);
        // 发送请求
        String responseVoStr = null;
        BlueInvoiceApplyRespVO decrypt = null;
        try {
            var responseVO = getAisinoApplyRespVO(url, finalParam);
            responseVoStr = JSON.toJSONString(responseVO);
            if (Objects.equals(responseVO.getCODE(), "200")) {
                String data = responseVO.getDATA();
                decrypt = aisinoPayloadService.decrypt(data, BlueInvoiceApplyRespVO.class);
            } else if (Objects.equals(responseVO.getCODE(), "500")) {
                throw new BusinessException(responseVO.getMSG());
            }
        } catch (Exception e) {
            addLog(AISINO_BLUE_INVOICE_APPLY, "航信开票上传接口：结果", e.getMessage(), null);
            throw new BusinessException(e.getMessage());
        }
        addLog(AISINO_BLUE_INVOICE_APPLY, "航信开票上传接口：结果", responseVoStr, decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public AisinoResultResp blueInvoiceResult(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Yp_FpUrl";
        String finalParam = JSON.toJSONString(payload);
        log.info("蓝票获取入参: {}", finalParam);
        addLog(AISINO_BLUE_RESULT, "获取蓝票返回：参数", finalParam, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, finalParam);
        String responseVoStr = JSON.toJSONString(responseVO);
        AisinoResultResp decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, AisinoResultResp.class);
        }
        addLog(AISINO_BLUE_RESULT, "获取蓝票返回：结果", responseVoStr, decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }
    @Override
    public AisinoResultResp blueInvoiceResultBase64(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Yp_Down";
        String finalParam = JSON.toJSONString(payload);
        log.info("蓝票Base64获取入参: {}", finalParam);
        addLog(AISINO_BLUE_BASE64_RESULT, "获取蓝票Base64返回：参数", finalParam, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, finalParam);
        String responseVoStr = JSON.toJSONString(responseVO);
        AisinoResultResp decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, AisinoResultResp.class);
        }
        addLog(AISINO_BLUE_BASE64_RESULT, "获取蓝票Base64返回：结果", responseVoStr, decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public AisinoRedInitialResultResp redInvoiceInitial(AisinoCoverParamPayload payload, String originParam) {
        String jsonString = JSON.toJSONString(payload);
        String url = baseUrl + "/openapi/qd/qpt/Kp_RedBillInit";
        addLog(AISINO_RED_INITIAL, "红票初始化结果反馈接口：参数", jsonString, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, jsonString);
        String responseVoStr = JSON.toJSONString(responseVO);
        AisinoRedInitialResultResp decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, AisinoRedInitialResultResp.class);
        }
        addLog(AISINO_RED_INITIAL, "红票初始化结果反馈接口：结果", responseVoStr,  decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public AisinoRedSaveResultResp redInvoiceSave(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Kp_RedBillSave";
        String jsonString = JSON.toJSONString(payload);
        addLog(AISINO_RED_SAVE, "红票保存结果反馈接口：参数", jsonString, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, jsonString);
        String responseVoStr = JSON.toJSONString(responseVO);
        AisinoRedSaveResultResp decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, AisinoRedSaveResultResp.class);
        }
        addLog(AISINO_RED_SAVE, "红票保存结果反馈接口：结果", responseVoStr,  decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public AisinoRedOpenResultResp redInvoiceOpen(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Kp_RedIssue";
        String jsonString = JSON.toJSONString(payload);
        addLog(AISINO_RED_OPEN, "红票开票结果反馈接口：参数", jsonString, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, jsonString);
        String responseVoStr = JSON.toJSONString(responseVO);
        AisinoRedOpenResultResp decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, AisinoRedOpenResultResp.class);
        }
        addLog(AISINO_RED_OPEN, "红票开票结果反馈接口：结果", responseVoStr,  decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public InvoiceResultRespVO getInvoiceInfoByFlowNo(AisinoCoverParamPayload payload, String originParam) {
        String url = baseUrl + "/openapi/qd/qpt/Kp_GetFpByLsh";
        String jsonString = JSON.toJSONString(payload);
        addLog(AISINO_INVOICE_INFO, "根据流水号查询发票信息：参数", jsonString, originParam);
        // 发送请求
        var responseVO = getAisinoApplyRespVO(url, jsonString);
        String responseVoStr = JSON.toJSONString(responseVO);

        InvoiceResultRespVO decrypt = null;
        if (Objects.equals(responseVO.getCODE(), "200")) {
            String data = responseVO.getDATA();
            decrypt = aisinoPayloadService.decrypt(data, InvoiceResultRespVO.class);
        }
        addLog(AISINO_INVOICE_INFO, "根据流水号查询发票信息：结果", responseVoStr, decrypt == null ? null : JSON.toJSONString(decrypt));
        return decrypt;
    }

    @Override
    public void saveInvoice(InvoiceParam invoiceParam) {
        List<InvoiceSaveParam> saveParams = new ArrayList<>();
        InvoiceMain invoiceMain = invoiceParam.getInvoiceMain();
        // 开票申请单号
        InvoiceApplyRpcDTO applyRpcDTO = invoiceApplyService.queryIdByApplyNo(invoiceMain.getSalesbillNo());
        Assert.notNull(applyRpcDTO.getApplyId(), invoiceMain.getSalesbillNo() + " 单据不存在");
        InvoiceSaveParam saveParam = new InvoiceSaveParam();
        // 开票申请表头id
        saveParam.setMasId(applyRpcDTO.getApplyId());
        // 发票号码
        saveParam.setInvNo(invoiceMain.getInvoiceNo());
        // 发票代码
        saveParam.setInvCode(invoiceMain.getInvoiceCode());
        // 流水号
        saveParam.setFlowNo(invoiceMain.getBatchNo());
        // 校验码
        saveParam.setCheckCode(invoiceMain.getCheckCode());
        // 原蓝票代码
        saveParam.setBlueInvCode(invoiceMain.getOriginInvoiceCode());
        // 原蓝票号码
        saveParam.setBlueInvNo(invoiceMain.getOriginInvoiceNo());
        // 开票金额
        saveParam.setTotalAmt(new BigDecimal(invoiceMain.getAmountWithTax()));
        // 开票日期
        saveParam.setInvDate(parse(invoiceMain.getPaperDrewDate()));
        // 红冲状态
        saveParam.setRedState(invoiceMain.getRedStatus());
        // 开票状态
        saveParam.setInvState(invoiceMain.getStatus());
        // pdf链接
        saveParam.setInvPdfUrl(invoiceMain.getPdfPath());
        // 开票失败原因
        saveParam.setInvFailCause("");
        saveParams.add(saveParam);
        invoiceApplyService.saveInvoice(saveParams);

        // 异步处理后续操作
        CompletableFuture.runAsync(() -> handleOutcomeOfInvoice(invoiceMain, applyRpcDTO), taskExecutor);
    }

    /**
     * 根据票易通的开票状态和红冲状态，处理后续流程，开票申请单和修改订货单状态
     *
     * @param invoiceMain 票易通开票主数据
     * @param applyRpcDTO 原开票申请单
     */
    private void handleOutcomeOfInvoice(InvoiceMain invoiceMain, InvoiceApplyRpcDTO applyRpcDTO) {
        log.info("开始确认关联单据状态,开票状态:{},红冲状态:{},开票申请单入参:{}", invoiceMain.getStatus(), invoiceMain.getRedStatus(), JSON.toJSONString(applyRpcDTO));
        List<String> sourceDocNoList = applyRpcDTO.getPaymentNoList();
        //订单红冲
        if (RedStateEnum.RED_PUNCH.getCode().equals(invoiceMain.getRedStatus())) {
            //重新生成付款记录并初始化订货单开票状态
            SaleInvDO saleInvDO = saleInvRepo.findByApplyNo(invoiceMain.getSalesbillNo());
            if (!Objects.isNull(saleInvDO)) {
                log.error("未找到开票申请单，单号为：{}", invoiceMain.getSalesbillNo());
            }
            // 查询开票申请单明细
            List<SaleInvDtlDTO> saleInvDtlDTOList = saleInvDtlRepoProc.getList(saleInvDO.getId());

            List<Long> salInvDtlIdList = new ArrayList<>();
            if (!CollectionUtils.isEmpty(saleInvDtlDTOList)) {
                salInvDtlIdList = saleInvDtlDTOList.stream().map(SaleInvDtlDTO::getSourceLineId).collect(Collectors.toList());
            }

            //获取流程的审批状态
            InvoiceRedraftDO invoiceRedraftDO = invoiceRedraftRepo.findByOrigApplyNo(invoiceMain.getSalesbillNo());
            //红冲完成且审批通过生成开票申请单
            if (!ObjectUtils.isEmpty(invoiceRedraftDO) && ProcInstStatus.APPROVED.equals(invoiceRedraftDO.getWorkflowProcInstStatus()) && ObjectUtils.isEmpty(invoiceRedraftDO.getOrigApplyNo())) {
                log.info("红冲后开始创建新的开票申请单");
                createInvoice(applyRpcDTO, saleInvDO, salInvDtlIdList, invoiceRedraftDO);
            }
        }
//        else if (RedStateEnum.PRE_RED_PUNCH.getCode().equals(invoiceMain.getRedStatus())) {
//            //订货单状态更新为开票成功
//            paymentRecordsService.updateInvoiceRelatedOrderStatus(sourceDocNoList, "30");
//        } else if (RedStateEnum.DEFAULT.getCode().equals(invoiceMain.getRedStatus())) {
//            //订货单状态更新为开票失败
//            paymentRecordsService.updateInvoiceRelatedOrderStatus(sourceDocNoList, "50");
//        }
    }

    /**
     * 开始重新开票
     *
     * @param applyRpcDTO      原开票申请单数据
     * @param saleInvDO        原开票数据
     * @param invoiceRedraftDO 开票审批流程数据
     */
    private void createInvoice(InvoiceApplyRpcDTO applyRpcDTO, SaleInvDO saleInvDO, List<Long> paymentRecordDetailIdList, InvoiceRedraftDO invoiceRedraftDO) {
        InvoiceApplyParam param = new InvoiceApplyParam();
        param.setOptDocNos(applyRpcDTO.getPaymentNoList());
        param.setIds(applyRpcDTO.getPaymentIds());
        if (!CollectionUtils.isEmpty(paymentRecordDetailIdList)) {
            param.setBizDtlKey(paymentRecordDetailIdList);
        }
//        if (!Objects.isNull(invoiceRedraftDO.getSaleTaxNo())) {
//            param.setCustTaxNo(invoiceRedraftDO.getSaleTaxNo());
//            param.setCustInvTitle(invoiceRedraftDO.getSaleInvTitle());
//            param.setAddress(invoiceRedraftDO.getSaleAdd());
//            param.setCustBank(invoiceRedraftDO.getSaleBank());
//            param.setCustBankAcc(invoiceRedraftDO.getSaleBankAcc());
//            param.setCustTel(invoiceRedraftDO.getSaleTel());
//        } else {
        param.setCustTaxNo(saleInvDO.getCustTaxNo());
        param.setCustInvTitle(saleInvDO.getCustInvTitle());
        param.setAddress(saleInvDO.getCustAdd());
        param.setCustBank(saleInvDO.getCustBank());
        param.setCustBankAcc(saleInvDO.getCustBankAcc());
        param.setCustTel(saleInvDO.getCustTel());
//        }
        param.setInvType(invoiceRedraftDO.getInvoiceType());
        log.info("开始生成开票申请单，入参为：{}", JSON.toJSONString(param));
        List<InvoiceSaveVO> saveList = invoiceAwaitService.invoiceApply(param);
        log.info("保存开票申请单返回结果为：{}", JSON.toJSONString(saveList));
        // 回填新的开票号
        if (!CollectionUtils.isEmpty(saveList)) {
            for (InvoiceSaveVO save : saveList) {
                if (!Objects.isNull(save.getApplyNo()))
                    log.info("开始执行更新新申请单号更新语句，数据为{},id:{}", save.getApplyNo(), invoiceRedraftDO.getId());
                JPAUpdateClause update = jpaQueryFactory.update(qdo)
                        .set(qdo.redraftApplyNo, save.getApplyNo())
                        .where(qdo.id.eq(invoiceRedraftDO.getId()));
                log.info("打印执行语句");
                update.execute();
            }
        }
    }

//    @Override
//    public OrderQueryRespVO queryBizOrder(String applyNo) {
//        // 构建接口相关参数
//        InfinityRouterParamVO routerParam = XforceRouterVO.of(XF_BIZ_ORDER_QUERY);
//        // 构建接口入参
//        Map<String, Object> requestBody = new HashMap<>();
//        requestBody.put("bizOrderNo", applyNo);
//        requestBody.put("serialNo", applyNo);
//        requestBody.put("accountType", "AR");
//        requestBody.put("includeDetails", true);
//        addLog(XF_BIZ_ORDER_QUERY, "业务单单张查询接口：参数", BeanUtils.toJsonStr(requestBody));
//        XforceRespVO responseVO = routerClientService.send(routerParam, requestBody, XforceRespVO.class);
//        addLog(XF_BIZ_ORDER_QUERY, "业务单单张查询接口：结果", BeanUtils.toJsonStr(responseVO));
//        if (ResponseVO.failed(responseVO)) {
//            throw new BusinessException("查询失败：[票易通] " + responseVO.getMessage());
//        }
//        return BeanUtils.toBean(responseVO.getResult(), OrderQueryRespVO.class);
//    }

//    @SneakyThrows
//    @Override
//    public InvoiceParam queryInvoice(String applyNo) {
//        // 构建接口相关参数
//        InfinityRouterParamVO routerParam = XforceRouterVO.of(XF_BIZ_INVOICE_QUERY);
//        // 构建接口入参
//        String requestStr = "{\"conditions\": [{\"conditionOP\": \"eq\",\"field\": \"bizOrderNo\",\"value\": \"" + applyNo + "\",\"values\": []}],\"page\": {\"pageNo\": 1,\"pageSize\": 10},\"sorts\": [{\"field\": \"invoiceNo\",\"value\": \"asc\"}]}";
//        Map<String, Object> requestBody = BeanUtils.toMap(requestStr);
//        requestBody.put("serialNo", applyNo);
//        addLog(XF_BIZ_INVOICE_QUERY, "销项发票查询：参数", requestStr);
//        XforceRespVO responseVO = routerClientService.send(routerParam, requestBody, XforceRespVO.class);
//        addLog(XF_BIZ_INVOICE_QUERY, "销项发票查询：结果", BeanUtils.toJsonStr(responseVO));
//        if (ResponseVO.failed(responseVO)) {
//            throw new BusinessException("查询失败：[票易通] " + responseVO.getMessage());
//        }
//
//        InvoiceQueryRespVO invoiceResp = BeanUtils.toBean(responseVO.getResult(), InvoiceQueryRespVO.class);
//        List<InvoiceQueryData> result = BeanUtils.toList(invoiceResp.getData(), InvoiceQueryData.class);
//        if (CollUtil.isEmpty(result)) {
//            throw new BusinessException("查询失败：[票易通] 未查到任何可用单据");
//        }
//        if (CollUtil.isNotEmpty(result)) {
//            InvoiceParam invoiceParam = new InvoiceParam();
//            InvoiceMain invoiceMain = XforceConvert.INSTANCE.queryToMain(result.get(0));
//            invoiceMain.setSalesbillNo(applyNo);
//            invoiceParam.setInvoiceMain(invoiceMain);
//            return invoiceParam;
//        }
//        return null;
//    }

//    private LocalDateTime parse(String date) {
//        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd HH:mm:ss");
//        return LocalDateTime.parse(date + " 00:00:00", formatter);
//    }
}
