package com.elitesland.fin.infinity.utils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2021/07/12
 */
public abstract class BeanConvertUtil {

    private static final ObjectMapper OBJECT_MAPPER;

    static {
        OBJECT_MAPPER = new ObjectMapper();
        OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * 转为实体工具方法
     */
    public static <T> T toEntity(String response, Class<T> cls) throws IOException {
        Objects.requireNonNull(response);
        Objects.requireNonNull(cls);
        return OBJECT_MAPPER.readValue(response, cls);
    }

    /**
     * 结果集转换
     */
    public static <T> List<T> toEntities(List<Map<String, Object>> response, Class<T> cls) throws IOException {
        return response.stream().map(row -> {
            try {
                return BeanConvertUtil.toEntity(toJsonStr(row), cls);
            } catch (IOException e) {
                e.printStackTrace();
            }
            return null;
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public static <T> String toJsonStr(T payload) {
        Objects.requireNonNull(payload);
        try {
            return OBJECT_MAPPER.writeValueAsString(payload);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return "[]";
    }

    /**
     * HashMap toString
     */
    public static String hashMapToString(Map<String, Object> param) {
        return param.entrySet().stream()
                .map(entry -> StringFormatUtil.bracket(entry.getKey()) + " : " + hashValueToString(entry.getValue()))
                .collect(Collectors.joining(", ", "{", "}"));
    }

    private static String hashValueToString(Object value) {
        if (value instanceof Object[]) {
            List<String> result = Arrays.asList(resolveStr(value));
            return result.toString();
        }
        if (value instanceof List) {
            return (String) ((ArrayList) value).stream().map(BeanConvertUtil::hashValueToString)
                    .collect(Collectors.joining(", ", "[", "]"));
        }
        if (value instanceof Map) {
            return hashMapToString((Map<String, Object>) value);
        }
        return StringFormatUtil.bracket(value);
    }

    public static String[] resolveStr(Object paramStr) {
        if (isEmpty(paramStr)) {
            return new String[]{};
        }
        return paramStr instanceof String ? new String[]{(String) paramStr} : resolveArr((Object[]) paramStr);
    }

    private static String[] resolveArr(Object[] paramArr) {
        return Arrays.copyOf(paramArr, paramArr.length, String[].class);
    }

    public static boolean isEmpty(Object str) {
        if (str == null) {
            return true;
        } else if (str.getClass().isArray()) {
            if (Array.getLength(str) == 0) {
                return true;
            } else if (Array.getLength(str) == 1) {
                Object obj = Array.get(str, 0);
                return obj instanceof String && ((String) obj).length() == 0;
            }
        } else {
            return str instanceof String && ((String) str).length() == 0;
        }
        return false;
    }

    public static HashMap<String, Object> entityToMap(Object obj) throws Exception {
        HashMap<String, Object> map = new HashMap<String, Object>();
        Class<?> clazz = obj.getClass();
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            PropertyDescriptor pd = new PropertyDescriptor(field.getName(), clazz);
            Method getMethod = pd.getReadMethod();
            Object o = getMethod.invoke(obj);
            if (null != o) {
                map.put(field.getName(), o);
            }
        }
        return map;
    }
}

