package com.elitesland.fin.infinity.aisino.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplySaveParam;
import com.elitesland.fin.application.facade.param.invoice.InvoiceRedApplySaveParam;
import com.elitesland.fin.domain.entity.saleinv.SaleInvdDtlDO;
import com.elitesland.fin.entity.invoice.TaxAccountMappingDO;
import com.elitesland.fin.infinity.aisino.utils.InvoiceIssueUtil;
import com.elitesland.fin.infinity.aisino.vo.param.*;
import com.elitesland.fin.infinity.aisino.vo.resp.InvoiceRedInitialRespVO;
import com.elitesland.fin.infr.repo.saleinv.TaxAccountMappingDORepo;
import com.elitesland.fin.param.saleinv.InvoiceDetailSaveParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 航信入参组装服务
 * Created by lance on 2025/9/11.
 */
@Slf4j
@Service
public class AisinoPayloadService {

    @Autowired
    private UdcProvider sysUdcService;

    @Autowired
    private TaxAccountMappingDORepo taxAccountMappingDORepo;

    // 加密秘钥
    @Value("${aisino.encrypt.key:Y#Y*B!PC}")
    private String encryptKey;

    @Value("${aisino.api.appid:YYBPC}")
    private String appId;

    public AisinoCoverParamPayload assembleAisinoPayload(String param, String taxNo) {
        log.info("assembleAisinoPayload: {}", param);
        TaxAccountMappingDO byTaxNo = taxAccountMappingDORepo.findByTaxNo(taxNo);
        if (byTaxNo == null) {
            throw new BusinessException("税号" + taxNo + "未配置开票账户");
        }
        String accountNo = byTaxNo.getAccountNo();
        return AisinoCoverParamPayload.builder()
                .APPID(appId)
                .USERNAME(accountNo)
                .NSRSBH(taxNo)
                .DATA(encrypt(param))
                .build();
    }

    /**
     * 蓝票申请参数初始化
     *
     * @param saveParam
     * @return
     */
    public AisinoBlueInvoiceApplyParam blueInvoiceApplyInitial(InvoiceApplySaveParam saveParam, SaleInvdDtlDO saleInvdDtlDO) {
        AisinoBlueInvoiceApplyParam row = new AisinoBlueInvoiceApplyParam();
        if (saleInvdDtlDO != null) {
            row.setFPQQLSH(saleInvdDtlDO.getFlowNo());
        } else {
            // 发票请求流水号 = 税号 + 最长30位随机数 唯一值）一个流水号对应一张发票不可重复（防止重复开票），同一流水号最多请求5次(开票失败建议请勿频繁请求）
            // 根据时间获取五位随机数
            String timestamp = String.valueOf(System.currentTimeMillis());
            String randomStr = timestamp.substring(timestamp.length() - 5);
            // 获取随机五位数
            String randomFiveDigits = String.format("%05d", (int) (Math.random() * 100000));
            row.setFPQQLSH(saveParam.getSaleTaxNo() + randomStr + randomFiveDigits);
        }
        AisinoBlueInvoiceApplyRequest request = new AisinoBlueInvoiceApplyRequest();

        // 开票申请单号
        request.setDJH(saveParam.getApplyNo());

        // 购方抬头
        request.setGMFMC(saveParam.getCustInvTitle());

        // 纳税人识别号
        request.setGMFSH(saveParam.getCustTaxNo() == null ? "" : saveParam.getCustTaxNo());

        // 开户银行
        request.setGMFYH(saveParam.getInvBankNo());

        // 银行账号
        request.setGMFYHZH(saveParam.getBankAccount());

        // 联查门店开票档案，个人传0，企业传1
        if (Objects.equals(saveParam.getInvTitleType(), "ENTERPRISE")) {
            request.setGMFRYXZ("1");
        } else if (Objects.equals(saveParam.getInvTitleType(), "PERSON")) {
            request.setGMFRYXZ("0");
        }

        // 固定传0
        request.setKPLX("0");

        // 对应的中台的发票类型，专票、普票
        if (saveParam.getInvType().equals("81")) {
            //电子发票（增值税专用发票）--对应01
            request.setFPZLDM("01");
        } else if (saveParam.getInvType().equals("82")) {
            //电子发票（普通发票）--对应02
            request.setFPZLDM("02");
        }

        // 开票申请金额
        request.setJSHJ(scale(saveParam.getTotalAmt()));

//        时间段+门店名称,时间段取选取订单的最小时间与最大时间。比如:20250601-20250630XX门店 + 流水号
        List<InvoiceDetailSaveParam> details = saveParam.getDetails();
        LocalDateTime minDate = null;
        LocalDateTime maxDate = null;
        for (InvoiceDetailSaveParam detail : details) {
            if (detail.getSourceTime() != null) {
                if (minDate == null || detail.getSourceTime().isBefore(minDate)) {
                    minDate = detail.getSourceTime();
                }
                if (maxDate == null || detail.getSourceTime().isAfter(maxDate)) {
                    maxDate = detail.getSourceTime();
                }
            }
        }

        String dateRange = "";
        if (minDate != null) {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd");
            dateRange = minDate.format(formatter) + "-" + maxDate.format(formatter);
        }
//        request.setBZ(dateRange + saveParam.getInvCustName() + ";" + row.getFPQQLSH());
        request.setBZ(dateRange + saveParam.getInvCustName());

        //传销售公司的开户行
        request.setXHFKHH(saveParam.getSaleBank() == null ? "" : saveParam.getSaleBank());

        //传销售公司的银行账号
        request.setXHFYHZH(saveParam.getSaleBankAcc() == null ? "" : saveParam.getSaleBankAcc());

        request.setGMFZRRBS("N");
        request.setSFZSGMFYHZH("N");
        request.setSFZSXSFYHZH("N");
        request.setSFZSGMFDZDH("N");
        request.setSFZSXSFDZDH("N");

        // 不含税金额
        BigDecimal netAmt = BigDecimal.ZERO;
        // 税额
        BigDecimal taxAmt = BigDecimal.ZERO;

        Map<String, String> taxTypeUdc = sysUdcService.getValueMapByUdcCode("yst-supp", "TAX_CLASS_CODE");

        List<ItemDetail> itemDetails = new ArrayList<>();
        int lineNo = 1;
        for (InvoiceDetailSaveParam detail : saveParam.getDetails()) {
            netAmt = netAmt.add(detail.getExclTaxAmt());
            taxAmt = taxAmt.add(detail.getTaxAmt());
            ItemDetail drow = new ItemDetail();
            // 序号
            drow.setXH(Integer.toString(lineNo));
            // 明细类别【发票行性质（传空字符串即可，为默认通用单）】
//            drow.setDetailCategory("");
            // 商品编码（税收分类编码） 不足19未末尾补0
            drow.setSPBM(detail.getTaxCode().length() < 19 ? detail.getTaxCode().concat(StringUtils.repeat("0", 19 - detail.getTaxCode().length())) : detail.getTaxCode());
            // 商品名称
            drow.setSPMC(detail.getItemName());
            // 商品简码 - 税收分类名称 ,用*包裹
            String taxTypeName = taxTypeUdc.get(detail.getTaxCode());
            if (taxTypeName != null) {
                drow.setSPJM("*" + taxTypeName + "*");
            }
            // 规格型号
            drow.setGGXH(detail.getItemType());
            // 单位
            drow.setJLDW(detail.getUom());
            // 税率，小数，如0.06
            drow.setSLV(detail.getTaxRate().setScale(2, RoundingMode.HALF_UP).toString());
            // 数量
            drow.setSL(Objects.isNull(detail.getQty()) ? "" : detail.getQty().toString());
            // 单价
            drow.setDJ(Objects.isNull(detail.getPrice()) ? "" : scale(detail.getPrice()));
            // 含税金额
            drow.setJE(Objects.isNull(detail.getTotalAmt()) ? "" : scale(detail.getTotalAmt()));
            // 不含税金额
//            drow.setAmountWithoutTax(scale(detail.getExclTaxAmt()));
            // 税额
            drow.setSE(Objects.isNull(detail.getTaxAmt()) ? "" : scale(detail.getTaxAmt()));
            // 抵扣额
            drow.setKCE("");
            // 0-正常行 1-折扣行 2-被折扣行
            if (detail.getLineNature() != null) {
                drow.setHXZ(detail.getLineNature());
            } else {
                if (detail.getDiscountAmt() != null && detail.getDiscountAmt().compareTo(BigDecimal.ZERO) > 0) {
                    drow.setHXZ("2");
                    // 被折扣行 和 折扣行的编码要一样；与其他折扣行的编码不同
                    drow.setZKBH(detail.getItemCode() + detail.getDiscountAmt());
                } else {
                    drow.setHXZ("0");
                }
            }

            // 含税标志 默认传1（含税）
            drow.setHSBZ("1");
            itemDetails.add(drow);

            lineNo++;
            // 是否有着折扣
            if (detail.getDiscountAmt() != null && detail.getDiscountAmt().compareTo(BigDecimal.ZERO) > 0) {
                ItemDetail discountRow = new ItemDetail();

                discountRow.setXH(Integer.toString(lineNo));
                // 商品编码 （税收分类编码） 不足19未末尾补0
                discountRow.setSPBM(detail.getTaxCode().length() < 19 ? detail.getTaxCode().concat(StringUtils.repeat("0", 19 - detail.getTaxCode().length())) : detail.getTaxCode());
                // 商品名称
                discountRow.setSPMC(detail.getItemName());
                // 商品简码 - 税收分类名称 ,用*包裹
                if (taxTypeName != null) {
                    discountRow.setSPJM("*" + taxTypeName + "*");
                }
                // 规格型号 如果为折扣行，传空
                discountRow.setGGXH("");
                // 单位 如果为折扣行，传空
                discountRow.setJLDW("");
                // 税率，小数，如0.06
                discountRow.setSLV(detail.getTaxRate().setScale(2, RoundingMode.HALF_UP).toString());

                // 被折扣行 和 折扣行的编码要一样；与其他折扣行的编码不同
                discountRow.setZKBH(detail.getItemCode() + detail.getDiscountAmt());

                // 数量 如果为折扣行，传空
                discountRow.setSL("");
                // 单价 如果为折扣行，传空
                discountRow.setDJ("");
                // 含税金额 如果为折扣行，为负数
                discountRow.setJE(scale(detail.getTotalAmt().negate()));
                // 税额 如果为折扣行，为负数
                discountRow.setSE(scale(detail.getTaxAmt().negate()));
                // 抵扣额 如果为折扣行，传空
                discountRow.setKCE("");
                // 商品行没有折扣额则传0
                //有折扣额的如是商品行则传2  折扣行传1
                discountRow.setHXZ("1");
                itemDetails.add(discountRow);
                lineNo++;
            }
        }
        // 不含税金额汇总
        request.setHJJE(scale(netAmt));
        // 税额汇总
        request.setHJSE(scale(taxAmt));

        request.setMXLIST(itemDetails);

        row.setREQUEST(request);
        return row;
    }


    public InvoiceRequestSerialNoParam getInvoiceInfoByFlowNo(SaleInvdDtlDO saleInvdDtlDO) {
        InvoiceRequestSerialNoParam param = new InvoiceRequestSerialNoParam();
        param.setFPQQLSH(saleInvdDtlDO.getFlowNo());
        return param;
    }


    public String convertRedInitial(InvoiceRedApplySaveParam saveParam) {
        AisinoRedPayload row = new AisinoRedPayload();
        // 蓝字发票号码
        row.setLZFPQDHM(saveParam.getInvNo());

        //  购买方纳税人识别号
        row.setGMFNSRSBH(saveParam.getCustTaxNo() == null ? "" : saveParam.getCustTaxNo());

        // 销售方纳税人识别号
        row.setXSFNSRSBH(saveParam.getSaleTaxNo() == null ? "" : saveParam.getSaleTaxNo());

        // 蓝字开票日期 YYYY-MM-DD HH:SS:DD
        row.setLZKPRQ(saveParam.getInvDate().format(java.time.format.DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));

        // 开票方纳税人识别号(销售方纳税人识别号)
        row.setKPFNSRSBH(saveParam.getSaleTaxNo() == null ? "" : saveParam.getSaleTaxNo());
        // 归集标签
        row.setGJBQ("1");
        // 特定要素类型代码
        row.setTDYSLXDM("");
        // 对应的中台的发票类型，专票、普票
        // 01-增值税专用发票  对应81
        // 02-增值税普通发票  对应82
        row.setFPLXDM(saveParam.getInvType());
        // 蓝字发票号码（纸质）不传
        row.setLZFPHM("");
        // 蓝字发票代码（纸质） 不传
        row.setLZFPDM("");
        //发票来源代码 固定传2
        row.setFPLYDM("2");
        return JSON.toJSONString(row);
    }


    public String convertRedSave(InvoiceRedInitialRespVO saveParam) {
        //01-开票有误
        //02-销货退回
        //03-服务中止
        //04-销售折让
        saveParam.setCHYYDM("01");
        return JSON.toJSONString(saveParam);
    }

    public String convertRedOpen(AisinoRedInvoiceParam saveParam) {
        RedInvoiceOpenParam row = new RedInvoiceOpenParam();
        row.setUUID(saveParam.getUUID());
        row.setFPQQLSH(saveParam.getFPQQLSH());
        row.setYFPHM(saveParam.getYFPHM());
        row.setHZFPXXQRDBH(saveParam.getHZFPXXQRDBH());
        row.setSFZZFP(saveParam.getSFZZFP());
        row.setZZFPHM(saveParam.getZZFPHM());
        row.setFPDM(saveParam.getFPDM());
        row.setFPZLDM(saveParam.getFPZLDM());
        return JSON.toJSONString(row);
    }

    public String getBlueResult(String applyNo, LocalDateTime applyTime) {
        AisinoBlueInvoiceResultParam result = new AisinoBlueInvoiceResultParam();
        result.setFPHM(applyNo);
        result.setKPRQ(applyTime.format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")));
        return JSON.toJSONString(result);
    }
    public String getBase64Result(String applyNo, LocalDateTime applyTime) {
        BlueInvoiceResultParam result = new BlueInvoiceResultParam();
        result.setFPHM(applyNo);
        result.setKPRQ(applyTime.format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")));
        result.setWJGS("PDF");
        return JSON.toJSONString(result);
    }

    public String encrypt(String row) {
        return getEncryptedData(JSONObject.toJSONString(row));
    }

    private String getEncryptedData(String row) {
        // 加密 数据
        byte[] encryptedBytes = InvoiceIssueUtil.encrypt(row.getBytes(), encryptKey);
        if (encryptedBytes == null) {
            log.error("加密失败,加密结果为空，加密数据:{}", row);
            throw new BusinessException("加密失败,加密结果为空，加密数据:" + row);
        }
        return InvoiceIssueUtil.byteToHexString(encryptedBytes);
    }

    public <T> T decrypt(String row, Class<T> clazz) {
        log.info("解密入参:{},类名:{}", row, clazz);
        String encryptedData = getDecryptedData(row);
        log.info("解密数据:{}", encryptedData);
        return JSON.parseObject(encryptedData, clazz);
    }

    private String getDecryptedData(String encryptedData) {
        // 将16进制字符串转换为字节数组
        byte[] encryptedBytes = InvoiceIssueUtil.parseHexStr2Byte(encryptedData);
        if (encryptedBytes == null) {
            throw new RuntimeException("解密失败,加密数据格式错误，加密数据:" + encryptedData);
        }
        // 解密数据
        try {
            byte[] decryptedBytes = InvoiceIssueUtil.decrypt(encryptedBytes, encryptKey);
            return new String(decryptedBytes);
        } catch (Exception e) {
            throw new RuntimeException("解密失败,解密过程异常，加密数据:" + encryptedData, e);
        }
    }

    private String scale(BigDecimal num) {
        return num.setScale(2, RoundingMode.HALF_UP).toString();
    }
}
