package com.elitesland.fin.provider.generateaccountflow;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.creditaccountflow.CreditAccountFlowConvert;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.dto.payment.PaymentRuleConfigDtlDTO;
import com.elitesland.fin.application.facade.param.account.AccountParam;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountParam;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowQueryParam;
import com.elitesland.fin.application.facade.param.payment.PaymentRuleConfigParam;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.creditaccount.CreditAccountService;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.payment.PaymentRuleConfigDomainService;
import com.elitesland.fin.dto.generateaccountflow.GenerateAccountFlowRpcDTO;
import com.elitesland.fin.enums.PaymentRule;
import com.elitesland.fin.param.generateaccountflow.GenerateAccountFlowRpcParam;
import com.elitesland.fin.service.generateaccountflow.GenerateAccountFlowRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.provider.account
 * @date 2023/3/7 14:00
 */
@RequiredArgsConstructor
@RestController
@RequestMapping(Application.URI_PREFIX + GenerateAccountFlowRpcService.PATH)
@Slf4j
public class GenerateAccountFlowRpcServiceImpl implements GenerateAccountFlowRpcService {


    private final AccountFlowService accountFlowService;

    private final CreditAccountFlowService creditAccountFlowService;

    private final PaymentRuleConfigDomainService paymentRuleConfigDomainService;

    private final AccountService accountService;

    private final CreditAccountService creditAccountService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<List<GenerateAccountFlowRpcDTO>> generateAccountFlow(List<GenerateAccountFlowRpcParam> param) {
        log.info("开始调用生成账户流水:{}", JSON.toJSONString(param));

        checkRefundParam(param);

        List<GenerateAccountFlowRpcDTO> generateAccountFlowRpcDTOList = new ArrayList<>();
        generateAccountFlowRpcDTOList.addAll(buildAccountFlowParam(param).stream().map(item -> {
            String flowNo = accountFlowService.generateAccountFlow(item);
            GenerateAccountFlowRpcDTO generateAccountFlowRpcDTO = new GenerateAccountFlowRpcDTO();

            generateAccountFlowRpcDTO.setAccountType(item.getAccountType());
            generateAccountFlowRpcDTO.setAccountTypeName(item.getAccountTypeName());
            generateAccountFlowRpcDTO.setAccountCode(item.getAccountCode());
            generateAccountFlowRpcDTO.setAccountName(item.getAccountName());
            generateAccountFlowRpcDTO.setTime(LocalDateTime.now());
            generateAccountFlowRpcDTO.setFlowNo(flowNo);
            generateAccountFlowRpcDTO.setAmount(item.getSourceDocAmount());
            generateAccountFlowRpcDTO.setSourceId(item.getSourceId());
            return generateAccountFlowRpcDTO;
        }).collect(Collectors.toList()));

        generateAccountFlowRpcDTOList.addAll(buildCreditAccountFlowParam(param).stream().map(item ->{
            String flowNo = creditAccountFlowService.generateCreditAccountFlow(item);
            GenerateAccountFlowRpcDTO generateAccountFlowRpcDTO = new GenerateAccountFlowRpcDTO();

            generateAccountFlowRpcDTO.setAccountType(item.getCreditAccountType());
            generateAccountFlowRpcDTO.setAccountTypeName(item.getCreditAccountTypeName());
            generateAccountFlowRpcDTO.setAccountCode(item.getCreditAccountCode());
            generateAccountFlowRpcDTO.setAccountName(item.getCreditAccountName());
            generateAccountFlowRpcDTO.setTime(LocalDateTime.now());
            generateAccountFlowRpcDTO.setFlowNo(flowNo);
            generateAccountFlowRpcDTO.setAmount(item.getSourceDocAmount());
            generateAccountFlowRpcDTO.setSourceId(item.getSourceId());
            return generateAccountFlowRpcDTO;
        }).collect(Collectors.toList()));

        return ApiResult.ok(generateAccountFlowRpcDTOList);
    }

    void checkRefundParam(List<GenerateAccountFlowRpcParam> param) {
        Assert.notEmpty(param, "参数不能为空");
        param.stream().forEach(item -> {
            Assert.notEmpty(item.getAccountType(), "账户类型为空");
            Assert.notEmpty(item.getAccountName(), "账户名称为空");
            Assert.isTrue(UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode().equals(item.getAccountType()) ||
                    UdcEnum.ACCOUNT_TYPE_STORE.getValueCode().equals(item.getAccountType()) ||
                    UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getAccountType()), "账户类型错误");
        });
    }

    private List<CreditAccountFlowParam> buildCreditAccountFlowParam(List<GenerateAccountFlowRpcParam> param) {

        List<CreditAccountFlowParam> creditAccountFlowParamList = param.stream()
                .filter(item -> UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode().equals(item.getAccountType()))
                .map(item -> {
                    CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();

                    creditAccountFlowParam.setSourceDoc(item.getDoc());
                    creditAccountFlowParam.setSourceDocType(item.getDocType());
                    creditAccountFlowParam.setSourceDocStatus(item.getDocStatus());
                    creditAccountFlowParam.setSourceDocAmount(item.getDocAmount());
                    creditAccountFlowParam.setSourceId(item.getSourceId());
                    creditAccountFlowParam.setSourceNo(item.getSourceNo());
                    creditAccountFlowParam.setCreditAccountType(item.getAccountType());
                    creditAccountFlowParam.setCreditAccountTypeName(item.getAccountTypeName());
                    creditAccountFlowParam.setCreditAccountCode(item.getAccountCode());
                    creditAccountFlowParam.setCreditAccountName(item.getAccountName());
                    creditAccountFlowParam.setWorkflowProcInstId(item.getWorkflowProcInstId());
                    creditAccountFlowParam.setItemFee(item.getItemFee());
                    creditAccountFlowParam.setItemFeeOuCode(item.getItemFeeOuCode());
                    creditAccountFlowParam.setTechFee(item.getTechFee());
                    creditAccountFlowParam.setTechFeeOuCode(item.getTechFeeOuCode());
                    creditAccountFlowParam.setOuCode(item.getOuCode());
                    creditAccountFlowParam.setTOuCode(item.getItemFeeOuCode());
                    creditAccountFlowParam.setOperationFee(item.getOperationFee());
                    creditAccountFlowParam.setOperationFeeOuCode(item.getOperationFeeOuCode());
                    creditAccountFlowParam.setMarketingFee(item.getMarketingFee());
                    creditAccountFlowParam.setMarketingFeeOuCode(item.getMarketingFeeOuCode());
                    creditAccountFlowParam.setRootDoc(item.getRootDoc());
                    creditAccountFlowParam.setRootDocNo(item.getRootDocNo());
                    return creditAccountFlowParam;
                }).collect(Collectors.toList());

        return creditAccountFlowParamList;
    }

    private List<AccountFlowParam> buildAccountFlowParam(List<GenerateAccountFlowRpcParam> param) {

        List<AccountFlowParam> accountFlowParamList = param.stream()
                .filter(item -> UdcEnum.ACCOUNT_TYPE_STORE.getValueCode().equals(item.getAccountType()) ||
                        UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode().equals(item.getAccountType()))
                .map(item -> {
                    AccountFlowParam accountFlowParam = new AccountFlowParam();

                    accountFlowParam.setSourceDoc(item.getDoc());
                    accountFlowParam.setSourceDocType(item.getDocType());
                    accountFlowParam.setSourceDocStatus(item.getDocStatus());
                    accountFlowParam.setSourceDocAmount(item.getDocAmount());
                    accountFlowParam.setSourceId(item.getSourceId());
                    accountFlowParam.setSourceNo(item.getSourceNo());
                    accountFlowParam.setAccountCode(item.getAccountCode());
                    accountFlowParam.setAccountType(item.getAccountType());
                    accountFlowParam.setAccountTypeName(item.getAccountTypeName());
                    accountFlowParam.setAccountName(item.getAccountName());
                    accountFlowParam.setWorkflowProcInstId(item.getWorkflowProcInstId());
                    accountFlowParam.setIgnore(item.getIgnore());
                    accountFlowParam.setItemFee(item.getItemFee());
                    accountFlowParam.setItemFeeOuCode(item.getItemFeeOuCode());
                    accountFlowParam.setTechFee(item.getTechFee());
                    accountFlowParam.setTechFeeOuCode(item.getTechFeeOuCode());
                    accountFlowParam.setCompanyCode(item.getOuCode());
                    accountFlowParam.setOuCode(item.getOuCode());
                    accountFlowParam.setTOuCode(item.getItemFeeOuCode());
                    accountFlowParam.setOperationFee(item.getOperationFee());
                    accountFlowParam.setOperationFeeOuCode(item.getOperationFeeOuCode());
                    accountFlowParam.setMarketingFee(item.getMarketingFee());
                    accountFlowParam.setMarketingFeeOuCode(item.getMarketingFeeOuCode());
                    accountFlowParam.setRootDoc(item.getRootDoc());
                    accountFlowParam.setRootDocNo(item.getRootDocNo());
                    return accountFlowParam;
                }).collect(Collectors.toList());

        return accountFlowParamList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<List<GenerateAccountFlowRpcDTO>> generateFlow(GenerateAccountFlowRpcParam param) {
        log.info("库存调拨单生成流水参数 {}",JSON.toJSONString(param));
        checkRpcParam(param);
        ApiResult<List<GenerateAccountFlowRpcDTO>> apiResult = ApiResult.ok();
        if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_TRN_CL.getDocStatus())
                && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_TRN_CL.getDocType())) {
            // 释放流水
            apiResult = generateReleaseFlow(param);
        } else if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_OUT_CF.getDocStatus())
                && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_OUT_CF.getDocType())) {
            // 释放流水
            if (!StrUtil.equals("NO_RELEASE", param.getScene())) {
                apiResult = generateReleaseFlow(param);
            }
            PaymentRuleConfigParam paymentRuleConfigParam = buildPaymentRuleConfigParam(param);
            if (apiResult.isSuccess()) {
                // 账户扣减
                List<PaymentRuleConfigDtlDTO> paymentResult = paymentRuleConfigDomainService.pay(paymentRuleConfigParam);
                List<GenerateAccountFlowRpcDTO> data = apiResult.getData();
                if (CollectionUtil.isEmpty(data)) {
                    data = new ArrayList<>();
                }
                buildFlowRpc(paymentResult, data);
                apiResult = ApiResult.ok(data);
            }
        } else if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_OUT_CE.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_OUT_CE.getDocType())) {
            // 释放流水
            apiResult = generateReleaseFlow(param);
        } else if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_IN_CF.getDocStatus())
                && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_IN_CF.getDocType())) {
            // 流水增加储值账户
            buildAccountParam(param);
            apiResult = generateAccountFlow(List.of(param));
        } else if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_TRN_APPED.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_TRN_APPED.getDocType())) {
            // 账户占用
            PaymentRuleConfigParam paymentRuleConfigParam = buildPaymentRuleConfigParam(param);
            List<PaymentRuleConfigDtlDTO> pay = paymentRuleConfigDomainService.pay(paymentRuleConfigParam);
            List<GenerateAccountFlowRpcDTO> collect = new ArrayList<>();
            buildFlowRpc(pay, collect);
            apiResult = apiResult.ok(collect);
        }
        return apiResult;

    }

    private void checkRpcParam(GenerateAccountFlowRpcParam param) {
        Assert.notBlank(param.getObjectCode(), "对象编码不能为空");
        Assert.notBlank(param.getOuCode(), "公司编码不能为空");
        AccountParam accountParam = new AccountParam();
        accountParam.setSecOuCode(param.getOuCode());
        accountParam.setAccountHolderCode(param.getObjectCode());
        accountParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<AccountDTO> accountDTOList = accountService.getAccountByAccountParam(accountParam);
        CreditAccountParam creditAccountParam = new CreditAccountParam();
        creditAccountParam.setObjectCode(param.getObjectCode());
        creditAccountParam.setOuCode(param.getObjectCode());
        creditAccountParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<CreditAccountDTO> creditAccountDTOList = creditAccountService.getAccountByAccountParam(creditAccountParam);
        Assert.isTrue(CollectionUtil.isNotEmpty(accountDTOList) || CollectionUtil.isNotEmpty(creditAccountDTOList), "没有查询到账户信息");
    }

    private void buildFlowRpc(List<PaymentRuleConfigDtlDTO> pay, List<GenerateAccountFlowRpcDTO> collect) {
        pay.stream().filter(item->StrUtil.isNotBlank(item.getFlowNo())).forEach(item -> {
           GenerateAccountFlowRpcDTO flowRpcDTO = new GenerateAccountFlowRpcDTO();
           flowRpcDTO.setFlowNo(item.getFlowNo());
           flowRpcDTO.setAccountCode(item.getAccountCode());
           flowRpcDTO.setAmount(item.getPaymentAmount());
           flowRpcDTO.setAccountName(item.getAccountName());
           flowRpcDTO.setAccountType(item.getAccountType());
           flowRpcDTO.setAccountTypeName(item.getAccountTypeName());
           collect.add(flowRpcDTO);
       });
    }

    private void buildAccountParam(GenerateAccountFlowRpcParam param) {
        AccountParam accountParam = new AccountParam();
        accountParam.setSecOuCode(param.getOuCode());
        accountParam.setAccountHolderCode(param.getObjectCode());
        accountParam.setAccountType(UdcEnum.ACCOUNT_TYPE_STORE.getValueCode());
        List<AccountDTO> accountDTOList = accountService.getAccountByAccountParam(accountParam);
        Assert.notEmpty(accountDTOList, "没有查询到储值账户");
        AccountDTO accountDTO = accountDTOList.get(0);
        param.setAccountCode(accountDTO.getAccountCode());
        param.setAccountName(accountDTO.getAccountName());
        param.setAccountType(accountDTO.getAccountType());
        param.setAccountTypeName(accountDTO.getAccountTypeName());
    }

    private PaymentRuleConfigParam buildPaymentRuleConfigParam(GenerateAccountFlowRpcParam param) {
        PaymentRuleConfigParam paymentRuleConfigParam = new PaymentRuleConfigParam();
        paymentRuleConfigParam.setPaymentRuleConfigDtlIds(buildRpcParam(param));
        paymentRuleConfigParam.setDocAmount(param.getDocAmount());
        paymentRuleConfigParam.setOuCode(param.getOuCode());
        paymentRuleConfigParam.setObjectCode(param.getObjectCode());
        paymentRuleConfigParam.setSourceId(param.getSourceId());
        paymentRuleConfigParam.setSourceNo(param.getSourceNo());
        paymentRuleConfigParam.setDoc(param.getDoc());
        paymentRuleConfigParam.setDocType(param.getDocType());
        paymentRuleConfigParam.setDocStatus(param.getDocStatus());
        paymentRuleConfigParam.setWorkflowProcInstId(param.getWorkflowProcInstId());
        return paymentRuleConfigParam;
    }

    private Set<Long> buildRpcParam(GenerateAccountFlowRpcParam param) {
        PaymentRuleConfigParam paymentRuleConfigParam = new PaymentRuleConfigParam();
        paymentRuleConfigParam.setDoc(param.getDoc());
        paymentRuleConfigParam.setDocType(param.getDoc().concat(FinConstant.UNDERLINE).concat(param.getDocType()));
        paymentRuleConfigParam.setDocStatus(param.getDoc().concat(FinConstant.UNDERLINE).concat(param.getDocStatus()));
        List<PaymentRuleConfigDtlDTO> configDtlDTOS = paymentRuleConfigDomainService.queryPaymentRuleConfigDtl(paymentRuleConfigParam);
        Set<Long> paymentRuleConfigDtlIds = configDtlDTOS.stream().map(PaymentRuleConfigDtlDTO::getId).collect(Collectors.toSet());
        return paymentRuleConfigDtlIds;

    }

    private ApiResult<List<GenerateAccountFlowRpcDTO>> generateReleaseFlow(GenerateAccountFlowRpcParam param) {

        // 根据来源单据查询流水
        AccountFlowQueryParam accountFlowParam = new AccountFlowQueryParam();
        accountFlowParam.setSourceId(param.getSourceId());
        accountFlowParam.setSourceNo(param.getSourceNo());
        List<AccountFlowVO> accountFlowVOS = accountFlowService.queryByAccountFlowQueryParam(accountFlowParam);

        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
        creditAccountFlowParam.setSourceId(param.getSourceId());
        creditAccountFlowParam.setSourceNo(param.getSourceNo());
        List<CreditAccountFlowVO> creditAccountFlowVOS = creditAccountFlowService.selectListByQueryParam(creditAccountFlowParam);

        if (CollectionUtil.isEmpty(accountFlowVOS) && CollectionUtil.isEmpty(creditAccountFlowVOS)) {
            throw new BusinessException("没有查询到来源单据【" + param.getSourceNo() + "】的流水");
        }

        List<GenerateAccountFlowRpcParam> rpcParams = new ArrayList<>();
        Map<String, BigDecimal> amtMap = new HashMap<>();
        // 占用金额
        amtMap.put(UdcEnum.IO_TYPE_T.getValueCode(),BigDecimal.ZERO);
        // 释放金额
        amtMap.put(UdcEnum.IO_TYPE_R.getValueCode(),BigDecimal.ZERO);
        // 账户增加
        amtMap.put(UdcEnum.IO_TYPE_I.getValueCode(),BigDecimal.ZERO);
        // 账户扣减
        amtMap.put(UdcEnum.IO_TYPE_O.getValueCode(),BigDecimal.ZERO);

        // 根据占用计算需要释放的流水
        if ((StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_TRN_CL.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_TRN_CL.getDocType()))
                || (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_OUT_CF.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_OUT_CF.getDocType()))) {
            creditAccountFlowVOS = creditAccountFlowVOS.stream().filter(item -> StrUtil.equalsAny(item.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode(), UdcEnum.IO_TYPE_R.getValueCode())).collect(Collectors.toList());
            accountFlowVOS = accountFlowVOS.stream().filter(item -> StrUtil.equalsAny(item.getTransactionType(), UdcEnum.IO_TYPE_T.getValueCode(), UdcEnum.IO_TYPE_R.getValueCode())).collect(Collectors.toList());
        }
        // 根据扣减计算需要增加的流水
        if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_OUT_CE.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_OUT_CE.getDocType())) {
            creditAccountFlowVOS = creditAccountFlowVOS.stream().filter(item -> StrUtil.equalsAny(item.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode(), UdcEnum.IO_TYPE_O.getValueCode())).collect(Collectors.toList());
            accountFlowVOS = accountFlowVOS.stream().filter(item -> StrUtil.equalsAny(item.getTransactionType(), UdcEnum.IO_TYPE_I.getValueCode(), UdcEnum.IO_TYPE_O.getValueCode())).collect(Collectors.toList());
        }


        // 组装账户生成流水参数
        buildAccountFlowParam(param, accountFlowVOS, rpcParams, amtMap);
        // 组装信用账户生成流水参数
        buildCreditAccountFlowParam(param, creditAccountFlowVOS, rpcParams, amtMap);

        // 需要被释放的金额
        BigDecimal releaseAmt = amtMap.get(UdcEnum.IO_TYPE_T.getValueCode())
                .subtract(amtMap.get(UdcEnum.IO_TYPE_R.getValueCode()));

        if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_TRN_CL.getDocStatus()) && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_TRN_CL.getDocType())) {
            Assert.isTrue(ObjectUtil.equals(param.getDocAmount(), releaseAmt), "单据【{}】释放金额{}与库存提供金额{}不一致", param.getSourceNo(), releaseAmt, param.getDocAmount());
        }

        log.info("库存调拨单组装之后调用生成流水参数{}",JSON.toJSONString(rpcParams));

        return generateAccountFlow(rpcParams);
    }

    private void buildAccountFlowParam(GenerateAccountFlowRpcParam param,
                                       List<AccountFlowVO> accountFlowVOS,
                                       List<GenerateAccountFlowRpcParam> rpcParams,
                                       Map<String, BigDecimal> amtMap) {

        if (CollectionUtil.isEmpty(accountFlowVOS)) {
            return;
        }

        Map<String, List<AccountFlowVO>> accountTypeMap = accountFlowVOS.stream().collect(Collectors.groupingBy(AccountFlowVO::getAccountType));

        Set<Map.Entry<String, List<AccountFlowVO>>> entries = accountTypeMap.entrySet();
        for (Map.Entry<String, List<AccountFlowVO>> entry : entries) {
            String key = entry.getKey();
            List<AccountFlowVO> value = entry.getValue();
            GenerateAccountFlowRpcParam rpcParam = new GenerateAccountFlowRpcParam();
            buildParam(param, rpcParam);
            AccountFlowVO accountFlowVO = value.get(0);
            rpcParam.setAccountCode(accountFlowVO.getAccountCode());
            rpcParam.setAccountType(accountFlowVO.getAccountType());
            rpcParam.setAccountName(accountFlowVO.getAccountName());

            BigDecimal newAmount = BigDecimal.ZERO;
            for (AccountFlowVO item : value) {
                newAmount = computeAmt(amtMap, newAmount, item.getAmount(), item.getTransactionType());
            }
            rpcParam.setDocAmount(newAmount);
            rpcParams.add(rpcParam);
        }
    }

    private void buildParam(GenerateAccountFlowRpcParam param, GenerateAccountFlowRpcParam rpcParam) {
        rpcParam.setSourceNo(param.getSourceNo());
        rpcParam.setSourceId(param.getSourceId());
        rpcParam.setDoc(param.getDoc());
        rpcParam.setDocType(param.getDocType());
        rpcParam.setDocStatus(param.getDocStatus());
        // 调拨单出库确认出库 释放和扣减分开执行
        if (StrUtil.equals(param.getDocStatus(), PaymentRule.STKTRN_OUT_CF.getDocStatus())
                && StrUtil.equals(param.getDocType(), PaymentRule.STKTRN_OUT_CF.getDocType())) {
            rpcParam.setDocStatus(PaymentRule.STKTRN_TRN_CF.getDocStatus());
            rpcParam.setDocType(PaymentRule.STKTRN_TRN_CF.getDocType());
        }
        rpcParam.setSourceId(param.getSourceId());
        rpcParam.setSourceNo(param.getSourceNo());
        rpcParam.setWorkflowProcInstId(param.getWorkflowProcInstId());
    }

    private void buildCreditAccountFlowParam(GenerateAccountFlowRpcParam param, List<CreditAccountFlowVO> creditAccountFlowVOS, List<GenerateAccountFlowRpcParam> rpcParams, Map<String, BigDecimal> amtMap) {

        if (CollectionUtil.isEmpty(creditAccountFlowVOS)) {
            return;
        }

        CreditAccountFlowVO creditAccountFlowVO = creditAccountFlowVOS.get(0);
        GenerateAccountFlowRpcParam rpcParam = new GenerateAccountFlowRpcParam();
        buildParam(param, rpcParam);
        rpcParam.setAccountCode(creditAccountFlowVO.getCreditAccountCode());
        rpcParam.setAccountType(creditAccountFlowVO.getCreditAccountType());
        rpcParam.setAccountName(creditAccountFlowVO.getCreditAccountName());
        rpcParam.setWorkflowProcInstId(param.getWorkflowProcInstId());

        // 本次金额
        BigDecimal newAmount = BigDecimal.ZERO;

        for (CreditAccountFlowVO item : creditAccountFlowVOS) {
            newAmount = computeAmt(amtMap, newAmount, item.getAmount(), item.getTransactionType());
        }

        rpcParam.setDocAmount(newAmount);
        rpcParams.add(rpcParam);
    }

    // 计算账户需要被释放/增加的金额 释放金额=已占用金额-已释放金额 增加金额=已扣减金额-已增加金额
    private BigDecimal computeAmt(Map<String, BigDecimal> amtMap, BigDecimal newAmount, BigDecimal oldAmount, String transactionType) {

        amtMap.put(transactionType, amtMap.get(transactionType).add(oldAmount));
        if (UdcEnum.IO_TYPE_T.getValueCode().equals(transactionType)
                || UdcEnum.IO_TYPE_O.getValueCode().equals(transactionType)) {
            newAmount = newAmount.add(oldAmount);
        }
        if (UdcEnum.IO_TYPE_R.getValueCode().equals(transactionType)
                || UdcEnum.IO_TYPE_I.getValueCode().equals(transactionType)) {
            newAmount = newAmount.subtract(oldAmount);
        }
        return newAmount;
    }

}
