package com.elitesland.fin.application.service.account;


import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;

import com.elitesland.fin.application.convert.account.AccountIocConvert;
import com.elitesland.fin.application.facade.dto.account.AccountIocDTO;
import com.elitesland.fin.application.facade.param.account.AccountIocParam;
import com.elitesland.fin.application.facade.vo.account.AccountIocParamVO;
import com.elitesland.fin.application.facade.vo.account.AccountIocRespVO;
import com.elitesland.fin.application.facade.vo.account.AccountIocSaveVO;
import com.elitesland.fin.common.FinRedisConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.account.AccountIocDomainService;
import com.elitesland.fin.entity.account.AccountIoc;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import org.springframework.util.CollectionUtils;


import java.util.*;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/05/24 18:23
 */
@Slf4j
@Service
@AllArgsConstructor
public class AccountIocServiceImpl implements AccountIocService {

    private final AccountIocDomainService accountIocDomainService;
    private final RedisUtils redisUtils;


    /**
     * 新增账户事务码数据
     *
     * @param saveVo 新增账户事务码数据
     * @return 新增账户事务码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> createOne(AccountIocSaveVO saveVo) {

        AccountIoc accountIoc = AccountIocConvert.INSTANCE.saveVoToEn(saveVo);
        //新增时默认启用
        if (StringUtils.isEmpty(accountIoc.getStatus())) {
            accountIoc.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        }
        //检查必填项
        checkIocMandatory(accountIoc);

        //校验编码和名称是否已存在
        checkIocCodeAndName(accountIoc);
        //保存
        Long id = accountIocDomainService.saveAccountIoc(accountIoc, Optional.empty());
        redisUtils.del(FinRedisConstant.ACCOUNT_IOC);
        return ApiResult.ok(id);
    }

    /**
     * 修改账户事务码数据
     *
     * @param saveVo 修改账户事务码对象
     * @return 账户事务码ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> update(AccountIocSaveVO saveVo) {

        val iocDTOOptional = accountIocDomainService.findById(saveVo.getId());
        if (iocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到账户事务码信息!");
        }

        AccountIoc accountIoc = AccountIocConvert.INSTANCE.saveVoToEn(saveVo);
        //检查必填项
        checkIocMandatory(accountIoc);
        //校验编码和名称是否已存在
        checkIocCodeAndName(accountIoc);
        //保存
        Long id = accountIocDomainService.saveAccountIoc(accountIoc, iocDTOOptional);
        redisUtils.del(FinRedisConstant.ACCOUNT_IOC);
        return ApiResult.ok(id);
    }


    /**
     * 检查必填项
     *
     * @param accountIoc 账户事务码对象
     */
    private void checkIocMandatory(AccountIoc accountIoc) {
        if (StringUtils.isEmpty(accountIoc.getIoCode())) {
            throw new BusinessException(ApiCode.FAIL, "事务码不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoName())) {
            throw new BusinessException(ApiCode.FAIL, "事务码名称不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoTypeName())) {
            throw new BusinessException(ApiCode.FAIL, "事务类型名称不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getIoType())) {
            throw new BusinessException(ApiCode.FAIL, "事务类型编码不能为空!");
        }
        if (StringUtils.isEmpty(accountIoc.getStatus())) {
            throw new BusinessException(ApiCode.FAIL, "是否启用不能为空!");
        }


    }

    /**
     * 校验编码和名称是否已存在
     *
     * @param accountIoc 账户事务码对象
     */
    private void checkIocCodeAndName(AccountIoc accountIoc) {

        Set<String> ioCodeSet = new HashSet<>();
        ioCodeSet.add(accountIoc.getIoCode());

        AccountIocParam accountIocParam = new AccountIocParam();
        accountIocParam.setIoCodeSet(ioCodeSet);

        List<AccountIocDTO> accountIocDTOList = accountIocDomainService.queryByAccountIocParam(accountIocParam);


        AccountIocDTO accountIocDTO = accountIocDTOList.stream().filter(dto -> StringUtils.equals(accountIoc.getIoCode(), dto.getIoCode()))
                .findFirst()
                .orElse(null);
        if (accountIocDTO != null) {
            Assert.equals(accountIoc.getId(), accountIocDTO.getId(), "账户事务码已存在");
        }

        Set<String> ioNameSet = new HashSet<>();
        ioNameSet.add(accountIoc.getIoName());

        accountIocParam = new AccountIocParam();
        accountIocParam.setIoNameSet(ioNameSet);

        List<AccountIocDTO> accountIocNameList = accountIocDomainService.queryByAccountIocParam(accountIocParam);
        accountIocDTO = accountIocNameList.stream().filter(dto -> StringUtils.equals(accountIoc.getIoName(), dto.getIoName()))
                .findFirst()
                .orElse(null);
        if (accountIocDTO != null) {
            Assert.equals(accountIoc.getId(), accountIocDTO.getId(), "事务码名称已存在");
        }
    }


    /**
     * 通过账户事务码ID查询单条数据
     *
     * @param id 账户事务码ID
     * @return 单条账户事务码数据
     */
    @Override
    @SysCodeProc
    public ApiResult<AccountIocRespVO> findIdOne(Long id) {
        val accountIocDTOOptional = accountIocDomainService.findById(id);
        if (accountIocDTOOptional.isEmpty()) {
            return ApiResult.fail("未查询到账户事务码信息!");
        }
        AccountIocDTO accountIocDTO = accountIocDTOOptional.get();

        //转换RespVO
        AccountIocRespVO accountIocRespVO = AccountIocConvert.INSTANCE.dtoToRespVo(accountIocDTO);

        return ApiResult.ok(accountIocRespVO);
    }

    /**
     * 通过账户事务码查询单条数据
     *
     * @param code 账户事务码编码
     * @return 单条账户事务码数据
     */
    @Override
    @SysCodeProc
    public Optional<AccountIocRespVO> findByCode(String code) {
        val accountDTOOptional = accountIocDomainService.findByCode(code);
        if (accountDTOOptional.isEmpty()) {
            return Optional.empty();
        }
        AccountIocDTO accountIocDTO = accountDTOOptional.get();

        //转换RespVO
        AccountIocRespVO accountIocRespVO = AccountIocConvert.INSTANCE.dtoToRespVo(accountIocDTO);

        return Optional.of(accountIocRespVO);
    }

    /**
     * 通过账户事务码集合查询数据
     *
     * @param codeList 账户事务码编码集合
     * @return 账户事务码数据集合
     */
    @Override
    @SysCodeProc
    public List<AccountIocRespVO> findByCodeBatch(List<String> codeList) {
        List<AccountIocDTO> accountIocDTOList = accountIocDomainService.findByCodeBatch(codeList);
        if (CollectionUtils.isEmpty(accountIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<AccountIocRespVO> accountIocRespVOList = accountIocDTOList.stream().map(
                AccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return accountIocRespVOList;
    }

    /**
     * 分页查询账户事务码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条账户事务码数据
     */
    @Override
    @SysCodeProc
    public PagingVO<AccountIocRespVO> search(AccountIocParamVO param) {

        val pagingVo = accountIocDomainService.search(param);
        if (CollectionUtils.isEmpty(pagingVo.getRecords())) {
            return PagingVO.<AccountIocRespVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }
        val vos = pagingVo.getRecords();


        //转换RespVO
        List<AccountIocRespVO> whAreaSettingRespVOList = vos.stream().map(
                AccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());


        return PagingVO.<AccountIocRespVO>builder()
                .total(pagingVo.getTotal())
                .records(whAreaSettingRespVOList)
                .build();
    }

    /**
     * 根据入参查询账户事务码
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条账户事务码数据
     */
    @Override
    @SysCodeProc
    public List<AccountIocRespVO> select(AccountIocParamVO param) {
        List<AccountIocDTO> accountIocDTOList = accountIocDomainService.selectByParam(param);

        if (CollectionUtils.isEmpty(accountIocDTOList)) {
            return Collections.EMPTY_LIST;
        }

        //转换RespVO
        List<AccountIocRespVO> accountIocRespVOList = accountIocDTOList.stream().map(
                AccountIocConvert.INSTANCE::dtoToRespVo).collect(Collectors.toList());

        return accountIocRespVOList;
    }

    @Override
    public List<AccountIocDTO> queryByAccountIocParam(AccountIocParam accountIocParam) {
        return accountIocDomainService.queryByAccountIocParam(accountIocParam);
    }


    /**
     * 根据账户事务码ID批量逻辑删除
     *
     * @param ids 账户事务码ID集合
     * @return 账户事务码ID集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> deleteBatch(List<Long> ids) {

        accountIocDomainService.updateDeleteFlagByIds(1, ids);
        redisUtils.del(FinRedisConstant.ACCOUNT_IOC);
        return ApiResult.ok();
    }

}
