package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.fin.application.convert.account.AccountConvert;
import com.elitesland.fin.application.facade.excel.account.AccountImportEntity;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


@Component
@Slf4j
@RequiredArgsConstructor
public class AccountImportServiceImpl implements DataImport<AccountImportEntity> {

    private final AccountRepo accountRepo;
    private final RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;
    private final UdcProvider udcProvider;

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    @Override
    public String getTmplCode() {
        return "fin_account_import";
    }

    @Override
    public List<String> executeImport(List<AccountImportEntity> dataList, int rowId) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        // 检查数据 并保存
        return saveData(dataList, rowId);
    }


    private List<String> saveData(List<AccountImportEntity> dataList, int rowId) {

        dataList.stream().forEach(item -> {
            if (FinConstant.YES.equals(item.getDefaultAccount())) {
                item.setDefaultAccount(FinConstant.TRUE);
            }

            if (FinConstant.NO.equals(item.getDefaultAccount())) {
                item.setDefaultAccount(FinConstant.FALSE);
            }
        });

        List<AccountDO> accountDOList = AccountConvert.INSTANCE.accountImportEntityList2AccountDOList(dataList);

        List<String> errorResult = new ArrayList<>();
        // 储存成功的数据
        List<AccountDO> saveList = new ArrayList<>();

        //所有账户信息
        List<AccountDO> allAccountDOList = accountRepo.findAll();
        Set<String> accountCodeSet = allAccountDOList.stream().map(AccountDO::getAccountCode).collect(Collectors.toSet());

        //excel里面的账号编码
        Set<String> excelAccountCodeSet = new HashSet<>();

        //归属公司
        List<String> secOuCodeList = accountDOList.stream().map(AccountDO::getSecOuCode).collect(Collectors.toList());
        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuCodes(secOuCodeList);
        List<OrgOuRpcDTO> orgOuRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(orgOuRpcDtoParam);

        Set<String> ouCodeSet = orgOuRpcDTOList.stream().map(OrgOuRpcDTO::getOuCode).collect(Collectors.toSet());

        //开户主体类型
        Map<String, String> principalTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.PRINCIPAL_TYPE_INVOICE.getModel(), UdcEnum.PRINCIPAL_TYPE_INVOICE.getCode());
        Collection<String> principalTypeCollection = principalTypeUdc.values();

        //开户主体类型udc取反
        Map<String, String> principalTypeUdcName = principalTypeUdc.entrySet().stream().collect(Collectors.toMap(entry -> entry.getValue(), entry -> entry.getKey(), (v1, v2) -> v1));

        //开户主体名称
        List<String> invTitleList = accountDOList.stream().map(AccountDO::getAccountHolderName).collect(Collectors.toList());


        //账户编号
        //账户名称

        //账户类型
        Map<String, String> finAccountTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.ACCOUNT_TYPE_STORE.getModel(), UdcEnum.ACCOUNT_TYPE_STORE.getCode());
        Collection<String> finAccountTypeCollection = finAccountTypeUdc.values();

        //账户类型udc取反
        Map<String, String> finAccountTypeUdcName = finAccountTypeUdc.entrySet().stream().collect(Collectors.toMap(entry -> entry.getValue(), entry -> entry.getKey(), (v1, v2) -> v1));


        Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);

        //开户银行
        Map<String, String> unionPayUdc = udcProvider.getValueMapByUdcCode(UdcEnum.UNION_PAY_700000000000001.getModel(), UdcEnum.UNION_PAY_700000000000001.getCode());

        //开户银行udc取反
        Map<String, String> unionPayUdcName = unionPayUdc.entrySet().stream().collect(Collectors.toMap(entry -> entry.getValue(), entry -> entry.getKey(), (v1, v2) -> v1));

        for (AccountDO accountDO : accountDOList) {
            List<String> errorList = new ArrayList<>();
            //必填校验
            checkImportMandatoryField(accountDO, errorList, rowId);

            //重复账户编码校验
//            checkImportRepeatAccountCode(accountDO, errorList, accountCodeSet, excelAccountCodeSet, rowId);

            //数据有效性校验
            checkDataValid(accountDO,
                    errorList,
                    ouCodeSet,
                    principalTypeCollection,
                    finAccountTypeCollection,
                    unionPayUdc,
                    pattern,
                    rowId);

            // 如果该条数据存在错误 添加错误信息
            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.addAll(errorList);
            } else {
                // 不存在错误信息
                errorResult.add(null);
                saveList.add(accountDO);
            }

            // 行数自增
            rowId++;
        }

        if (CollectionUtils.isNotEmpty(saveList)) {
            saveData(saveList, finAccountTypeUdcName, principalTypeUdcName, unionPayUdcName);
            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }
    }

    private void saveData(List<AccountDO> saveList,
                          Map<String, String> finAccountTypeUdcName,
                          Map<String, String> principalTypeUdcName,
                          Map<String, String> unionPayUdcName) {

        //构建默认值
        buildDefaultValue(saveList, finAccountTypeUdcName, principalTypeUdcName, unionPayUdcName);
        accountRepo.saveAll(saveList);
    }

    public void buildDefaultValue(List<AccountDO> accountDOList,
                                  Map<String, String> finAccountTypeUdcName,
                                  Map<String, String> principalTypeUdcName,
                                  Map<String, String> unionPayUdcName) {
        accountDOList.stream().forEach(item -> {
            //开户主体类型
            if (StringUtils.isNotEmpty(item.getAccountHolderType())) {
                item.setAccountHolderType(principalTypeUdcName.get(item.getAccountHolderType()));
            }

            //账户名称
            if (StringUtils.isNotEmpty(item.getAccountName())) {
                item.setAccountName(unionPayUdcName.get(item.getAccountName()));
            }

            //账户类型
            if (StringUtils.isNotEmpty(item.getAccountType())) {
                item.setAccountType(finAccountTypeUdcName.get(item.getAccountType()));
            }

            //状态
            item.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        });
    }

    public void checkDataValid(AccountDO accountDO,
                               List<String> errorList,
                               Set<String> ouCodeSet,
                               Collection<String> principalTypeCollection,
                               Collection<String> finAccountTypeCollection,
                               Map<String, String> unionPayUdc,
                               Pattern pattern,
                               int rowId) {

        //归属公司
        if (StringUtils.isNotEmpty(accountDO.getSecOuCode())) {
            if (!ouCodeSet.contains(accountDO.getSecOuCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "公司编码", "公司编码在系统不存在"));
            }
        }



        //开户主体类型
        if (StringUtils.isNotEmpty(accountDO.getAccountHolderType())) {
            if (!principalTypeCollection.contains(accountDO.getAccountHolderType())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "开户主体类型", "开户主体类型在系统不存在"));
            }
        }

        //账户名称
        if (StringUtils.isNotEmpty(accountDO.getAccountName()) && StrUtil.isNotBlank(accountDO.getAccountType())) {
            // 同一个开户主体名称只有一个账户类型的账户
            if (accountRepo.existsByAccountNameAndAccountType(accountDO.getAccountName(), accountDO.getAccountType())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称已存在"));
            }
        }

        //账户类型
        if (StringUtils.isNotEmpty(accountDO.getAccountType())) {
            if (!finAccountTypeCollection.contains(accountDO.getAccountType())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户类型", "账户类型在系统不存在"));
            }
        }

        //账户金额
        if (accountDO.getAccountAmount() != null) {
            if (!pattern.matcher(accountDO.getAccountAmount().toString()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户金额", "账户金额格式有误"));
            }
        }

        //账户占用金额
        if (accountDO.getAccountOccupancyAmount() != null) {
            if (!pattern.matcher(accountDO.getAccountOccupancyAmount().toString()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户占用金额", "账户占用金额格式有误"));
            }
        }

        //账户可用金额
        if (accountDO.getAccountAvailableAmount() != null) {
            if (!pattern.matcher(accountDO.getAccountAvailableAmount().toString()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户可用金额", "账户可用金额格式有误"));
            }
        }

        //计算金额是否正确
        if (accountDO.getAccountAmount() != null &&
                accountDO.getAccountOccupancyAmount() != null &&
                accountDO.getAccountAvailableAmount() != null) {
            if (accountDO.getAccountAmount().compareTo(accountDO.getAccountOccupancyAmount().add(accountDO.getAccountAvailableAmount())) != 0) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "金额", "账户金额不等于账户占用金额+账户可用金额"));
            }
        }
        if (StrUtil.isNotBlank(accountDO.getAccountHolderCode()) && StrUtil.isNotBlank(accountDO.getSecOuCode()) && StrUtil.isNotBlank(accountDO.getAccountType())) {
            if(accountRepo.existsByAccountHolderCodeAndAccountType(accountDO.getAccountHolderCode(),accountDO.getAccountType())){
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户已经存在"));
            }
        }

    }

    private void checkImportRepeatAccountCode(AccountDO accountDO, List<String> errorList, Set<String> accountCodeSet, Set<String> excelAccountCodeSet, int rowId) {
        if (accountCodeSet.contains(accountDO.getAccountCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码在系统已经存在"));
        }

        if (StringUtils.isNotEmpty(accountDO.getAccountCode())) {
            if (excelAccountCodeSet.contains(accountDO.getAccountCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码重复"));
            }
            excelAccountCodeSet.add(accountDO.getAccountCode());
        }

    }

    private void checkImportMandatoryField(AccountDO accountDO, List<String> errorList, int rowId) {

        if (StringUtils.isEmpty(accountDO.getSecOuCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "归属公司编码", "归属公司编码必填"));
        }

        if (StringUtils.isEmpty(accountDO.getSecFranchiseeCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "归属加盟商编码", "归属加盟商编码编码必填"));
        }

        if (StringUtils.isEmpty(accountDO.getAccountHolderName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "开户主体名称", "开户主体名称必填"));
        }

        if (StringUtils.isEmpty(accountDO.getAccountCode())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码必填"));
        }

        if (StringUtils.isEmpty(accountDO.getAccountName())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称必填"));
        }

        if (StringUtils.isEmpty(accountDO.getAccountType())) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户类型", "账户类型必填"));
        }

        if (accountDO.getAccountAmount() == null) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户金额", "账户金额必填"));
        }

        if (accountDO.getAccountAvailableAmount() == null) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户可用金额", "账户可用金额必填"));
        }

        if (accountDO.getAccountOccupancyAmount() == null) {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户占用金额", "账户占用金额必填"));
        }

    }
}
