package com.elitesland.fin.domain.service.paytype;


import cn.hutool.core.collection.CollUtil;
import com.elitesland.fin.application.convert.paytype.PayTypeConvert;
import com.elitesland.fin.domain.entity.paytype.PayType;
import com.elitesland.fin.domain.entity.paytype.PayTypeDO;
import com.elitesland.fin.domain.param.paytype.PayTypePageParam;
import com.elitesland.fin.infr.dto.paytype.PayTypeDTO;
import com.elitesland.fin.infr.factory.paytype.PayTypeFactory;
import com.elitesland.fin.infr.repo.paytype.PayTypeRepo;
import com.elitesland.fin.infr.repo.paytype.PayTypeRepoProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.domain.service.payOrder
 * @date 2022/3/10 11:53
 */
@Service
@RequiredArgsConstructor
public class PayTypeDomainServiceImpl implements PayTypeDomainService {

    private final PayTypeRepo payTypeRepo;

    private final PayTypeRepoProc payTypeRepoProc;

    private final PayTypeFactory payTypeFactory;

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long makeDefault(Long id) {
        List<PayTypeDTO> dtos = payTypeRepoProc.findByIds(List.of(id));
        dtos.forEach(dto -> {
            if (dto.getDefaultFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已默认状态不能在默认");
            }
        });
        //更新之前默认为false
        payTypeRepoProc.updateDefault();
        //更新选中数据为默认
        return payTypeRepoProc.updateDefaultById(id, true);
    }

    @Override
    public PagingVO<PayTypeDTO> page(PayTypePageParam payTypeParam) {
        return payTypeFactory.payTypePage(payTypeParam);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> stopEnable(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选中数据!");
        }
        List<PayTypeDTO> dtos = payTypeRepoProc.findByIds(ids);
        dtos.forEach(dto -> {
            if (!dto.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已停用状态不能在停用");
            }
        });
        payTypeRepoProc.updateEnable(ids, false);
        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> enable(List<Long> ids) {
        payTypeRepoProc.updateEnable(ids, true);
        return ids;
    }

    @Override
    public List<PayTypeDTO> findAll() {
        return payTypeRepoProc.findAll();
    }

    @Override
    public PayTypeDTO defaultValue() {
        return payTypeRepoProc.defaultValue();
    }

    @Override
    public List<PayTypeDTO> queryByIds(Collection<Long> ids) {
        return payTypeRepoProc.findByIds(ids);
    }


    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(PayType payType) {
        payType.checkNotNull();
        Boolean existCode = payTypeRepoProc.existPayCode(payType.getPayTypeCode());
        if (existCode) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "付款单类型编码已存在");
        }
        Boolean existName = payTypeRepoProc.existPayName(payType.getPayTypeName());
        if (existName) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "付款单类型名称已存在");
        }
        //数据第一条永远保持是默认
        if (payTypeRepoProc.isFirst()) {
            payType.setDefault(false);
        } else {
            payType.setDefault(true);
        }
        PayTypeDO typeDO = PayTypeConvert.INSTANCE.convert(payType);
        PayTypeDO typeDO1 = payTypeRepo.save(typeDO);
        return typeDO1.getId();
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> delete(List<Long> ids) {
        List<PayTypeDTO> dtos = payTypeRepoProc.findByIds(ids);
        dtos.forEach(dto -> {
            if (dto.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除未启用的数据");
            }
            if (dto.getDefaultFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "默认类型不能删除");
            }
        });
        payTypeRepoProc.deleteByIds(ids);
        return ids;
    }
}
