package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.FinAccountPeriodConvert;
import com.elitesland.fin.application.convert.accountingengine.FinAccountPeriodLineConvert;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountPeriodLineParam;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountPeriodParam;
import com.elitesland.fin.application.facade.vo.accountingengine.FinAccountPeriodVO;
import com.elitesland.fin.domain.entity.accountingengine.FinAccountPeriodDO;
import com.elitesland.fin.repo.accountingengine.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class FinAccountPeriodServiceImpl implements FinAccountPeriodService {


    private final FinAccountPeriodRepo finAccountPeriodRepo;

    private final FinAccountPeriodRepoProc finAccountPeriodRepoProc;

    private final FinAccountPeriodLineRepo finAccountPeriodLineRepo;


    @SysCodeProc
    @Override
    public PagingVO<FinAccountPeriodVO> page(FinAccountPeriodParam finAccountPeriodParam) {
        return FinAccountPeriodConvert.INSTANCE.DTOToVO(finAccountPeriodRepoProc.page(finAccountPeriodParam));
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long saveOrUpdate(FinAccountPeriodParam finAccountPeriodParam) {

        checkSaveOrUpdateParam(finAccountPeriodParam);

        checkIdempotent(finAccountPeriodParam);

        checkDetailList(finAccountPeriodParam);

        FinAccountPeriodDO finAccountPeriodDO = finAccountPeriodRepo.save(FinAccountPeriodConvert.INSTANCE.paramToDO(finAccountPeriodParam));
        finAccountPeriodLineRepo.deleteAllByMasId(finAccountPeriodDO.getId());

        List<FinAccountPeriodLineParam> finAccountPeriodLineParamList = finAccountPeriodParam.getDetailList();
        finAccountPeriodLineParamList.stream().forEach(item -> item.setMasId(finAccountPeriodDO.getId()));
        finAccountPeriodLineRepo.saveAll(FinAccountPeriodLineConvert.INSTANCE.paramToDO(finAccountPeriodLineParamList));

        return finAccountPeriodDO.getId();
    }

    private void checkDetailList(FinAccountPeriodParam finAccountPeriodParam) {
        List<FinAccountPeriodLineParam> finAccountPeriodLineParamList = finAccountPeriodParam.getDetailList();

        for (int i = 0; i < finAccountPeriodLineParamList.size(); i++) {

            LocalDateTime currentActiveStartTime = finAccountPeriodLineParamList.get(i).getActiveStartTime();
            LocalDateTime currentActiveEndTime = finAccountPeriodLineParamList.get(i).getActiveEndTime();

            for (int j = 0; j < finAccountPeriodLineParamList.size(); j++) {
                if (i == j) {
                    continue;
                }

                LocalDateTime activeStartTime = finAccountPeriodLineParamList.get(j).getActiveStartTime();
                LocalDateTime activeEndTime = finAccountPeriodLineParamList.get(j).getActiveEndTime();

                Assert.isTrue(currentActiveEndTime.compareTo(activeStartTime) < 0 ||
                                currentActiveStartTime.compareTo(activeEndTime) > 0,
                        "起始日期结束日期不能重叠");
            }
        }
    }

    private void checkIdempotent(FinAccountPeriodParam finAccountPeriodParam) {

        FinAccountPeriodDO finAccountPeriodDO = finAccountPeriodRepoProc.findByAccountPeriodCode(finAccountPeriodParam.getAccountPeriodCode());

        if (finAccountPeriodDO != null) {

            Assert.equals(finAccountPeriodDO.getId(), finAccountPeriodParam.getId(), "会计期间编码已经存在");
        }
    }

    private void checkSaveOrUpdateParam(FinAccountPeriodParam finAccountPeriodParam) {
        Assert.notEmpty(finAccountPeriodParam.getAccountPeriodCode(), "会计期间编码必填");
        Assert.notEmpty(finAccountPeriodParam.getAccountPeriodName(), "会计期间名称必填");
        Assert.notEmpty(finAccountPeriodParam.getStatus(), "状态必填");

        List<FinAccountPeriodLineParam> finAccountPeriodLineParamList = finAccountPeriodParam.getDetailList();
        Assert.notEmpty(finAccountPeriodLineParamList, "明细必填");

        finAccountPeriodLineParamList.stream().forEach(item -> {
            Assert.notEmpty(item.getPeriodStyle(), "期间格式必填");
            Assert.notEmpty(item.getYear(), "年份必填");
            Assert.notEmpty(item.getQuarter(), "季度必填");
            Assert.notEmpty(item.getMonth(), "月份必填");
            Assert.notNull(item.getActiveStartTime(), "起始有效日期必填");
            Assert.notNull(item.getActiveEndTime(), "终止有效日期必填");
        });
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void enableOrDisable(FinAccountPeriodParam finAccountPeriodParam) {

        checkEnableOrDisableParam(finAccountPeriodParam);

        List<FinAccountPeriodDO> finAccountPeriodDOList = finAccountPeriodRepo.findAllById(finAccountPeriodParam.getIds());
        finAccountPeriodDOList.stream().forEach(item -> Assert.isFalse(finAccountPeriodParam.getStatus().equals(item.getStatus()), "数据已经启用/禁用"));

        finAccountPeriodDOList.stream().forEach(item -> item.setStatus(finAccountPeriodParam.getStatus()));
    }

    private void checkEnableOrDisableParam(FinAccountPeriodParam finAccountPeriodParam) {
        Assert.notEmpty(finAccountPeriodParam.getIds(), "id必填");
        Assert.notEmpty(finAccountPeriodParam.getStatus(), "状态必填");
    }
}
