package com.elitesland.fin.application.service.unionpay.interceptor;

import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.context.TenantSession;
import com.elitescloud.boot.tenant.client.support.impl.DefaultTenantProvider;
import com.elitescloud.cloudt.common.common.CloudtOptional;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitesland.fin.application.service.unionpay.config.UnionPayProperties;
import com.elitesland.fin.infinity.utils.ExceptionUtil;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.protocol.HTTP;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.core.env.Environment;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import static com.elitesland.fin.application.service.unionpay.config.UnionPayProperties.CHINA_PAY_PREFIX;

/**
 * @author eric.hao
 * @since 2023/06/08
 */
@Slf4j
@Data
@NoArgsConstructor
public class UnionPayAuthInterceptor implements HandlerInterceptor, EnvironmentAware {

    private UnionPayProperties unionPayProperties;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        response.addHeader(HTTP.CONTENT_TYPE, "application/json;charset=UTF-8");
        try {
            log.info("[UNION_PAY] Current TenantId: {}", unionPayProperties.getTenantId());
            // 设置当前租户
            setCurrentTenant(unionPayProperties.getTenantId());
            return true;
        } catch (Exception e) {
            log.error("[UNION_PAY] Auth Interceptor ERROR: {}", ExceptionUtil.formatException(e));
            response.setStatus(500);
            return false;
        } finally {
        }
    }

    @Override
    public void postHandle(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, ModelAndView modelAndView) throws Exception {
        resetCurrentTenant();
    }

    @Override
    public void afterCompletion(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse, Object o, Exception e) throws Exception {
    }

    private SysTenantDTO hasUsedTenant;
    private boolean hasUsedDefault;
    private boolean hasNot;

    private void setCurrentTenant(Long tenantId) {
        // 如果之前不使用租户，则先取消
        var hasNot = TenantSession.getNoTenant();
        if (hasNot) {
            TenantSession.clearNoTenant();
        }

        // 如果之前使用默认的，则先取消使用默认
        hasUsedDefault = TenantSession.getUseDefault();
        if (hasUsedDefault) {
            TenantSession.clearUseDefault();
        }

        // 如果之前设置了session的租户，则先更换
        hasUsedTenant = TenantSession.getCurrentTenant();

        // 设置了对应租户，设置切换
        DefaultTenantProvider tenantProvider = SpringUtil.getBean(DefaultTenantProvider.class);
        CloudtOptional<SysTenantDTO> tenantDTO = tenantProvider.getById(tenantId);
        TenantSession.setCurrentTenant(tenantDTO.get());
    }

    private void resetCurrentTenant() {
        // 恢复原session租户
        if (hasUsedTenant == null) {
            TenantSession.clearCurrentTenant();
        } else {
            TenantSession.setCurrentTenant(hasUsedTenant);
        }

        // 恢复原是否使用默认
        if (hasUsedDefault) {
            TenantSession.setUseDefault();
        }

        // 恢复不使用
        if (hasNot) {
            TenantSession.setNoTenant();
        }
    }

    @Override
    public void setEnvironment(Environment environment) {
        Binder env = Binder.get(environment);
        BindResult<UnionPayProperties> properties = env.bind(CHINA_PAY_PREFIX, UnionPayProperties.class);
        this.unionPayProperties = properties.isBound() ? properties.get() : new UnionPayProperties();
    }
}
