package com.elitesland.fin.domain.service.inputinv;

import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.convert.inputinv.InputInvConvert;
import com.elitesland.fin.application.convert.inputinv.InputInvDtlConvert;
import com.elitesland.fin.application.facade.param.inputInv.InputInvParam;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.inputinv.InputInv;
import com.elitesland.fin.domain.entity.inputinv.InputInvDO;
import com.elitesland.fin.domain.entity.inputinv.InputInvDtl;
import com.elitesland.fin.domain.entity.inputinv.InputInvDtlDO;
import com.elitesland.fin.domain.param.inputinv.InputInvPageParam;
import com.elitesland.fin.infr.dto.inputinv.InputInvDTO;
import com.elitesland.fin.infr.dto.inputinv.InputInvDtlDTO;
import com.elitesland.fin.infr.factory.inputinv.InputInvFactory;
import com.elitesland.fin.infr.repo.inputinv.InputInvDtlRepo;
import com.elitesland.fin.infr.repo.inputinv.InputInvDtlRepoProc;
import com.elitesland.fin.infr.repo.inputinv.InputInvRepo;
import com.elitesland.fin.infr.repo.inputinv.InputInvRepoProc;
import com.elitesland.workflow.ProcessInfo;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/5/6 13:08
 */
@Service
@RequiredArgsConstructor
public class InputInvDomainServiceImpl implements InputInvDomainService {

    private final InputInvRepoProc inputInvRepoProc;

    private final InputInvRepo inputInvRepo;

    private final InputInvFactory inputInvFactory;

    private final InputInvDtlRepoProc inputInvDtlRepoProc;

    private final InputInvDtlRepo inputInvDtlRepo;

    private final SeqNumProvider sysNumberRuleService;

    private final String message = "已存在的发票号码,请检查!";

    @Override
    public PagingVO<InputInvDTO> page(InputInvPageParam param) {
        return inputInvFactory.payOrderPage(param);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> deleteByIds(List<Long> ids) {
        List<InputInvDTO> inputInvDTOS = inputInvRepoProc.queryByIds(ids);
        inputInvDTOS.forEach(dto -> {
            if (!dto.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除状态为草稿的进项发票!");
            }
        });
        inputInvRepoProc.deleteByIds(ids);
        inputInvDtlRepoProc.deleteByMasId(ids);
        return ids;
    }

    @Override
    public InputInvDTO queryById(Long id, Boolean flag) {
        if (flag) {
            InputInvDTO inputInvDTO = inputInvRepoProc.queryById(id);
            List<InputInvDtlDTO> inputInvDtlDTOS = inputInvDtlRepoProc.queryByMasId(id);
            inputInvDTO.setDtlDTOList(inputInvDtlDTOS);
            return inputInvDTO;
        } else {
            return inputInvRepoProc.queryById(id);
        }
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(InputInv inputInv) {
        List<String> invNo = inputInv.getDtlList().stream().map(InputInvDtl::getInvNo).collect(Collectors.toList());
        //校验非空
        inputInv.checkInvNo();
        inputInv.checkNotNull();
        //有ID则为修改
        if (inputInv.getId() != null) {
            inputInv.checkOrderState();
            Boolean aBoolean = inputInvDtlRepoProc.checkUpdate(invNo, inputInv.getId());
            if (aBoolean) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, message);
            }
        } else {
            //新增数据才会生成付款单号
            Boolean aBoolean = inputInvDtlRepoProc.checkAdd(invNo);
            if (aBoolean) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, message);
            }
            String inputNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.JXFP, null);
            inputInv.setInvRegNo(inputNo);

        }
        inputInv.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        return updateOrSave(inputInv).getId();
    }

    private InputInvDO updateOrSave(InputInv inputInv) {
        //更新字段 主表信息
        InputInvDO inputInvDO = InputInvConvert.INSTANCE.convertToDo(inputInv);
        //先删除之前明细信息
        if (inputInv.getId() != null) {
            inputInvDtlRepoProc.deleteByMasId(List.of(inputInv.getId()));
        }
        InputInvDO save = inputInvRepo.save(inputInvDO);
        //细单信息
        List<InputInvDtlDO> inputInvDtlDOS = InputInvDtlConvert.INSTANCE.convertToDO(inputInv.getDtlList());
        //设置主键关联ID的值
        inputInvDtlDOS.forEach(dtl -> {
            //获取新增后的主表ID
            dtl.setMasId(save.getId());
        });
        inputInvDtlRepo.saveAll(inputInvDtlDOS);
        return save;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long submit(InputInv inputInv) {
        List<String> invNo = inputInv.getDtlList().stream().map(InputInvDtl::getInvNo).collect(Collectors.toList());
        inputInv.checkInvNo();
        inputInv.checkNotNull();
        inputInv.count();
        //无ID则为新增  校验订单状态
        if (inputInv.getId() != null) {
            inputInv.checkOrderState();
            Boolean aBoolean = inputInvDtlRepoProc.checkUpdate(invNo, inputInv.getId());
            if (aBoolean) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, message);
            }
        } else {
            Boolean aBoolean = inputInvDtlRepoProc.checkAdd(invNo);
            if (aBoolean) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, message);
            }
            //新增数据才会生成付款单号
            String inputNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.JXFP, null);
            inputInv.setInvRegNo(inputNo);
        }
        inputInv.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        InputInvDO inputInvDO = updateOrSave(inputInv);
        return inputInvDO.getId();
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void approved(Long id, SysUserDTO user) {
        inputInvRepoProc.updateAudit(id, null, user);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void reject(InputInvParam param, SysUserDTO user) {
        inputInvRepoProc.updateAudit(param.getId(), param.getAuditRejection(), user);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateWorkInfo(ProcessInfo processInfo, Long resId) {
        inputInvRepoProc.updateWorkInfo(processInfo, resId);
    }

    @Override
    public void saveAll(List<InputInv> inputInvs) {
        if (CollectionUtils.isEmpty(inputInvs)) {
            return;
        }
        inputInvs.stream().forEach(v -> {
            v.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
        });
        List<InputInvDO> inputInvDOS = inputInvs.stream().map(v -> {
            InputInvDO inputInvDO = InputInvConvert.INSTANCE.convertToDo(v);
            inputInvDO.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
            return inputInvDO;
        }).collect(Collectors.toList());
        inputInvRepo.saveAll(inputInvDOS);
    }
    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void savaApOrderNo(String createMode, String sourceNo, String apOrderNo) {
        inputInvRepoProc.savaApOrderNo(createMode,sourceNo,apOrderNo);
    }
    @Transactional(rollbackFor = {Exception.class})
    @Override
    public void removeApOrderNo(String createMode, String sourceNo) {
        inputInvRepoProc.removeApOrderNo(createMode,sourceNo);
    }

    @Override
    public List<InputInvDTO> findAllByParam(InputInvPageParam param) {
        return inputInvRepoProc.findAllByParam(param);
    }
}
