package com.elitesland.fin.application.service.creditaccount;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.cloudt.apm.common.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.param.creditaccount.*;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountDetailVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountPageVO;
import com.elitesland.fin.application.facade.vo.creditaccount.CreditAccountSnapshotVo;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.SysNumEnum;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.entity.creditaccount.CreditAccountSnapshotDO;
import com.elitesland.fin.repo.creditaccount.*;
import com.elitesland.fin.rpc.sale.RmiSaleRpcService;
import com.elitesland.fin.rpc.system.SystemRpcService;
import com.elitesland.sale.api.vo.param.crm.AllowShipCustGroupDParam;
import com.elitesland.sale.api.vo.resp.crm.AllowShipCustGroupDVO;
import com.elitesland.sale.constant.AllowShipCustGroupEnum;
import com.elitesland.sale.constant.AllowShipStatus;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
@Slf4j
@Service
@RequiredArgsConstructor
public class CreditAccountServiceImpl implements CreditAccountService {

    private final CreditAccountRepo creditAccountRepo;
    private final CreditAccountRepoProc creditAccountRepoProc;
    private final RmiSaleRpcService rmiSaleRpcService;
    private final SystemRpcService systemRpcService;
    private final CreditAccountInitialLimitRepo creditAccountInitialLimitRepo;
    private final CreditAccountInitialLimitRepoProc creditAccountInitialLimitRepoProc;
    private final CreditAccountSnapshotRepoProc creditAccountSnapshotRepoProc;
    private final CreditAccountSnapshotRepo creditAccountSnapshotRepo;
    private final FlexFieldUtilService flexFieldUtilService;

    @Override
    @SysCodeProc
    public PagingVO<CreditAccountPageVO> pageSearch(CreditAccountPageParam accountPageParam) {
        PagingVO<CreditAccountPageVO> pageVOPagingVO = creditAccountRepoProc.search(accountPageParam);
        if(pageVOPagingVO!=null){
            flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.CREDIT_ACCOUNT,pageVOPagingVO.getRecords());
        }
        return pageVOPagingVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object saveOrUpdate(CreditAccountSaveParam param) {
        saveOrUpdateCheck(param);
        param.setStatus(UdcEnum.APPLY_STATUS_ACTIVE.getValueCode());
        CreditAccountDO creditAccountDO = CreditAccountConvert.INSTANCE.saveParam2DO(param);
        creditAccountDO.setCreditAccountLimit(BigDecimal.ZERO);
        creditAccountDO.setCreditAccountOccupancyLimit(BigDecimal.ZERO);
        creditAccountDO.setCreditAccountUsedLimit(BigDecimal.ZERO);
        creditAccountDO.setCreditAccountAvailableLimit(BigDecimal.ZERO);
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.CREDIT_ACCOUNT, creditAccountDO);
        CreditAccountDO save = creditAccountRepo.save(creditAccountDO);
        return save.getId();
    }

    @Override
    @SysCodeProc
    public CreditAccountDetailVO queryDetailById(Long id) {
        Assert.notNull(id, "查询详情信息未传参数");
        Optional<CreditAccountDO> byId = creditAccountRepo.findById(id);
        if (byId.isEmpty()) {
            throw new BusinessException("没有查询到信用账户详情");
        }
        CreditAccountDetailVO creditAccountDetailVO = CreditAccountConvert.INSTANCE.do2DetailVO(byId.get());
        flexFieldUtilService.handleSingleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.CREDIT_ACCOUNT, creditAccountDetailVO);
        return creditAccountDetailVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(CreditAccountStatusParam param) {

        List<CreditAccountDO> creditAccountDOList = creditAccountRepo.findAllById(param.getId());
        if (CollectionUtil.isEmpty(creditAccountDOList)) {
            throw new BusinessException("没有查询到信用账户信息，请重新选择");
        }
        // 更新状态
        creditAccountRepo.updateStatusByIds(param.getStatus(), param.getId());
//        if (StrUtil.equals(param.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode())) {
//            creditAccountRepo.updateStatusByIds(param.getStatus(), param.getId());
//            return;
//        }

        // 启用账户
//        Set<String> accountCodeList = creditAccountDOList.stream().map(item -> item.getCreditAccountCode()).collect(Collectors.toSet());
//        List<CreditAccountInitialLimitDO> initialLimitDOList = creditAccountInitialLimitRepo.findByCreditAccountCodeIn(accountCodeList);
//        // 结束初始化的账户
//        Map<String, String> accountCodeStatusMap = initialLimitDOList.stream().filter(item -> StrUtil.equals(item.getStatus(), UdcEnum.INITIAL_STATUS_FINISHED.getValueCode())).collect(Collectors.toMap(CreditAccountInitialLimitDO::getCreditAccountCode, CreditAccountInitialLimitDO::getStatus, (t1, t2) -> t1));
//
//        // 删除草稿状态的期初余额
//        creditAccountInitialLimitRepo.deleteByAccountCodesAndStatus(accountCodeList, UdcEnum.INITIAL_STATUS_DRAFT.getValueCode());
//
//        List<CreditAccountParam> list = new ArrayList<>();
//        List<CreditAccountInitialLimitDO> initialLimitDOS = new ArrayList<>();
//        creditAccountDOList.stream().forEach(item -> {
//            CreditAccountParam creditAccountParam = new CreditAccountParam();
//            creditAccountParam.setId(item.getId());
//            creditAccountParam.setCreditAccountCodeName(item.getCreditAccountCode());
//            creditAccountParam.setStatus(param.getStatus());
//            // 已经结束初始化的账户
//            if (accountCodeStatusMap.containsKey(item.getCreditAccountCode())) {
//                creditAccountParam.setCreditAccountLimit(item.getCreditAccountLimit());
//                creditAccountParam.setCreditAccountOccupancyLimit(item.getCreditAccountOccupancyLimit());
//                creditAccountParam.setCreditAccountUsedLimit(item.getCreditAccountUsedLimit());
//                creditAccountParam.setCreditAccountAvailableLimit(item.getCreditAccountAvailableLimit());
//            } else {
//                CreditAccountInitialLimitDO initialLimitDO = new CreditAccountInitialLimitDO();
//                creditAccountParam.setCreditAccountLimit(BigDecimal.ZERO);
//                creditAccountParam.setCreditAccountOccupancyLimit(BigDecimal.ZERO);
//                creditAccountParam.setCreditAccountUsedLimit(BigDecimal.ZERO);
//                creditAccountParam.setCreditAccountAvailableLimit(BigDecimal.ZERO);
//                initialLimitDO.setFinishedTime(LocalDateTime.now());
//                initialLimitDO.setCreditAccountName(item.getCreditAccountName());
//                initialLimitDO.setCreditAccountCode(item.getCreditAccountCode());
//                initialLimitDO.setInitialAccountLimit(BigDecimal.ZERO);
//                initialLimitDO.setInitialAccountUsedLimit(BigDecimal.ZERO);
//                initialLimitDO.setInitialAccountOccupancyLimit(BigDecimal.ZERO);
//                initialLimitDO.setInitialAccountAvailableLimit(BigDecimal.ZERO);
//                initialLimitDO.setStatus(UdcEnum.INITIAL_STATUS_FINISHED.getValueCode());
//                initialLimitDOS.add(initialLimitDO);
//            }
//            list.add(creditAccountParam);
//        });
//        creditAccountRepoProc.updateByParam(list);
//        if(CollectionUtil.isNotEmpty(initialLimitDOS)){
//            creditAccountInitialLimitRepo.saveAll(initialLimitDOS);
//        }
    }


    @Override
    public List<CreditAccountDTO> getAccountByAccountParam(CreditAccountParam param) {
        log.info("财务查询信用账户接口,参数:{}", JSON.toJSONString(param));
        return creditAccountRepoProc.getAccountByAccountParam(param);
    }

    /**
     * 信用账户校验 不同的授信组织 同一个客户可以在不同的公司进行创建多个不同的账户
     *
     * @param param
     */
    private void saveOrUpdateCheck(CreditAccountSaveParam param) {

        // 必填项校验
//        Assert.notBlank(param.getOuCode(), "授信组织信息必填");
        Assert.notBlank(param.getObjectType(), "对象类型必填");
        Assert.notBlank(param.getObjectName(), "对象名称必填");
        Assert.notBlank(param.getObjectCode(), "对象信息必填");
        Assert.notBlank(param.getCreditAccountType(), "账户类型必填");
        Assert.notBlank(param.getCreditAccountName(), "账户名称必填");
        // 新增校验
        if (ObjectUtil.isNull(param.getId())) {

            String code = systemRpcService.sysNumberRuleGenerateCode("yst-fin", SysNumEnum.FIN_XYZH_NO.getCode(), new ArrayList<>());
            param.setCreditAccountCode(code);

            if (creditAccountRepo.existsByCreditAccountCode(code)) {
                throw new BusinessException("账户编码[" + code + "]已存在，请重新提交");
            }

            // 1.校验客户是否存在账户
            CreditAccountDO creditAccountDO = creditAccountRepo.findByObjectCode(param.getObjectCode());
            if (ObjectUtil.isNotNull(creditAccountDO)) {
                throw new BusinessException("已经存在相同账户，账户编码为:" + creditAccountDO.getCreditAccountCode());
            }

            // 根据客户组编码查询客户组  创建客户组的时候 校验是否该客户组已有客户创建了账户
            List<AllowShipCustGroupDVO> custGroupInfo = getAllowShipCustGroupInfo(param);
            // 账户类型属于客户组
            if (StrUtil.equals(param.getObjectType(), UdcEnum.OBJECT_TYPE_CUST_GROUP.getValueCode())) {

                if (CollectionUtil.isEmpty(custGroupInfo)) {
                    throw new BusinessException("未查询到客户组信息");
                }

                // 查询客户组中是否存在客户 根据客户编码查询账户信息
                List<String> custCodeList = custGroupInfo.stream().map(AllowShipCustGroupDVO::getCustCode).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(custCodeList)) {
                    List<CreditAccountDO> creditList = creditAccountRepo.findByObjectCodeIn(custCodeList);
                    List<String> collect = creditList.stream().map(CreditAccountDO::getCreditAccountCode).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(creditList)) {
                        throw new BusinessException("客户已经存在账户 " + collect + " 中，不允许重复创建，请调整客户组范围后再创建");
                    }
                }

            }// 账户类型属于客户
            else if (StrUtil.equals(param.getObjectType(), UdcEnum.OBJECT_TYPE_CUST.getValueCode())) {
                // 2.校验客户是否有对应客户组已经被创建
                if (CollectionUtil.isNotEmpty(custGroupInfo)) {
                    String groupCode = custGroupInfo.get(0).getCode();
                    CreditAccountDO creditDO = creditAccountRepo.findByObjectCode(groupCode);
                    if (ObjectUtil.isNotNull(creditDO)) {
                        throw new BusinessException("当前创建的客户已经存在账户，账户编码为:" + creditDO.getCreditAccountCode());
                    }
                }
            }
        }
    }

    private List<AllowShipCustGroupDVO> getAllowShipCustGroupInfo(CreditAccountSaveParam param) {
        AllowShipCustGroupDParam groupDParam = new AllowShipCustGroupDParam();
        if (StrUtil.equals(param.getObjectType(), UdcEnum.OBJECT_TYPE_CUST_GROUP.getValueCode())) {
            groupDParam.setCodes(List.of(param.getObjectCode()));
        } else {
            groupDParam.setCustCodes(List.of(param.getObjectCode()));
        }
        groupDParam.setType(AllowShipCustGroupEnum.CREIDT_ACCOUNT_GROUP.getType());
        groupDParam.setStatus(AllowShipStatus.ACTIVE.getType());
        List<AllowShipCustGroupDVO> custGroupInfo = rmiSaleRpcService.getCustGroupInfo(groupDParam);
        return custGroupInfo;
    }




    @Override
    @SysCodeProc
    public PagingVO<CreditAccountSnapshotVo> querySnapshot(CreditAccountSnapshotParam param) {
        PagingVO<CreditAccountSnapshotVo> creditAccountSnapshotVoPagingVO = creditAccountSnapshotRepoProc.querySnapshot(param);
        flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.CREDIT_ACCOUNT,creditAccountSnapshotVoPagingVO.getRecords());
        return creditAccountSnapshotVoPagingVO;

    }

    @Override
    public void creditAccountSnapshot(String param) {
        LocalDateTime dateTime = LocalDateTime.now();
        List<CreditAccountDO> creditAccountDOList = creditAccountRepo.findAll();
        if(CollectionUtil.isEmpty(creditAccountDOList)){
            return;
        }
        List<CreditAccountSnapshotDO> snapshotDOS = CreditAccountConvert.INSTANCE.creditDos2Snapshots(creditAccountDOList);
        snapshotDOS.stream().forEach(snapshot->{
            snapshot.setSnapshotTime(dateTime);
            snapshot.setId(null);
        });
        creditAccountSnapshotRepo.saveAll(snapshotDOS);
    }

    @Override
    @SysCodeProc
    public List<CreditAccountSnapshotDO> selectCreditAccountSnapshotByParam(CreditAccountSnapshotParam queryParam) {
        List<CreditAccountSnapshotDO> creditAccountSnapshotDOList = creditAccountSnapshotRepoProc.selectCreditAccountSnapshotByParam(queryParam);
        if (CollectionUtils.isEmpty(creditAccountSnapshotDOList)) {
            return Collections.EMPTY_LIST;
        }
        return creditAccountSnapshotDOList;
    }

    @Override
    public void updateRelateAccountCodeById(Long id, String relateAccountCode) {
        creditAccountRepoProc.updateRelateAccountCodeById(id, relateAccountCode);
    }

    @Override
    @Transactional
    public void updateRelateAccount(String objectCode, String relateAccountCode) {
        creditAccountRepoProc.updateRelateAccountCodeByObjectCode(objectCode, relateAccountCode);
    }
}
