package com.elitesland.fin.application.service.invoice.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitesland.fin.application.facade.dto.mq.InvoiceApplyProcessMqMessageDTO;
import com.elitesland.fin.application.facade.param.invoice.InvoiceApplySaveParam;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDtlDO;
import com.elitesland.fin.infr.repo.saleinv.SaleInvDtlRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvRepo;
import com.elitesland.fin.param.saleinv.InvoiceDetailSaveParam;
import com.elitesland.fin.utils.BeanUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.redisson.RedissonRedLock;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class InvoiceApplyProcessListener implements MessageQueueListener<InvoiceApplyProcessMqMessageDTO> {

    private final RedLockUtils redLockUtils;
    private final AbstractInvoiceSaveServiceImpl abstractInvoiceSaveService;
    private final SaleInvRepo saleInvRepo;
    private final SaleInvDtlRepo saleInvDtlRepo;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{InvoiceApplyProcessMqMessageDTO.SUBMIT_CHANNEL};
    }

    @Override
    public void onConsume(@NotBlank String s, @NotNull InvoiceApplyProcessMqMessageDTO messageDTO) {
        log.info("开票中发票处理提交：{}", JSON.toJSONString(messageDTO));
        RedissonRedLock redLock = null;
        try {
            // 分布式锁
            redLock = redLockUtils.getRedLock("invoice_apply_process" + messageDTO.getBusinessKey());
            boolean lockFlag = redLock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("税号为:{}的发票申请单已在处理中,获取锁失败", messageDTO.getBusinessKey());
                return;
            }
            // 幂等，判断发票状态
            List<Long> idList = messageDTO.getIdList();

            List<SaleInvDO> allByInvState = saleInvRepo.findAllById(idList);
            if (CollectionUtil.isEmpty(allByInvState)) {
                log.error("未找到待开票的发票申请单:{}", idList);
                return;
            }

            allByInvState = allByInvState.stream()
                    .filter(saleInvDO -> {
                        if (!Objects.equals(saleInvDO.getOrderState(), UdcEnum.INVOICE_STATUS_DRAFT.getValueCode())) {
                            log.error("发票申请单:{},不处于待开票状态，跳过处理", saleInvDO.getApplyNo());
                            return false;
                        }
                        return true;
                    })
                    .sorted((o1, o2) -> o2.getTotalAmt().compareTo(o1.getTotalAmt()))
                    .collect(Collectors.toList());

            List<Long> masIdList = allByInvState.stream().map(BaseModel::getId).toList();
            List<SaleInvDtlDO> allByMasIdIn = saleInvDtlRepo.findAllByMasIdIn(masIdList);
            Map<Long, List<SaleInvDtlDO>> saleInvDtlMap = allByMasIdIn.stream().collect(Collectors.groupingBy(SaleInvDtlDO::getMasId));

            List<InvoiceApplySaveParam> invoiceApplySaveParamList = new ArrayList<>();
            for (SaleInvDO record : allByInvState) {
                List<SaleInvDtlDO> saleInvDtlDOS = saleInvDtlMap.get(record.getId());

                InvoiceApplySaveParam saveParam = new InvoiceApplySaveParam();
                BeanUtils.copyProperties(record, saveParam);

                // 设置发票详情
                if (saleInvDtlDOS != null && !saleInvDtlDOS.isEmpty()) {
                    List<InvoiceDetailSaveParam> detailList = saleInvDtlDOS.stream().map(detail -> {
                        InvoiceDetailSaveParam invoiceDetail = new InvoiceDetailSaveParam();
                        BeanUtils.copyProperties(detail, invoiceDetail);
                        return invoiceDetail;
                    }).toList();
                    saveParam.setDetails(detailList);
                }
                invoiceApplySaveParamList.add(saveParam);
            }

            if (CollectionUtil.isNotEmpty(invoiceApplySaveParamList)) {
                log.info("找到 {} 条待开申请发票，开始提交...", invoiceApplySaveParamList.size());
                // 提交每个待开申请发票，发送MQ
                for (InvoiceApplySaveParam value : invoiceApplySaveParamList) {
                    if (value.getTotalAmt().compareTo(BigDecimal.ZERO) < 0) {
                        // todo 开红票 如何找蓝票
                    } else {
                        abstractInvoiceSaveService.invoiceApply(value, null);
                    }
                }
            }
        } catch (Exception e) {
            log.error("提交发票时发生异常,提交税号:{}", messageDTO.getBusinessKey(), e);
        } finally {
            if (redLock != null) {
                redLock.unlock();
                log.info("开票中发票确认完成，释放锁成功");
            }
        }
    }
}
