package com.elitesland.fin.application.service.workflow.invoice;

import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.invoiceredraft.InvoiceRedraftDO;
import com.elitesland.fin.domain.entity.invoiceredraft.QInvoiceRedraftDO;
import com.elitesland.fin.domain.entity.saleinv.QSaleInvdDtlDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDO;
import com.elitesland.fin.domain.entity.saleinv.SaleInvDtlDO;
import com.elitesland.fin.enums.InvTitleTypeEnum;
import com.elitesland.fin.infr.repo.invoiceredraft.InvoiceRedraftRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvDtlRepo;
import com.elitesland.fin.infr.repo.saleinv.SaleInvRepo;
import com.elitesland.fin.rpc.system.SystemRpcService;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 *
 * Created by lance on 2025/12/2.
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class InvoiceRedraftCallbackServiceImpl implements InvoiceRedraftCallbackService {

    private final InvoiceRedraftRepo invoiceRedraftRepo;
    private final JPAQueryFactory jpaQueryFactory;
    private final QInvoiceRedraftDO qdo = QInvoiceRedraftDO.invoiceRedraftDO;
    private final QSaleInvdDtlDO qSaleInvdDtlDO = QSaleInvdDtlDO.saleInvdDtlDO;
    private final SaleInvRepo saleInvRepo;
    private final SaleInvDtlRepo saleInvDtlRepo;
    private final SystemRpcService systemRpcService;

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void processStatusChange(String businessKey, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        //1,单据是否存在
        InvoiceRedraftDO invoiceRedraftDO = getInvRedraftDO(businessKey);

        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qdo)
                .set(qdo.workflowProcInstStatus, procInstStatus) //审批状态
                .where(qdo.id.eq(invoiceRedraftDO.getId()));

        JPAUpdateClause saleInvdDtlDOJpaUpdateClause = jpaQueryFactory.update(qSaleInvdDtlDO)
                .where(qSaleInvdDtlDO.invNo.eq(invoiceRedraftDO.getInvoiceNo()));

        switch (procInstStatus) {
            case NOTSUBMIT: // 未提交
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());//未生效

                saleInvdDtlDOJpaUpdateClause.set(qSaleInvdDtlDO.invoiceRedraftState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
            case INTERRUPT: // 中断执行
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());//草稿
                jpaUpdateClause.setNull(qdo.workflowProcInstId);//清空流程实例ID
                jpaUpdateClause.setNull(qdo.workflowProcInstStatus);//清空流程实例状态

                saleInvdDtlDOJpaUpdateClause.set(qSaleInvdDtlDO.invoiceRedraftState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
            case REJECTED: // 审批拒绝
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());//未生效
                jpaUpdateClause.set(qdo.workflowProcInstStatus, ProcInstStatus.REJECTED);

                saleInvdDtlDOJpaUpdateClause.set(qSaleInvdDtlDO.invoiceRedraftState, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
                break;
            case INVALID: // 作废
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());//已作废

                saleInvdDtlDOJpaUpdateClause.set(qSaleInvdDtlDO.invoiceRedraftState, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                break;
            case APPROVING: // 审批中
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_DOING.getValueCode());//未生效
                //审批状态
                jpaUpdateClause.set(qdo.workflowProcInstStatus, ProcInstStatus.APPROVING);

                saleInvdDtlDOJpaUpdateClause.set(qSaleInvdDtlDO.invoiceRedraftState, UdcEnum.APPLY_STATUS_DOING.getValueCode());
                break;
            case APPROVED: // 审批通过
                //查询票易通该单据的状态，若为红冲状态，则进行重新开票的生成
                jpaUpdateClause.setNull(qdo.workflowCurrentNodeName);//清空当前审批节点名称
                jpaUpdateClause.setNull(qdo.workflowCurrentNodeKey);//清空当前审批节点key
                jpaUpdateClause.setNull(qdo.workflowCurrentUserIds);//清空当前审批人
                jpaUpdateClause.set(qdo.docStatus, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());//已生效
                jpaUpdateClause.set(qdo.workflowEndTime, LocalDateTime.now());//审批完成时间
                jpaUpdateClause.set(qdo.workflowProcInstStatus, ProcInstStatus.APPROVED);//审批状态

                // 调用航信接口，发送红冲请求
                SaleInvDO originalSaleInvDO = saleInvRepo.findByApplyNo(invoiceRedraftDO.getOrigApplyNo());
                if (originalSaleInvDO == null) {
                    throw new BusinessException("未找到原开票申请单，原单号:" + invoiceRedraftDO.getOrigApplyNo());
                }
                List<SaleInvDtlDO> saleInvDtlDOS = saleInvDtlRepo.findAllByMasId(originalSaleInvDO.getId());
                if (CollectionUtil.isEmpty(saleInvDtlDOS)) {
                    throw new BusinessException("未找到原开票申请单的明细，原单号:" + invoiceRedraftDO.getOrigApplyNo());
                }

                SaleInvDO saleInvDO = new SaleInvDO();
                BeanUtils.copyProperties(originalSaleInvDO, saleInvDO);

                saleInvDO.setId(null);
                // 新开票申请单单号
                String applyNo = systemRpcService.sysNumberRuleGenerateCode(FinConstant.FIN, FinConstant.XXFP, new ArrayList<>());
                saleInvDO.setApplyNo(applyNo);

                saleInvDO.setInvType(invoiceRedraftDO.getInvoiceType());
                saleInvDO.setInvTitleType(invoiceRedraftDO.getInvTitleType());
                saleInvDO.setCustInvTitle(invoiceRedraftDO.getInvTitle());
                if (Objects.equals(invoiceRedraftDO.getInvTitleType(), InvTitleTypeEnum.ENTERPRISE.getCode())) {
                    saleInvDO.setCustTaxNo(invoiceRedraftDO.getTaxerNo());
                } else if (Objects.equals(invoiceRedraftDO.getInvTitleType(), InvTitleTypeEnum.PERSONAL.getCode())) {
                    // 个人类型不传税号
                    saleInvDO.setCustTaxNo(null);
                }
                saleInvDO.setOrderState(UdcEnum.INVOICE_STATUS_DRAFT.getValueCode());

                // 保存新开票申请单
                SaleInvDO save = saleInvRepo.save(saleInvDO);
                List<SaleInvDtlDO> newSaleInvDtlDOS = new ArrayList<>();
                saleInvDtlDOS.forEach(saleInvDtlDO -> {
                    SaleInvDtlDO saleInvDtlDO1 = new SaleInvDtlDO();
                    BeanUtils.copyProperties(saleInvDtlDO, saleInvDtlDO1);
                    saleInvDtlDO1.setMasId(save.getId());
                    newSaleInvDtlDOS.add(saleInvDtlDO1);
                });

                // 保存明细
                if (CollectionUtil.isNotEmpty(newSaleInvDtlDOS)) {
                    saleInvDtlRepo.saveAll(newSaleInvDtlDOS);
                }
                break;
        }
        //执行
        jpaUpdateClause.execute();
        saleInvdDtlDOJpaUpdateClause.execute();
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        return null;
    }

    /**
     * 查询单据
     */
    private InvoiceRedraftDO getInvRedraftDO(String businessKey) {
        Long masId = Long.valueOf(businessKey.split("#")[1].trim());
        Optional<InvoiceRedraftDO> byId = invoiceRedraftRepo.findById(masId);
        if (byId.isEmpty()) {
            throw new RuntimeException("重新开票(id:" + masId + ")不存在，无法审批");
        }
        return byId.get();
    }
}
