package com.elitesland.fin.application.service.limitadjustorder;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.log.model.bo.OperationLogDTO;
import com.elitescloud.boot.log.service.OperationLogMqMessageService;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.limitadjustorder.LimitAdjustOrderConvert;
import com.elitesland.fin.application.facade.dto.limitadjustorder.LimitAdjustOrderDTO;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowParam;
import com.elitesland.fin.application.facade.param.limitadjustorder.LimitAdjustOrderParam;
import com.elitesland.fin.application.facade.param.limitadjustorder.LimitAdjustOrderSaveParam;
import com.elitesland.fin.application.facade.vo.limitadjustorder.LimitAdjustOrderVO;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.fin.common.*;
import com.elitesland.fin.entity.creditaccount.CreditAccountDO;
import com.elitesland.fin.entity.limitadjustorder.LimitAdjustOrderDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepo;
import com.elitesland.fin.repo.limitadjustorder.LimitAdjustOrderRepo;
import com.elitesland.fin.repo.limitadjustorder.LimitAdjustOrderRepoProc;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2023/2/25 10:07
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class LimitAdjustOrderServiceImpl implements LimitAdjustOrderService {

    private final LimitAdjustOrderRepo limitAdjustOrderRepo;

    private final LimitAdjustOrderRepoProc limitAdjustOrderRepoProc;

    private final SysNumberGenerator sysNumberGenerator;

    private final WorkflowRpcService workflowRpcService;

    private final JPAQueryFactory jpaQueryFactory;

    private final CreditAccountFlowService creditAccountFlowService;

    private final CreditAccountFlowRepo creditAccountFlowRepo;

    private final CreditAccountRepo creditAccountRepo;

    private final CreditAccountRepoProc creditAccountRepoProc;

    private final TransactionTemplate transactionTemplate;
    private final OperationLogMqMessageService operationLogMqMessageService;
    private final FlexFieldUtilService flexFieldUtilService;

    @SysCodeProc
    @Override
    public PagingVO<LimitAdjustOrderVO> page(LimitAdjustOrderParam limitAdjustOrderParam) {

        PagingVO<LimitAdjustOrderVO> limitAdjustOrderVOPagingVO = LimitAdjustOrderConvert.INSTANCE
                .limitAdjustOrderDTOPagingVO2LimitAdjustOrderVOPagingVO(limitAdjustOrderRepoProc.page(limitAdjustOrderParam));
        flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.LIMIT_ADJUST_ORDER, limitAdjustOrderVOPagingVO.getRecords());
        return limitAdjustOrderVOPagingVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdate(LimitAdjustOrderSaveParam limitAdjustOrderSaveParam) {
        checkSave(limitAdjustOrderSaveParam);

        CreditAccountDO creditAccountDO = creditAccountRepoProc.findByCreditAccountCode(limitAdjustOrderSaveParam.getCreditAccountCode());
        // 调出单
        if (StrUtil.equals(limitAdjustOrderSaveParam.getAdjustType(), UdcEnum.ADJUST_TYPE_2.getValueCode())
                && NumberUtil.isGreater(limitAdjustOrderSaveParam.getAdjustLimit(), creditAccountDO.getCreditAccountLimit())) {
            throw new BusinessException(String.format("调整单调出金额%s大于信用账户额度%s",
                    limitAdjustOrderSaveParam.getAdjustLimit().setScale(2, RoundingMode.HALF_UP),
                    creditAccountDO.getCreditAccountLimit().setScale(2, RoundingMode.HALF_UP)));
        }

        if (limitAdjustOrderSaveParam.getId() == null) {
            LimitAdjustOrderDO limitAdjustOrderDO = LimitAdjustOrderConvert.INSTANCE.limitAdjustOrderParam2LimitAdjustOrderDO(limitAdjustOrderSaveParam);

            limitAdjustOrderDO.setWorkflowProcInstStatus(ProcInstStatus.NOTSUBMIT);
            limitAdjustOrderDO.setDocState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
            limitAdjustOrderDO.setDocNo(sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode()));
            flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.LIMIT_ADJUST_ORDER, limitAdjustOrderDO);
            limitAdjustOrderRepo.save(limitAdjustOrderDO);

            /***人工记录业务操作日志****/
            OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    BusinessObjectEnum.FIN_LIMIT_ADJUST,
                    limitAdjustOrderDO.getDocNo(), OperationTypeEnum.ADD,
                    "新增额度调整单" );
            operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
            return limitAdjustOrderDO.getId();
        }else{
            LimitAdjustOrderDO limitAdjustOrderDO = limitAdjustOrderRepoProc.findById(limitAdjustOrderSaveParam.getId());
            checkAdjustOrderState(limitAdjustOrderDO);
            LimitAdjustOrderConvert.INSTANCE.limitAdjustOrderParam2LimitAdjustOrderDO(limitAdjustOrderSaveParam, limitAdjustOrderDO);
            flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.LIMIT_ADJUST_ORDER, limitAdjustOrderDO);
            /***人工记录业务操作日志****/
            OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    BusinessObjectEnum.FIN_LIMIT_ADJUST,
                    limitAdjustOrderDO.getDocNo(), OperationTypeEnum.UPDATE,
                    "修改额度调整单" );
            operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
            return limitAdjustOrderDO.getId();
        }
    }

    @Override
    public void submit(Long id) {

        LimitAdjustOrderDO limitAdjustOrderDO = limitAdjustOrderRepoProc.findById(id);
		log.info("额度调整单,{}",JSONUtil.toJsonStr(limitAdjustOrderDO));
        CreditAccountDO creditAccountDO = creditAccountRepoProc.findByCreditAccountCode(limitAdjustOrderDO.getCreditAccountCode());
		log.info("信用账户信息,{}",JSONUtil.toJsonStr(creditAccountDO));
        Assert.isTrue(StrUtil.equals(limitAdjustOrderDO.getDocState(), UdcEnum.APPLY_STATUS_DRAFT.getValueCode()),
                "调整单{}不处于待提交状态，请勿提交", limitAdjustOrderDO.getDocNo());

        // 调出单
        if (StrUtil.equals(limitAdjustOrderDO.getAdjustType(), UdcEnum.ADJUST_TYPE_2.getValueCode())
                && NumberUtil.isGreater(limitAdjustOrderDO.getAdjustLimit(), creditAccountDO.getCreditAccountLimit())) {
            throw new BusinessException(String.format("调整单调出金额%s大于信用账户额度%s",
                    limitAdjustOrderDO.getAdjustLimit().setScale(2, RoundingMode.HALF_UP),
                    creditAccountDO.getCreditAccountLimit().setScale(2, RoundingMode.HALF_UP)));
        }
        try {
            //埋点日志
            OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    BusinessObjectEnum.FIN_LIMIT_ADJUST,
                    limitAdjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_SUBMIT,
                    "提交审批" );

            workFlow(limitAdjustOrderDO);

            operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);

        } catch (Exception e) {
            log.error("信用调整单调用工作流异常,参数：{}，原因： {}", limitAdjustOrderDO, e.getMessage());
            throw new BusinessException(e.getMessage());
        }
    }

    private void checkAdjustOrderState(LimitAdjustOrderDO limitAdjustOrderDO) {
        Assert.equals(limitAdjustOrderDO.getDocState(), UdcEnum.APPLY_STATUS_DRAFT.getValueCode(), "当前状态无法编辑");
    }

    private CreditAccountFlowParam buildCreditAccountFlowParam(LimitAdjustOrderDO limitAdjustOrderDO,String docStatus) {
        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
        creditAccountFlowParam.setSourceNo(limitAdjustOrderDO.getDocNo());
        creditAccountFlowParam.setSourceId(limitAdjustOrderDO.getId());
        creditAccountFlowParam.setSourceDoc(UdcEnum.DOC_CLS_AO.getValueCode());
        creditAccountFlowParam.setSourceDocType(UdcEnum.ADJUST_TYPE_2.getValueCode());
        creditAccountFlowParam.setSourceDocStatus(docStatus);
        creditAccountFlowParam.setSourceDocAmount(limitAdjustOrderDO.getAdjustLimit());
        creditAccountFlowParam.setCreditAccountCode(limitAdjustOrderDO.getCreditAccountCode());
        return creditAccountFlowParam;
    }

    private void checkSave(LimitAdjustOrderSaveParam limitAdjustOrderParam) {
        Assert.notEmpty(limitAdjustOrderParam.getAdjustType(), "调整类型不能为空!");
        Assert.notEmpty(limitAdjustOrderParam.getAdjustReason(), "调整原因不能为空!");
        Assert.notEmpty(limitAdjustOrderParam.getCreditAccountName(), "信用账户名称不能为空!");
        Assert.notEmpty(limitAdjustOrderParam.getCreditAccountCode(), "信用账户编码不能为空!");
        Assert.notNull(limitAdjustOrderParam.getAdjustLimit(), "调整金额不能为空!");
        if (limitAdjustOrderParam.getAdjustLimit().compareTo(BigDecimal.ZERO) <= 0) {
            throw new BusinessException("调整金额不能小于或等于0");
        }
    }

    public void workFlow(LimitAdjustOrderDO limitAdjustOrderDO) {
		log.info("工作流审批:{}", JSONUtil.toJsonStr(limitAdjustOrderDO));
		log.info("工作流审批CAN_START_PROC_STATUSES:{}", WorkflowConstant.CAN_START_PROC_STATUSES);
        String docNo = limitAdjustOrderDO.getDocNo();

        if (limitAdjustOrderDO.getWorkflowProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(limitAdjustOrderDO.getWorkflowProcInstStatus())) {
            /**
             * 为什么要加上面3个判断，是因为流程已经启动了,就没必要再启动流程(场景:在'审批中'的过程中，做保存操作)
             * 工作流规则:
             *  1,一个业务单据在一个流程没结束前是不能再起的同一个工作流
             *  2,一个业务单据可以同时有2个"不同类型"的流程再跑
             */
            String procInstName = "信用调整单申请(" + docNo + ")审批";
            String businessKey = docNo + "#" + limitAdjustOrderDO.getId();
			log.info("提交工作流,{}-{}",procInstName,businessKey);
            ProcessInfo processInfo = workflowRpcService.startProcess(
                    WorkFlowDefKey.FIN_CREDIT_ADJUST.name(),
                    procInstName,
                    businessKey,
                    new HashMap<>());
			log.info("工作流返回:{}", JSONUtil.toJsonStr(processInfo));
            limitAdjustOrderRepoProc.updateWorkflow(limitAdjustOrderDO, processInfo);
        }
    }

    @Override
    public void expireLimitAdjustOrder() {

        LimitAdjustOrderParam limitAdjustOrderParam = new LimitAdjustOrderParam();
        limitAdjustOrderParam.setCreditType(UdcEnum.CREDIT_TYPE_2.getValueCode());
        limitAdjustOrderParam.setDocState(UdcEnum.APPLY_STATUS_ACTIVE.getValueCode());
        limitAdjustOrderParam.setCurrentTime(LocalDateTime.now());
        List<LimitAdjustOrderDTO> limitAdjustOrderDTOList = limitAdjustOrderRepoProc.queryByParam(limitAdjustOrderParam);
        if (CollectionUtils.isNotEmpty(limitAdjustOrderDTOList)) {
            limitAdjustOrderDTOList.stream().forEach(item -> {
                CreditAccountFlowParam creditAccountFlowParam = buildCreditAccountFlowParam(item, UdcEnum.APPLY_STATUS_EXPIRED.getValueCode());
                creditAccountFlowService.generateCreditAccountFlow(creditAccountFlowParam);
            });

            limitAdjustOrderParam.setIds(limitAdjustOrderDTOList.stream().map(LimitAdjustOrderDTO::getId).collect(Collectors.toList()));
            limitAdjustOrderParam.setDocState(UdcEnum.APPLY_STATUS_EXPIRED.getValueCode());
            limitAdjustOrderRepoProc.updateDocStateByIds(limitAdjustOrderParam);
        }
    }

    @Override
    public void activeLimitAdjustOrder() {
        LimitAdjustOrderParam limitAdjustOrderParam = new LimitAdjustOrderParam();
        limitAdjustOrderParam.setDocState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
        limitAdjustOrderParam.setCurrEffectiveTime(LocalDateTime.now());
        List<LimitAdjustOrderDTO> limitAdjustOrderDTOList = limitAdjustOrderRepoProc.queryByParam(limitAdjustOrderParam);
        if (CollectionUtils.isNotEmpty(limitAdjustOrderDTOList)) {
            limitAdjustOrderDTOList.stream().forEach(item -> {
                CreditAccountFlowParam creditAccountFlowParam = buildCreditAccountFlowParam(item, UdcEnum.APPLY_STATUS_ACTIVE.getValueCode());
                creditAccountFlowService.generateCreditAccountFlow(creditAccountFlowParam);
            });

            limitAdjustOrderParam.setIds(limitAdjustOrderDTOList.stream().map(LimitAdjustOrderDTO::getId).collect(Collectors.toList()));
            limitAdjustOrderParam.setDocState(UdcEnum.APPLY_STATUS_ACTIVE.getValueCode());
            limitAdjustOrderRepoProc.updateDocStateByIds(limitAdjustOrderParam);
        }
    }

    private CreditAccountFlowParam buildCreditAccountFlowParam(LimitAdjustOrderDTO limitAdjustOrderDTO, String sourceDocStatus) {
        CreditAccountFlowParam creditAccountFlowParam = new CreditAccountFlowParam();
        creditAccountFlowParam.setSourceNo(limitAdjustOrderDTO.getDocNo());
        creditAccountFlowParam.setSourceId(limitAdjustOrderDTO.getId());
        creditAccountFlowParam.setSourceDoc(UdcEnum.DOC_CLS_AO.getValueCode());
        creditAccountFlowParam.setSourceDocType(limitAdjustOrderDTO.getAdjustType());
        creditAccountFlowParam.setSourceDocStatus(sourceDocStatus);
        creditAccountFlowParam.setSourceDocAmount(limitAdjustOrderDTO.getAdjustLimit());
        creditAccountFlowParam.setCreditAccountCode(limitAdjustOrderDTO.getCreditAccountCode());
        return creditAccountFlowParam;
    }
}
