package com.elitesland.fin.application.service.adjustorder;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.log.model.bo.OperationLogDTO;
import com.elitescloud.boot.log.service.OperationLogMqMessageService;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.convert.adjustorder.AdjustOrderConvert;
import com.elitesland.fin.application.facade.param.adjustorder.AdjustOrderPageParam;
import com.elitesland.fin.application.facade.param.adjustorder.AdjustOrderParam;
import com.elitesland.fin.application.facade.param.adjustorder.AdjustOrderSaveParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.param.flow.SaveFlowParam;
import com.elitesland.fin.application.facade.vo.adjustorder.AdjustOrderVO;
import com.elitesland.fin.application.service.account.AccountService;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.fin.common.*;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepoProc;
import com.elitesland.fin.repo.flow.AccountFlowRepo;
import com.elitesland.fin.rpc.workflow.WorkflowRpcService;
import com.elitesland.fin.service.flow.AccountFlowRpcService;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2023/2/25 10:07
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AdjustOrderServiceImpl implements AdjustOrderService {

    private final AdjustOrderRepo adjustOrderRepo;

    private final AdjustOrderRepoProc adjustOrderRepoProc;

    private final AccountService accountService;

    private final SysNumberGenerator sysNumberGenerator;

    private final AccountFlowService accountFlowService;

    private final WorkflowRpcService workflowRpcService;

    private final JPAQueryFactory jpaQueryFactory;

    private final AccountFlowRpcService accountFlowRpcService;

    private final TransactionTemplate transactionTemplate;
    private final AccountRepo accountRepo;
    private final AccountRepoProc accountRepoProc;
    private final AccountFlowRepo accountFlowRepo;
    private final OperationLogMqMessageService operationLogMqMessageService;
    private final FlexFieldUtilService flexFieldUtilService;

    @SysCodeProc
    @Override
    public PagingVO<AdjustOrderVO> page(AdjustOrderPageParam adjustOrderPageParam) {
        PagingVO<AdjustOrderVO> page = adjustOrderRepoProc.page(adjustOrderPageParam);
        if(page.isNotEmpty()){
            flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.ADJUST_ORDER,page.getRecords());
        }
        return page;

    }

    @SysCodeProc
    @Override
    public AdjustOrderVO getOne(Long id) {
        AdjustOrderVO adjustOrderVO = adjustOrderRepoProc.queryById(id);
        flexFieldUtilService.handleSingleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.ADJUST_ORDER,adjustOrderVO);
        return adjustOrderVO;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> approve(List<Long> ids) {
        List<AdjustOrderVO> adjustOrderVOS = adjustOrderRepoProc.queryByIds(ids);
        String docNos = adjustOrderVOS.stream()
                .peek(vo -> {
                    if (!vo.getState().equals(UdcEnum.APPLY_STATUS_DOING.getValueCode())) {
                        throw new BusinessException("只有审批中状态单据才可审批!");
                    }
                })
                .map(AdjustOrderVO::getDocNo)
                .collect(Collectors.joining(","));

        SysUserDTO user = getUser();
        adjustOrderRepoProc.updateByIds(ids, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode(), user);
        //处理流水
        if (!CollectionUtils.isEmpty(adjustOrderVOS)) {
            AdjustOrderVO adjustOrderVO = adjustOrderVOS.get(0);
            SaveFlowParam saveFlowParam = VoToFlowParam(adjustOrderVO, FinConstant.APPROVED);
            saveFlow(saveFlowParam);
        }

        adjustOrderVOS.forEach(vo -> {
            /***人工记录业务操作日志****/
            OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    BusinessObjectEnum.FIN_ADJUST,
                    vo.getDocNo(), OperationTypeEnum.APPROVE_SUBMIT,
                    "审批调整单");
            operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        });

        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> reject(AdjustOrderParam adjustOrderParam) {
        List<AdjustOrderVO> adjustOrderVOS = adjustOrderRepoProc.queryByIds(adjustOrderParam.getIds());
        adjustOrderVOS.forEach(vo -> {
            if (!vo.getState().equals(UdcEnum.APPLY_STATUS_DOING.getValueCode())) {
                throw new BusinessException("只有审批中状态单据才可拒绝!");
            }
        });
        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.FIN_ADJUST,
                adjustOrderParam.getDocNo(), OperationTypeEnum.APPROVE_REJECT,
                "审批拒绝调整单:" );
        SysUserDTO user = getUser();
        adjustOrderRepoProc.updateReject(adjustOrderParam.getIds(), UdcEnum.COM_APPR_STATUS_REJECTED.getValueCode(), user, adjustOrderParam.getAuditRejectReason());
        //处理流水 目前只处理单个
        if (!CollectionUtils.isEmpty(adjustOrderVOS)) {
            AdjustOrderVO adjustOrderVO = adjustOrderVOS.get(0);
            SaveFlowParam saveFlowParam = VoToFlowParam(adjustOrderVO, FinConstant.REJECT);
            saveFlow(saveFlowParam);
        }


        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        return adjustOrderParam.getIds();
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(AdjustOrderSaveParam adjustOrderSaveParam) {
        checkSave(adjustOrderSaveParam);
        AdjustOrderDO adjustOrderDO = AdjustOrderConvert.INSTANCE.saveParamToDO(adjustOrderSaveParam);
        AdjustOrderDO save;
        adjustOrderDO.setWorkflowProcInstStatus(ProcInstStatus.NOTSUBMIT);
        adjustOrderDO.setState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        OperationLogDTO dto ;
        if (ObjectUtil.isNull(adjustOrderSaveParam.getId())) {
            String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());
            adjustOrderDO.setDocNo(docNo);
            adjustOrderDO.setId(null);
            /***人工记录业务操作日志****/
              dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    BusinessObjectEnum.FIN_ADJUST,
                    adjustOrderDO.getDocNo(), OperationTypeEnum.ADD,
                    "新增调整单" );
        } else {
			checkAdjustOrderState(adjustOrderSaveParam);
            // 流程实例id赋值
            adjustOrderDO.setWorkflowProcInstId(adjustOrderSaveParam.getProcInstId());
			/***人工记录业务操作日志****/
			dto = operationLogMqMessageService.quickNewOperationLogDTO(
					BusinessObjectEnum.FIN_ADJUST,
					adjustOrderDO.getDocNo(), OperationTypeEnum.UPDATE,
					"编辑调整单");
		}
		// 调出需要校验调整金额是否超过账户可用金额
		if(UdcEnum.ADJUST_TYPE_2.getValueCode().equals(adjustOrderSaveParam.getAdjType())){
			checkTotalAmtLimit(adjustOrderSaveParam);
		}
        log.info("保存账户调整单,{}", JSONUtil.toJsonStr(adjustOrderDO));
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.ADJUST_ORDER, adjustOrderDO);
        save = adjustOrderRepo.save(adjustOrderDO);
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);

        return save.getId();
    }

	/**
	 * 校验调整金额是否超过账户可用金额
	 * @param adjustOrderSaveParam
	 */
	private void checkTotalAmtLimit(AdjustOrderSaveParam adjustOrderSaveParam) {
		// 查询账户
		AccountDO accountDO = accountRepoProc.findByAccountCode(adjustOrderSaveParam.getAccCode());
		// 校验调出单的账户余额
		Assert.isTrue(NumberUtil.isGreaterOrEqual(accountDO.getAccountAvailableAmount(),
                adjustOrderSaveParam.getTotalAmt()), "调出金额 [" + adjustOrderSaveParam.getTotalAmt() + "] 不能超过账户可用金额[" + accountDO.getAccountAvailableAmount().stripTrailingZeros().toPlainString() + "]");
	}

	@Transactional(rollbackFor = {Exception.class})
    @Override
    public Long update(AdjustOrderSaveParam adjustOrderSaveParam) {
        checkSave(adjustOrderSaveParam);
        AdjustOrderDO adjustOrderDO = AdjustOrderConvert.INSTANCE.saveParamToDO(adjustOrderSaveParam);
        if (StringUtils.isEmpty(adjustOrderSaveParam.getDocNo())) {
            String docNo = sysNumberGenerator.generate(SysNumEnum.FIN_ADJ_ORDER.getCode());
            adjustOrderDO.setDocNo(docNo);
        }
        // 流程实例id赋值
        adjustOrderDO.setWorkflowProcInstId(adjustOrderSaveParam.getProcInstId());
        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.FIN_ADJUST,
                adjustOrderDO.getDocNo(), OperationTypeEnum.UPDATE,
                "修改调整单" );
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.ADJUST_ORDER, adjustOrderDO);
        AdjustOrderDO save = adjustOrderRepo.save(adjustOrderDO);
        SaveFlowParam saveFlowParam = DoToFlowParam(save, FinConstant.SUBMIT);
        saveFlow(saveFlowParam);

        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        return save.getId();
    }

    @Override
    public Long submit(Long id) {

        // 查询调整单
        Optional<AdjustOrderDO> byId = adjustOrderRepo.findById(id);
        if (byId.isEmpty()) {
            throw new BusinessException("没有查询到调整单");
        }
        AdjustOrderDO adjustOrderDO = byId.get();

        if (!StrUtil.equals(adjustOrderDO.getState(), UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
            throw new BusinessException("调整单不处于待提交状态，请勿提交");
        }
        /***人工记录业务操作日志****/
        OperationLogDTO dto = operationLogMqMessageService.quickNewOperationLogDTO(
                BusinessObjectEnum.FIN_ADJUST,
                adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_SUBMIT,
                "提交审批" );
        workflowAndGenerateAccountFlow(adjustOrderDO);

        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
        return adjustOrderDO.getId();
    }

    public void workflowAndGenerateAccountFlow(AdjustOrderDO adjustOrderDO) {
        List<String> flowNos = new ArrayList<>();
        // 调出单生成占用流水
        if (StrUtil.equals(adjustOrderDO.getAdjType(), UdcEnum.ADJUST_TYPE_2.getValueCode())) {

            // 查询账户
            AccountDO accountDO = accountRepoProc.findByAccountCode(adjustOrderDO.getAccCode());
            // 校验调出单的账户余额
            Assert.isTrue(NumberUtil.isGreaterOrEqual(accountDO.getAccountAvailableAmount(),
                    adjustOrderDO.getTotalAmt()), "调出金额 [" + adjustOrderDO.getTotalAmt().setScale(2, RoundingMode.HALF_UP) + "] 不能超过账户可用金额[" + accountDO.getAccountAvailableAmount().setScale(2, RoundingMode.HALF_UP) + "]");

            // 生成流水
            AccountFlowParam flowParam = adjustDoToFlowParam(adjustOrderDO, UdcEnum.APPLY_STATUS_DOING.getValueCode());
            String flowNo = accountFlowService.generateAccountFlow(flowParam);
            flowNos.add(flowNo);
        }
        try {
            workFlow(adjustOrderDO);
        } catch (Exception e) {
            log.error("调整单调用工作流异常,参数：{}，原因： {}", adjustOrderDO, e.getMessage());
            // 调出单工作流异常 释放流水 做补偿
            if (StrUtil.equals(adjustOrderDO.getAdjType(), UdcEnum.ADJUST_TYPE_2.getValueCode())) {
                // 工作流异常回滚流水
                AccountFlowParam flowParam = adjustDoToFlowParam(adjustOrderDO, UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
                String flowNo = accountFlowService.generateAccountFlow(flowParam);
                flowNos.add(flowNo);
                accountFlowRepo.deleteByFlowNos(flowNos);
            }
            throw new BusinessException(e.getMessage());
        }
    }

    private AdjustOrderDO checkAdjustOrderState(AdjustOrderSaveParam adjustOrderSaveParam) {
        Optional<AdjustOrderDO> optional = adjustOrderRepo.findById(adjustOrderSaveParam.getId());
        Assert.isTrue(optional.isPresent(), "没有查询到当前[" + adjustOrderSaveParam.getDocNo() + "]调整单!");
        Assert.isFalse(StrUtil.equals(optional.get().getState(), UdcEnum.APPLY_STATUS_DOING.getValueCode()), "当前[" + adjustOrderSaveParam.getDocNo() + "]调整单正在审核中");
        Assert.isFalse(StrUtil.equals(optional.get().getState(), UdcEnum.APPLY_STATUS_COMPLETE.getValueCode()), "当前[" + adjustOrderSaveParam.getDocNo() + "]调整单已经审核通过");
        Assert.isFalse(StrUtil.equals(optional.get().getState(), UdcEnum.APPLY_STATUS_REJECTED.getValueCode()), "当前[" + adjustOrderSaveParam.getDocNo() + "]调整单被拒绝，请重新发起申请单");
        return optional.get();
    }

    private AccountFlowParam adjustDoToFlowParam(AdjustOrderDO adjustOrderDO, String sourceDocStatus) {
        AccountFlowParam flowRpcParam = new AccountFlowParam();
        flowRpcParam.setSourceNo(adjustOrderDO.getDocNo());
        flowRpcParam.setSourceId(adjustOrderDO.getId());
        flowRpcParam.setSourceDoc(UdcEnum.DOC_CLS_AO.getValueCode());
        flowRpcParam.setSourceDocType(adjustOrderDO.getAdjType());
        flowRpcParam.setSourceDocStatus(sourceDocStatus);
        flowRpcParam.setSourceDocAmount(adjustOrderDO.getTotalAmt());
        flowRpcParam.setAccountCode(adjustOrderDO.getAccCode());
		// 用于判断是否启用工作流
        flowRpcParam.setWorkflowProcInstId(adjustOrderDO.getWorkflowProcInstId());
        return flowRpcParam;
    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            throw new BusinessException("当前登陆人信息获取为空!");
        }
        return userDetails.getUser();
    }

    private BigDecimal checkAmt(BigDecimal amt) {
        return amt == null ? BigDecimal.ZERO : amt;
    }

    private void checkSave(AdjustOrderSaveParam adjustOrderParam) {
        Assert.notEmpty(adjustOrderParam.getAdjType(), "调整类型不能为空!");
        Assert.notEmpty(adjustOrderParam.getAdjReason(), "调整原因不能为空!");
        Assert.notEmpty(adjustOrderParam.getAccName(), "账户名称不能为空!");
        Assert.notEmpty(adjustOrderParam.getAccCode(), "账户编码不能为空!");
        Assert.notNull(adjustOrderParam.getTotalAmt(), "调整金额不能为空!");
        if (adjustOrderParam.getTotalAmt().compareTo(BigDecimal.ZERO) <= 0) {
            throw new BusinessException("调整金额不能小于或等于0");
        }
    }

    private void saveFlow(SaveFlowParam saveFlowParam) {
        //调整类型=调出的调整单提交
        //整车储值账户整车返利账户&& 调出时 调整类型=调出的调整单审核拒绝
        //调整类型=调出的调整单审核通过
        //调整类型=调进的调整单审核通过
        List<AccountFlowParam> param = new ArrayList<>();
        AccountFlowParam accountFlowParam = new AccountFlowParam();
        accountFlowParam.setAmount(saveFlowParam.getTotalAmt());
        accountFlowParam.setAccCode(saveFlowParam.getAccCode());
        accountFlowParam.setAccName(saveFlowParam.getAccName());
        accountFlowParam.setSourceNo(saveFlowParam.getDocNo());
        accountFlowParam.setSourceDoc(UdcEnum.FIN_SOURCE_DOC_TYPE_ADJUST.getValueCode());
        accountFlowParam.setAccType(saveFlowParam.getAccType());
        accountFlowParam.setTransactionTime(saveFlowParam.getTime());
        accountFlowParam.setSourcePlatform(UdcEnum.FIN_SOURCE_DOC_TYPE_ADJUST.getValueCode());
        String transactionType = "";
        String dataSource = "";
        //整车
        if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.SUBMIT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDZY.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD01.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.REJECT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDSF.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD02.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_ZCFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_1.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode();
            //配件账户
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.SUBMIT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDSF.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD01.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.REJECT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDSF.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD02.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJCZZH.getValueCode()) || saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_PJFLZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_1.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode();
        }
        //附件包
        else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_FJCZZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.SUBMIT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDZY.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD01.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_FJCZZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.REJECT)) {
            transactionType = UdcEnum.FIN_TRANSACTION_TYPE_TZDSF.getValueCode();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD02.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_FJCZZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_FJCZZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_1.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode();
        }
        //统筹
        else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_TCZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_1.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode();
        } else if ((saveFlowParam.getAccType().equals(UdcEnum.FIN_ACCOUNT_TYPE_TCZH.getValueCode())) &&
                saveFlowParam.getAdjType().equals(UdcEnum.ADJUST_TYPE_2.getValueCode()) && saveFlowParam.getScene().equals(FinConstant.APPROVED)) {
            transactionType = saveFlowParam.getAdjReason();
            dataSource = UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode();
        }
        accountFlowParam.setTransactionType(transactionType);
        accountFlowParam.setDataSource(dataSource);
        param.add(accountFlowParam);
        if (!StringUtils.isEmpty(transactionType) && !StringUtils.isEmpty(dataSource)) {
            accountFlowService.save(param);
        }
    }

    private SaveFlowParam VoToFlowParam(AdjustOrderVO adjustOrderVO, String sence) {
        SaveFlowParam saveFlowParam = new SaveFlowParam();
        saveFlowParam.setAdjType(adjustOrderVO.getAdjType());
        saveFlowParam.setAccType(adjustOrderVO.getAccType());
        saveFlowParam.setTotalAmt(adjustOrderVO.getTotalAmt());
        saveFlowParam.setAccCode(adjustOrderVO.getAccCode());
        saveFlowParam.setAccName(adjustOrderVO.getAccName());
        saveFlowParam.setDocNo(adjustOrderVO.getDocNo());
        saveFlowParam.setTime(LocalDateTime.now());
        saveFlowParam.setAdjReason(adjustOrderVO.getAdjReason());
        saveFlowParam.setScene(sence);
        return saveFlowParam;
    }

    private SaveFlowParam DoToFlowParam(AdjustOrderDO save, String sence) {
        SaveFlowParam saveFlowParam = new SaveFlowParam();
        saveFlowParam.setAdjType(save.getAdjType());
        saveFlowParam.setAccType(save.getAccType());
        saveFlowParam.setTotalAmt(save.getTotalAmt());
        saveFlowParam.setAccCode(save.getAccCode());
        saveFlowParam.setAccName(save.getAccName());
        saveFlowParam.setDocNo(save.getDocNo());
        saveFlowParam.setTime(LocalDateTime.now());
        saveFlowParam.setAdjReason(save.getAdjReason());
        saveFlowParam.setScene(sence);
        return saveFlowParam;
    }

    public void workFlow(AdjustOrderDO adjustOrderDO) {
        String docNo = adjustOrderDO.getDocNo();
        // 开启工作流
        /************************************************************************
         *                              工作流开始                                *
         ************************************************************************/
		log.info("工作流开始,{}",WorkflowConstant.CAN_START_PROC_STATUSES);
        if (adjustOrderDO.getWorkflowProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(adjustOrderDO.getWorkflowProcInstStatus())) {
            /**
             * 为什么要加上面3个判断，是因为流程已经启动了,就没必要再启动流程(场景:在'审批中'的过程中，做保存操作)
             * 工作流规则:
             *  1,一个业务单据在一个流程没结束前是不能再起的同一个工作流
             *  2,一个业务单据可以同时有2个"不同类型"的流程再跑
             */
            String procInstName = "账户余额调整单申请(" + docNo + ")审批";
            String businessKey = docNo + "#" + adjustOrderDO.getId();
            ProcessInfo processInfo = workflowRpcService.startProcess(
                    WorkFlowDefKey.FIN_ADJUST_ORDER.name(),
                    procInstName,
                    businessKey,
                    new HashMap<>());
            // 修改业务审批数据
            adjustOrderRepoProc.updateAdjustOrder(adjustOrderDO, processInfo);
            /************************************************************************
             *                              工作流结束                                *
             ************************************************************************/
        }
    }
}
