package com.elitesland.fin.application.service.accountingengine;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.accountingengine.InvSobAccountPeriodConvert;
import com.elitesland.fin.application.facade.dto.accountingengine.InvSobAccountPeriodDTO;
import com.elitesland.fin.application.facade.param.accountingengine.InvSobAccountPeriodParam;
import com.elitesland.fin.application.facade.vo.accountingengine.InvSobAccountPeriodVO;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.accountingengine.*;
import com.elitesland.fin.repo.accountingengine.*;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author gyj
 * @date 2023/10/10
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class InvSobAccountPeriodServiceImpl implements InvSobAccountPeriodService {


    private final InvSobAccountPeriodRepo invSobAccountPeriodRepo;

    private final InvSobAccountPeriodRepoProc invSobAccountPeriodRepoProc;

    private final FinSetOfBookRepoProc finSetOfBookRepoProc;

    private final FinAccountPeriodRepoProc finAccountPeriodRepoProc;

    private final FinAccountPeriodLineRepo finAccountPeriodLineRepo;

    private final FinSetOfBookOuRepo finSetOfBookOuRepo;


    @SysCodeProc
    @Override
    public PagingVO<InvSobAccountPeriodVO> page(InvSobAccountPeriodParam invSobAccountPeriodParam) {
        return InvSobAccountPeriodConvert.INSTANCE.DTOToVO(invSobAccountPeriodRepoProc.page(invSobAccountPeriodParam));
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void generate(InvSobAccountPeriodParam invSobAccountPeriodParam) {

        checkGenerateParam(invSobAccountPeriodParam);

        List<InvSobAccountPeriodDO> invSobAccountPeriodDOList = buildGenerateList(invSobAccountPeriodParam);

        List<InvSobAccountPeriodDTO> InvSobAccountPeriodDTOList = invSobAccountPeriodRepoProc.query(invSobAccountPeriodParam);

        //不重复创建
        invSobAccountPeriodDOList = invSobAccountPeriodDOList.stream().filter(InvSobAccountPeriodDO ->
                        !InvSobAccountPeriodDTOList.stream().filter(InvSobAccountPeriodDTO ->
                                        InvSobAccountPeriodDTO.getSobCode().equals(InvSobAccountPeriodDO.getSobCode()) &&
                                                InvSobAccountPeriodDTO.getAccountPeriodCode().equals(InvSobAccountPeriodDO.getAccountPeriodCode()) &&
                                                InvSobAccountPeriodDTO.getOuCode().equals(InvSobAccountPeriodDO.getOuCode()) &&
                                                InvSobAccountPeriodDTO.getActiveStartTime().equals(InvSobAccountPeriodDO.getActiveStartTime()) &&
                                                InvSobAccountPeriodDTO.getActiveEndTime().equals(InvSobAccountPeriodDO.getActiveEndTime()))
                                .findFirst()
                                .isPresent())
                .collect(Collectors.toList());
        invSobAccountPeriodDOList.stream().forEach(invSobAccountPeriodDO -> invSobAccountPeriodDO.setStatus(UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode()));
        invSobAccountPeriodRepo.saveAll(invSobAccountPeriodDOList);
    }

    private void checkGenerateParam(InvSobAccountPeriodParam invSobAccountPeriodParam) {
        Assert.notEmpty(invSobAccountPeriodParam.getSobCode(), "账套编码必填");
        Assert.notEmpty(invSobAccountPeriodParam.getYear(), "年份必填");
        Assert.notEmpty(invSobAccountPeriodParam.getOuCodeList(), "公司编码必填");
    }

    private List<InvSobAccountPeriodDO> buildGenerateList(InvSobAccountPeriodParam invSobAccountPeriodParam) {
        FinSetOfBookDO finSetOfBookDO = finSetOfBookRepoProc.findBySobCode(invSobAccountPeriodParam.getSobCode());
        Assert.notNull("账套不存在");

        FinAccountPeriodDO finAccountPeriodDO = finAccountPeriodRepoProc.findByAccountPeriodCode(finSetOfBookDO.getAccountPeriodCode());
        Assert.notNull("条会计期间不存在");

        List<FinAccountPeriodLineDO> finAccountPeriodLineDOList = finAccountPeriodLineRepo.findAllByMasIdIn(Lists.newArrayList(finAccountPeriodDO.getId()));

        List<FinSetOfBookOuDO> finSetOfBookOuDOList = finSetOfBookOuRepo.findAllByMasIdIn(Lists.newArrayList(finSetOfBookDO.getId()));

        List<String> ouCodeList = invSobAccountPeriodParam.getOuCodeList();

        List<InvSobAccountPeriodDO> invSobAccountPeriodDOList = ouCodeList.stream().flatMap(ouCode ->
                finAccountPeriodLineDOList.stream()
                        .filter(finAccountPeriodLineDO -> invSobAccountPeriodParam.getYear().equals(finAccountPeriodLineDO.getYear()))
                        .map(finAccountPeriodLineDO -> {
                            InvSobAccountPeriodDO invSobAccountPeriodDO = new InvSobAccountPeriodDO();
                            invSobAccountPeriodDO.setSobCode(finSetOfBookDO.getSobCode());
                            invSobAccountPeriodDO.setSobName(finSetOfBookDO.getSobName());
                            invSobAccountPeriodDO.setAccountPeriodCode(finAccountPeriodDO.getAccountPeriodCode());
                            invSobAccountPeriodDO.setAccountPeriodName(finAccountPeriodDO.getAccountPeriodName());
                            invSobAccountPeriodDO.setOuCode(ouCode);

                            FinSetOfBookOuDO finSetOfBookOuDOResult = finSetOfBookOuDOList.stream()
                                    .filter(finSetOfBookOuDO -> finSetOfBookOuDO.getOuCode().equals(ouCode))
                                    .findFirst()
                                    .orElse(null);
                            Assert.notNull(finSetOfBookOuDOResult, "公司不存在");

                            invSobAccountPeriodDO.setOuName(finSetOfBookOuDOResult.getOuName());
                            invSobAccountPeriodDO.setPeriodStyle(finAccountPeriodLineDO.getPeriodStyle());
                            invSobAccountPeriodDO.setStatus(UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_NOT_OPEN.getValueCode());
                            invSobAccountPeriodDO.setActiveStartTime(finAccountPeriodLineDO.getActiveStartTime());
                            invSobAccountPeriodDO.setActiveEndTime(finAccountPeriodLineDO.getActiveEndTime());

                            return invSobAccountPeriodDO;
                        })).collect(Collectors.toList());

        return invSobAccountPeriodDOList;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void openOrClose(InvSobAccountPeriodParam invSobAccountPeriodParam) {

        checkEnableOrDisableParam(invSobAccountPeriodParam);

        List<InvSobAccountPeriodDO> invSobAccountPeriodDOList = invSobAccountPeriodRepo.findAllById(invSobAccountPeriodParam.getIds());

        invSobAccountPeriodDOList.stream().forEach(item -> {
            if (StrUtil.equals(item.getStatus(), UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_CLOSED.getValueCode())
                    && StrUtil.equals(invSobAccountPeriodParam.getStatus(), UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode())) {
                throw new BusinessException("已关闭的不能再打开");
            }
            Assert.isFalse(invSobAccountPeriodParam.getStatus().equals(item.getStatus()), "已打开/关闭");
        });

        invSobAccountPeriodDOList.stream().forEach(item -> item.setStatus(invSobAccountPeriodParam.getStatus()));
    }

    private void checkEnableOrDisableParam(InvSobAccountPeriodParam invSobAccountPeriodParam) {
        Assert.notEmpty(invSobAccountPeriodParam.getIds(), "id必填");
        Assert.notEmpty(invSobAccountPeriodParam.getStatus(), "状态必填");
    }
}
