package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.fin.application.convert.creditaccount.CreditAccountInitialLimitConvert;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountInitialLimitDTO;
import com.elitesland.fin.application.facade.excel.creditaccount.InitialLimitImportEntity;
import com.elitesland.fin.application.facade.param.creditaccount.CreditAccountInitialLimitParam;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.creditaccount.CreditAccountInitialLimitDO;
import com.elitesland.fin.repo.creditaccount.CreditAccountInitialLimitRepo;
import com.elitesland.fin.repo.creditaccount.CreditAccountInitialLimitRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.repo.creditaccountflow.CreditAccountFlowRepo;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 期初余额导入
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class InitialLimitImportServiceImpl implements DataImport<InitialLimitImportEntity> {

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final CreditAccountRepoProc creditAccountRepoProc;

    private final CreditAccountInitialLimitRepo creditAccountInitialLimitRepo;

    private final CreditAccountInitialLimitRepoProc creditAccountInitialLimitRepoProc;

    private final CreditAccountFlowRepo creditAccountFlowRepo;
    private final FlexFieldUtilService flexFieldUtilService;

    @Override
    public String getTmplCode() {
        return "yst_fin_initial_limit_import";
    }

    @Override
    public List<String> executeImport(List<InitialLimitImportEntity> dataList, int rowId) {

        String userName = SecurityContextUtil.currentUserName();
        Long userId = SecurityContextUtil.currentUserId();
        if (CollectionUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        // 账户名称列表
        Set<String> creditAccountNameList = dataList.stream().map(InitialLimitImportEntity::getCreditAccountName).collect(Collectors.toSet());
        // 账户编码列表
        Set<String> creditAccountCodeList = dataList.stream().map(InitialLimitImportEntity::getCreditAccountCode).collect(Collectors.toSet());
        List<String> errorResult = new ArrayList<>();
        //
        beforeImportCheck(dataList, rowId, creditAccountNameList, creditAccountCodeList, errorResult);

        if (CollectionUtil.isNotEmpty(errorResult)) {
            return errorResult;
        }

        // 构建查询参数
        List<CreditAccountInitialLimitParam> params = new ArrayList<>();
        dataList.stream().forEach(item -> {
            CreditAccountInitialLimitParam param = new CreditAccountInitialLimitParam();
            param.setCreditAccountCode(item.getCreditAccountCode());
            param.setCreditAccountName(item.getCreditAccountName());
            params.add(param);
        });


        // 按照账户编码和账户名称查询信用账户
        List<CreditAccountInitialLimitDTO> limitDTOList = creditAccountRepoProc.findByInitialParam(params);
        List<String> codeList = new ArrayList<>();
        List<String> nameList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(limitDTOList)) {
            codeList = limitDTOList.stream().map(CreditAccountInitialLimitDTO::getCreditAccountCode).collect(Collectors.toList());
            nameList = limitDTOList.stream().map(CreditAccountInitialLimitDTO::getCreditAccountName).collect(Collectors.toList());
        }

        // 查询期初额度已导入的账户
        List<String> finishedCodeList = List.of();
        List<String> finishedNameList = List.of();
        Map<String, CreditAccountInitialLimitDTO> draftMap = Map.of();
        List<CreditAccountInitialLimitDTO> dtoList = creditAccountInitialLimitRepoProc.findByParam(params);
        if (CollectionUtil.isNotEmpty(dtoList)) {
            finishedCodeList = dtoList.stream().filter(dto -> StrUtil.equals(dto.getStatus(), UdcEnum.INITIAL_STATUS_FINISHED.getValueCode())).map(CreditAccountInitialLimitDTO::getCreditAccountCode).collect(Collectors.toList());
            finishedNameList = dtoList.stream().filter(dto -> StrUtil.equals(dto.getStatus(), UdcEnum.INITIAL_STATUS_FINISHED.getValueCode())).map(CreditAccountInitialLimitDTO::getCreditAccountName).collect(Collectors.toList());
            draftMap = dtoList.stream().filter(dto -> StrUtil.equals(dto.getStatus(), UdcEnum.INITIAL_STATUS_DRAFT.getValueCode())).collect(Collectors.toMap(item -> item.getCreditAccountCode() + "_" + item.getCreditAccountName(), t -> t, (t1, t2) -> t1));
        }

        // 储存新导入的数据
        List<InitialLimitImportEntity> saveList = new ArrayList<>();
        // 存放已经遍历过的数据
        List<String> codes = new ArrayList<>();
        List<String> names = new ArrayList<>();
        // 储存已经存在的数据
        for (InitialLimitImportEntity importEntity : dataList) {
            List<String> errorList = new ArrayList<>();

            // 校验账户
            checkAccount(rowId, errorList, codeList, nameList, finishedCodeList, finishedNameList, importEntity, codes, names);

            // 校验金额
            checkLimit(rowId, importEntity, errorList);

            // 导入已存在的草稿数据
            if (CollectionUtil.isNotEmpty(draftMap) && draftMap.containsKey(importEntity.getCreditAccountCode() + "_" + importEntity.getCreditAccountName())) {
                // 设置ID 做更新操作
                importEntity.setId(draftMap.get(importEntity.getCreditAccountCode() + "_" + importEntity.getCreditAccountName()).getId());
                importEntity.setCreator(userName);
                importEntity.setCreateUserId(userId);
                importEntity.setCreateTime(LocalDateTime.now());
            }

            // 如果该条数据存在错误 添加错误信息
            if (CollectionUtils.isNotEmpty(errorList)) {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            } else {
                // 不存在错误信息
                errorResult.add(null);
                importEntity.setStatus(UdcEnum.INITIAL_STATUS_DRAFT.getValueCode());
                // 草稿状态的进行更新操作
                saveList.add(importEntity);
            }
            // 行数自增
            rowId++;
        }

        if (CollectionUtils.isNotEmpty(saveList)) {
            saveData(saveList);
            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }
    }

    private void checkLimit(int rowId, InitialLimitImportEntity importEntity, List<String> errorList) {

        Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);

        if (NumberUtil.isNumber(importEntity.getInitialAccountLimit())) {
            if (!pattern.matcher(importEntity.getInitialAccountLimit()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户额度", "账户额度，最多两位小数"));
            }
            if (NumberUtil.isLess(new BigDecimal(importEntity.getInitialAccountLimit()), BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户额度", "账户额度，不能小于0"));
            }
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户额度", "账户额度，只能填纯数值类型数据"));
        }

        if (NumberUtil.isNumber(importEntity.getInitialAccountUsedLimit())) {
            if (!pattern.matcher(importEntity.getInitialAccountUsedLimit()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户使用额度", "账户使用额度，最多两位小数"));
            }
            if (NumberUtil.isLess(new BigDecimal(importEntity.getInitialAccountUsedLimit()), BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户使用额度", "账户使用额度，不能小于0"));
            }
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户使用额度", "账户使用额度，只能填纯数值类型数据"));
        }

        if (NumberUtil.isNumber(importEntity.getInitialAccountOccupancyLimit())) {
            if (!pattern.matcher(importEntity.getInitialAccountOccupancyLimit()).matches()) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户占用额度", "账户占用额度，最多两位小数"));
            }
            if (NumberUtil.isLess(new BigDecimal(importEntity.getInitialAccountOccupancyLimit()), BigDecimal.ZERO)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户占用额度", "账户占用额度，不能小于0"));
            }
        } else {
            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户占用额度", "账户占用额度，只能填纯数值类型数据"));
        }

    }

    private void checkAccount(int rowId, List<String> errorResult, List<String> codeList, List<String> nameList, List<String> finishedCodeList, List<String> finishedNameList, InitialLimitImportEntity importEntity, List<String> codes, List<String> names) {

        // 重复数据导入
        if (CollectionUtil.isNotEmpty(codes) && codes.contains(importEntity.getCreditAccountCode())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码重复"));
        }
        if (CollectionUtil.isNotEmpty(names) && names.contains(importEntity.getCreditAccountName())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称重复"));
        }

        // 为空校验
        if (StrUtil.isBlank(importEntity.getCreditAccountName())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称不能为空"));
        } else {
            names.add(importEntity.getCreditAccountName());
        }
        if (StrUtil.isBlank(importEntity.getCreditAccountCode())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码不能为空"));
        } else {
            codes.add(importEntity.getCreditAccountCode());
        }

        // 匹配信用账户信息
        if (CollectionUtil.isEmpty(nameList) || !nameList.contains(importEntity.getCreditAccountName())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "", "没有查询到对应账户"));
        } else if (CollectionUtil.isEmpty(codeList) || !codeList.contains(importEntity.getCreditAccountCode())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "", "没有查询到对应账户"));
        }

        // 匹配是否有已初始化的账户信息
        if (CollectionUtil.isNotEmpty(finishedNameList) && finishedNameList.contains(importEntity.getCreditAccountName())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户已经导入过额度"));
        }
        if (CollectionUtil.isNotEmpty(finishedCodeList) && finishedCodeList.contains(importEntity.getCreditAccountCode())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户已经导入过额度"));
        }

        if (creditAccountFlowRepo.existsByCreditAccountCode(importEntity.getCreditAccountCode())) {
            errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户已经生成过流水，不能导入"));
        }

    }

    private void beforeImportCheck(List<InitialLimitImportEntity> dataList, int rowId, Set<String> creditAccountNameList, Set<String> creditAccountCodeList, List<String> errorResult) {
        if (CollectionUtil.isEmpty(creditAccountNameList)) {
            for (int i = 0; i < dataList.size(); i++) {
                errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户名称", "账户名称不能为空"));
                rowId++;
            }
        }
        if (CollectionUtil.isEmpty(creditAccountCodeList)) {
            for (int i = 0; i < dataList.size(); i++) {
                errorResult.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "账户编码", "账户编码不能为空"));
                rowId++;
            }
        }
    }


    public void saveData(List<InitialLimitImportEntity> saveList) {

        List<CreditAccountInitialLimitDO> doList = CreditAccountInitialLimitConvert.INSTANCE.importEntitys2Dos(saveList);
        flexFieldUtilService.handFlexFieldValueFeferenceList(FinFlexFieldCodeConstant.CREDIT_ACCOUNT_INITIAL_LIMIT,doList);
        creditAccountInitialLimitRepo.saveAll(doList);

    }
}
