package com.elitesland.fin.application.service.paymentperiod;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.paymentperiod.AgingRangeConvert;
import com.elitesland.fin.application.convert.paymentperiod.AgingRangeOuConvert;
import com.elitesland.fin.application.facade.param.paymentperiod.*;
import com.elitesland.fin.application.facade.vo.paymentperiod.*;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.SysNumberGenerator;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.service.paymentperiod.AgingRangeDomainService;
import com.elitesland.fin.domain.service.paymentperiod.AgingRangeDtlDomainService;
import com.elitesland.fin.domain.service.paymentperiod.AgingRangeOuDomainService;
import com.elitesland.fin.infr.dto.paymentperiod.AgingRangeOuDTO;
import com.elitesland.fin.infr.dto.paymentperiod.AgingRangeOuHandleDTO;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeDtlRepo;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeDtlRepoProc;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeRepo;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/04/16
 * @Version 1.0
 * @Content:
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AgingRangeServiceImpl implements AgingRangeService {

    private final AgingRangeDomainService domainService;
    private final AgingRangeDtlDomainService dtlDomainService;
    private final AgingRangeRepo agingRangeRepo;
    private final AgingRangeRepoProc agingRangeRepoProc;
    private final AgingRangeDtlRepo agingRangeDtlRepo;
    private final AgingRangeDtlRepoProc agingRangeDtlRepoProc;
    private final SysNumberGenerator sysNumberGeneratorWrapper;
    private final UdcProvider udcProvider;
    private final AgingRangeOuDomainService agingRangeOuDomainService;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(AgingRangeSaveParam saveParam){
        List<AgingRangeDtlSaveParam> dtlSaveParamList = saveParam.getDtlSaveParamList();
        //校验必填项
        checkMandatoryField(saveParam);
        checkMandatoryFieldDtlList(dtlSaveParamList);

        //更新时校验ID是否正确
        AgingRangeVO AgingRangeVO = null;
        Boolean flag = true;
        if (Objects.nonNull(saveParam.getId())) {
            AgingRangeVO = domainService.findById(saveParam.getId());
            //修改时的业务校验逻辑
            checkUpdateBusiness(AgingRangeVO);
            flag = false;
        }

        //业务校验逻辑
        checkSave(saveParam, flag);
        checkSaveDtl(saveParam, dtlSaveParamList, flag);

        //新增/修改保存时默认值
        saveDefaultValue(flag, saveParam, AgingRangeVO);

        // ACTIVE_STATUS 启用停用
        //保存任务主表
        Long id = domainService.save(saveParam);
        saveParam.setId(id);

        //明细全删全插,先删除
        dtlDomainService.deleteByMasId(id);
        //明细保存时默认值
        saveDtlDefaultValue(dtlSaveParamList, saveParam);
        //保存明细
        dtlDomainService.saveDtl(dtlSaveParamList);

        return id;
    }

    /**
     * 新增/修改保存时默认值
     *
     * @param flag   新增/修改标识 true 新增 false 修改
     * @param saveVO 入参
     * @param respVO 数据库旧数据
     * @return
     */
    private void saveDefaultValue(Boolean flag, AgingRangeSaveParam saveVO, AgingRangeVO respVO) {

        if (flag) {
            //新增时-发号器新增编码
            //saveVO.setAgingRangeCode(getProtocolCode());
        } else {
            //修改 协议类型和协议编码不允许编辑
            if (StringUtils.isBlank(saveVO.getAgingRangeCode())) {
                saveVO.setAgingRangeCode(respVO.getAgingRangeCode());
            }

        }
       /* if (Objects.isNull(saveVO.getDeleteFlag())) {
            saveVO.setDeleteFlag(0);
        }*/
        //状态:默认为启用
        if (StringUtils.isBlank(saveVO.getStatus())) {
            saveVO.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        }


    }

    /**
     * 明细新增/修改时默认值
     *
     *  如果账期开始=0，账期总数=账期结束-账期开始；
     *  如果账期开始！=0，账期总数=账期结束-账期开始+1
     *  如果账期结束没值，账期总是也没值
     *
     * @param dtlSaveVOList 明细入参
     * @param saveVO        主表信息
     * @return
     */
    private void saveDtlDefaultValue(List<AgingRangeDtlSaveParam> dtlSaveVOList, AgingRangeSaveParam saveVO) {

        /*List<AgingRangeDtlSaveParam> sortedDtlList = dtlSaveVOList.stream().sorted(
                Comparator.comparing(AgingRangeDtlSaveParam::getLineNo).reversed()).collect(Collectors.toList());*/

        dtlSaveVOList.forEach(dtlSaveVO -> {
            dtlSaveVO.setMasId(saveVO.getId());

            if (Objects.nonNull(dtlSaveVO.getPaymentPeriodEnd())){
                if (dtlSaveVO.getPaymentPeriodBegin() == 0){
                    Integer differenceValue = dtlSaveVO.getPaymentPeriodEnd() - dtlSaveVO.getPaymentPeriodBegin();
                    dtlSaveVO.setTotalPeriod(differenceValue);

                } else {
                    Integer differenceValue = dtlSaveVO.getPaymentPeriodEnd() - dtlSaveVO.getPaymentPeriodBegin() + 1;
                    dtlSaveVO.setTotalPeriod(differenceValue);
                }
            }

        });
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long update(AgingRangeSaveParam saveParam){

        return null;
    }



    /**
     * 账龄区间主表的业务校验逻辑
     *
     * @param saveVO 入参
     * @param flag   新增/修改标识 true 新增 false 修改
     * @return
     */
    private void checkSave(AgingRangeSaveParam saveVO, Boolean flag) {
        AgingRangePageParam pageParam1 = new AgingRangePageParam();
        pageParam1.setAgingRangeCode(saveVO.getAgingRangeCode());
        //pageParam1.setProtocolType(saveVO.getProtocolType());
        AgingRangeVO agreementVO1 = domainService.checkSelectByParam(pageParam1);
        if (flag) {
            if (Objects.nonNull(agreementVO1) && Objects.equals(agreementVO1.getAgingRangeCode(), saveVO.getAgingRangeCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "编码已存在!");
            }
        } else {
            if (Objects.nonNull(agreementVO1) && Objects.equals(agreementVO1.getAgingRangeCode(), saveVO.getAgingRangeCode())
                    && !Objects.equals(saveVO.getId(), agreementVO1.getId())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "编码已存在!");
            }
        }

        AgingRangePageParam pageParam2 = new AgingRangePageParam();
        pageParam2.setAgingRangeName(saveVO.getAgingRangeName());
        //pageParam2.setProtocolType(saveVO.getProtocolType());
        AgingRangeVO agreementVO2 = domainService.checkSelectByParam(pageParam2);
        if (flag) {
            if (Objects.nonNull(agreementVO2) && Objects.equals(agreementVO2.getAgingRangeName(), saveVO.getAgingRangeName())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "名称已存在!");
            }
        } else {
            if (Objects.nonNull(agreementVO2) && Objects.equals(agreementVO2.getAgingRangeName(), saveVO.getAgingRangeName())
                    && !Objects.equals(saveVO.getId(), agreementVO2.getId())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "名称已存在!");
            }
        }
    }

    /**
     * 账龄区间明细的业务校验逻辑
     *
     * 校验：1.必填项。
     *      2.前一行的账期结束和后一行的账期开始必须连续，不能相同，不能跳。
     *      3.最后一行账期结束日无需填写
     *
     *      4.新增一行明细，行号1时，账期开始必须是0
     *
     *      5.如果账期开始=0，账期总数=账期结束-账期开始；
     *      如果账期开始！=0，账期总数=账期结束-账期开始+1
     *      如果账期结束没值，账期总是也没值
     *
     * @param saveVO        主表入参
     * @param dtlSaveVOList 明细入参
     * @param flag          新增/修改标识 true 新增 false 修改
     * @return
     */
    private void checkSaveDtl(AgingRangeSaveParam saveVO, List<AgingRangeDtlSaveParam> dtlSaveVOList, Boolean flag) {

        if (dtlSaveVOList.size() > 1){
            //倒叙处理
            List<AgingRangeDtlSaveParam> sortedDtlList = dtlSaveVOList.stream().sorted(
                    Comparator.comparing(AgingRangeDtlSaveParam::getLineNo).reversed()).collect(Collectors.toList());

            for (int i = 0; i < sortedDtlList.size(); i++) {
                AgingRangeDtlSaveParam currentSaveParam = sortedDtlList.get(i);


                //当前是最后一条时
                if(i == (sortedDtlList.size() - 1)){
                    if (currentSaveParam.getPaymentPeriodBegin() != 0){
                        throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第"+currentSaveParam.getLineNo()+"行的行号为1时,账期开始必须是0");
                    }
                }else {
                    AgingRangeDtlSaveParam previousSaveParam = sortedDtlList.get(i+1);
                    if (Objects.isNull(previousSaveParam.getPaymentPeriodEnd())){
                        throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第"+previousSaveParam.getLineNo()+"行的账期结束为空");
                    }
                    Integer differenceValue = currentSaveParam.getPaymentPeriodBegin() - previousSaveParam.getPaymentPeriodEnd();
                    if (differenceValue != 1){
                        throw new BusinessException(ApiCode.BUSINESS_EXCEPTION,
                                "第"+previousSaveParam.getLineNo()+"行的账期结束与第"+currentSaveParam.getLineNo()+"行的账期开始必须连续、不能相同");
                    }
                    if (previousSaveParam.getLineNo() == 1){
                        if (previousSaveParam.getPaymentPeriodBegin() != 0){
                            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "第"+previousSaveParam.getLineNo()+"行的行号为1时,账期开始必须是0");
                        }
                    }
                }

            }

        }else if(dtlSaveVOList.size() == 1) {
            AgingRangeDtlSaveParam agingRangeDtlSaveParam = dtlSaveVOList.get(0);
            if (agingRangeDtlSaveParam.getPaymentPeriodBegin() != 0){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "行号1时,账期开始必须是0");
            }

        }


    }





    /**
     * 修改时的业务校验逻辑
     *
     * @param respVO 数据库旧数据信息
     * @return
     */
    private void checkUpdateBusiness(AgingRangeVO respVO) {
        //启用状态，支持编辑修改相关信息。
        if (!Objects.equals(respVO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只有禁用状态的才可编辑修改!");
        }

    }


    /**
     * 校验必填项:收付款主表
     *
     * @param saveVO 入参
     */
    private void checkMandatoryField(AgingRangeSaveParam saveVO) {
        if (StringUtils.isBlank(saveVO.getAgingRangeCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账龄区间编码为空!");
        }
        if (StringUtils.isBlank(saveVO.getAgingRangeName())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账龄区间名称为空!");
        }
        if (StringUtils.isBlank(saveVO.getTimeUnitType())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "时间单位为空!");
        }
        if (StringUtils.isBlank(saveVO.getUnitDays())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "单位天数为空!");
        }
    }

    /**
     * 校验必填项:收付款明细
     *
     * @param saveVOList 入参
     */
    private void checkMandatoryFieldDtlList(List<AgingRangeDtlSaveParam> saveVOList) {
        /*if (CollectionUtils.isEmpty(saveVOList)) {
            throw new BusinessException(ApiCode.VALIDATE_FAILED, "明细数据为空!");
        }*/

        if (!CollectionUtils.isEmpty(saveVOList)) {
            saveVOList.forEach(saveVO -> {
                checkMandatoryFieldDtl(saveVO);
            });
        }

    }

    /**
     * 校验必填项:收付款明细
     *
     * @param saveVO 入参
     */
    private void checkMandatoryFieldDtl(AgingRangeDtlSaveParam saveVO) {
        if (Objects.isNull(saveVO.getLineNo())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "行号为空!");
        }
        if (StringUtils.isBlank(saveVO.getPaymentPeriodDesc())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期描述为空!");
        }
        if (Objects.isNull(saveVO.getPaymentPeriodBegin())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账期开始为空!");
        }

    }

    /**
     * 根据发号器规则自动生成协议编码
     *
     * @param
     * @return 协议编码
     */
    private String getProtocolCode() {
        String code = sysNumberGeneratorWrapper.generate(FinConstant.AGREEMENT_PROTOCOL_CODE);
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "生成账龄区间编码失败!");
        }
        return code;
    }



    /**
     * 账龄区间分页查询
     *
     * @param pageParam 入参
     * @return 账龄区间信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AgingRangePagingVO> page(AgingRangePageParam pageParam) {

        PagingVO<AgingRangePagingVO> pagingVO = domainService.page(pageParam);

        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AgingRangePagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        List<AgingRangePagingVO> pagingVOList = pagingVO.getRecords();


        Map<String, String> yearTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getModel(), UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getCode());
        Map<String, String> monthTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getModel(), UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getCode());
        Map<String, String> dayTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getModel(), UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getCode());

        pagingVOList.forEach(agingRangePagingVO -> {
            String unitDays = agingRangePagingVO.getUnitDays();
            String timeUnitType = agingRangePagingVO.getTimeUnitType();

            if (StringUtils.isNotBlank(unitDays)){
                if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_YEAR.getValueCode()) && MapUtil.isNotEmpty(yearTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(yearTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_MONTH.getValueCode()) && MapUtil.isNotEmpty(monthTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(monthTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_DAY.getValueCode()) && MapUtil.isNotEmpty(dayTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(dayTypeUnitDaysUdc.get(unitDays));
                }
            }
        });

        return PagingVO.<AgingRangePagingVO>builder()
                .total(pagingVO.getTotal())
                .records(pagingVOList)
                .build();
    }

    /**
     * 根据账龄区间ID查询账龄区间详情数据
     *
     * @param id 账龄区间ID
     * @return 账龄区间详情数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO findIdOne(Long id) {
        AgingRangeVO agreementVO = domainService.selectById(id);
        translateAgingRange(Collections.singletonList(agreementVO));

        List<AgingRangeDtlVO> agreementDtlVOList = dtlDomainService.selectByMasId(agreementVO.getId());

        if (CollectionUtils.isEmpty(agreementDtlVOList)) {
            return agreementVO;
        }
        translateDtl(agreementDtlVOList);

        agreementVO.setDtlVOList(agreementDtlVOList);

        return agreementVO;
    }

    /**
     * 根据账龄区间编码查询账龄区间详情数据
     *
     * @param code 账龄区间编码
     * @return 账龄区间详情数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO findCodeOne(String code) {
        AgingRangeVO agreementVO = domainService.selectByCode(code);
        translateAgingRange(Collections.singletonList(agreementVO));

        List<AgingRangeDtlVO> agreementDtlVOList = dtlDomainService.selectByMasId(agreementVO.getId());

        if (CollectionUtils.isEmpty(agreementDtlVOList)) {
            return agreementVO;
        }
        translateDtl(agreementDtlVOList);

        agreementVO.setDtlVOList(agreementDtlVOList);

        return agreementVO;
    }


    /**
     * 组装填充账龄区间的相关关联字段信息
     *
     * @param respVOList 账龄区间数据信息
     * @return
     */
    private void translateAgingRange(List<AgingRangeVO> respVOList) {
        Map<String, String> yearTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getModel(), UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getCode());
        Map<String, String> monthTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getModel(), UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getCode());
        Map<String, String> dayTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getModel(), UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getCode());

        respVOList.forEach(agingRangeVO -> {
            String unitDays = agingRangeVO.getUnitDays();
            String timeUnitType = agingRangeVO.getTimeUnitType();

            if (StringUtils.isNotBlank(unitDays)){
                if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_YEAR.getValueCode()) && MapUtil.isNotEmpty(yearTypeUnitDaysUdc)){
                    agingRangeVO.setUnitDaysName(yearTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_MONTH.getValueCode()) && MapUtil.isNotEmpty(monthTypeUnitDaysUdc)){
                    agingRangeVO.setUnitDaysName(monthTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_DAY.getValueCode()) && MapUtil.isNotEmpty(dayTypeUnitDaysUdc)){
                    agingRangeVO.setUnitDaysName(dayTypeUnitDaysUdc.get(unitDays));
                }
            }
        });
    }

    /**
     * 组装填充账龄区间明细的相关关联字段信息
     *
     * @param respVOList 账龄区间数据信息
     * @return
     */
    private void translateDtl(List<AgingRangeDtlVO> respVOList) {

    }

    /**
     * 根据账龄区间ID批量删除
     *
     * @param ids 账龄区间ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteBatch(List<Long> ids) {

        AgingRangePageParam queryVO = new AgingRangePageParam();
        queryVO.setIdList(ids);
        List<AgingRangeVO> respVOList = domainService.selectByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }

        List<AgingRangeVO> checkRespVOList = respVOList.stream().filter(agingRangeVO -> !Objects.equals(agingRangeVO.getStatus(), UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(checkRespVOList)) {
            String checkItemResult = checkRespVOList.stream().map(checkVO ->
                    checkVO.getAgingRangeCode()
            ).collect(Collectors.joining(";", "只有禁用状态的才可删除.[", "],请检查!"));
            throw new BusinessException(ApiCode.FAIL, checkItemResult);
        }

        List<Long> idList = respVOList.stream().map(AgingRangeVO::getId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        //逻辑删除主表
        domainService.updateDeleteFlagBatch(1, idList);
        //逻辑删除明细
        dtlDomainService.updateDeleteFlagBatch(1, idList);
        //逻辑删除分配公司
        agingRangeOuDomainService.updateDeleteFlagBatchByMasId(1, idList);
    }


    /**
     * 根据账龄区间ID批量启用
     *
     * @param ids 账龄区间ID集合
     * @return
     */
    @Override
    public void enableBatch(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择需要启用的数据!");
        }
        AgingRangePageParam checkEqualParam = new AgingRangePageParam();
        checkEqualParam.setIdList(ids);
        checkEqualParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        List<AgingRangeVO> checkEqualList = domainService.selectByParam(checkEqualParam);
        if (!CollectionUtils.isEmpty(checkEqualList)) {
            String checkItemResult = checkEqualList.stream().map(checkVO ->
                    checkVO.getAgingRangeCode()
            ).collect(Collectors.joining(";", "已启用的不可重复启用.[", "],请检查!"));
            throw new BusinessException(ApiCode.FAIL, checkItemResult);
        }

        domainService.updateStatusBatch(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode(), ids);

    }


    /**
     * 根据账龄区间ID批量停用
     *
     * @param ids 账龄区间ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void disableBatch(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择需要禁用的数据!");
        }
        AgingRangePageParam checkEqualParam = new AgingRangePageParam();
        checkEqualParam.setIdList(ids);
        checkEqualParam.setStatus(UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode());
        List<AgingRangeVO> checkEqualList = domainService.selectByParam(checkEqualParam);
        if (!CollectionUtils.isEmpty(checkEqualList)) {
            String checkItemResult = checkEqualList.stream().map(checkVO ->
                    checkVO.getAgingRangeCode()
            ).collect(Collectors.joining(";", "已禁用的不可重复禁用.[", "],请检查!"));
            throw new BusinessException(ApiCode.FAIL, checkItemResult);
        }
        domainService.updateStatusBatch(UdcEnum.FIN_ACTIVE_STATUS_CLOSED.getValueCode(), ids);
    }


    @Override
    @SysCodeProc
    public PagingVO<AgingRangeComPagingVO> commonPage(AgingRangeComPageParam comPageParam){
        AgingRangePageParam pageParam = AgingRangeConvert.INSTANCE.comToPageParam(comPageParam);

        PagingVO<AgingRangePagingVO> pagingVO = domainService.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AgingRangeComPagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        List<AgingRangeComPagingVO> comPagingVOList = pagingVO.getRecords().stream().map(AgingRangeConvert.INSTANCE::voToComPagingVo).collect(Collectors.toList());
        Map<String, String> yearTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getModel(), UdcEnum.TIME_UNIT_DAYS_YEAR_TYPE_360.getCode());
        Map<String, String> monthTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getModel(), UdcEnum.TIME_UNIT_DAYS_MONTH_TYPE_30.getCode());
        Map<String, String> dayTypeUnitDaysUdc = udcProvider.getValueMapByUdcCode(UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getModel(), UdcEnum.TIME_UNIT_DAYS_DAY_TYPE_1.getCode());

        comPagingVOList.forEach(agingRangePagingVO -> {
            String unitDays = agingRangePagingVO.getUnitDays();
            String timeUnitType = agingRangePagingVO.getTimeUnitType();

            if (StringUtils.isNotBlank(unitDays)){
                if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_YEAR.getValueCode()) && MapUtil.isNotEmpty(yearTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(yearTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_MONTH.getValueCode()) && MapUtil.isNotEmpty(monthTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(monthTypeUnitDaysUdc.get(unitDays));
                }else if (Objects.equals(timeUnitType,UdcEnum.AGING_RANGE_TIME_UNIT_TYPE_DAY.getValueCode()) && MapUtil.isNotEmpty(dayTypeUnitDaysUdc)){
                    agingRangePagingVO.setUnitDaysName(dayTypeUnitDaysUdc.get(unitDays));
                }
            }
        });

        return PagingVO.<AgingRangeComPagingVO>builder()
                .total(pagingVO.getTotal())
                .records(comPagingVOList)
                .build();
    }



    /**
     * 分页查询账龄区间的分配公司
     *
     * @param pageParam 入参
     * @return 账龄区间的分配公司集合
     */
    @SysCodeProc
    @Override
    public PagingVO<AgingRangeOuVO> ouPage(AgingRangeOuPageParam pageParam){

        PagingVO<AgingRangeOuDTO> pagingVO = agingRangeOuDomainService.ouPage(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AgingRangeOuVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }


        List<AgingRangeOuVO> agingRangeOuVOList = pagingVO.getRecords().stream().map(AgingRangeOuConvert.INSTANCE::dtoToVo).collect(Collectors.toList());


        return PagingVO.<AgingRangeOuVO>builder()
                .total(pagingVO.getTotal())
                .records(agingRangeOuVOList)
                .build();
    }

    /**
     * 账龄区间添加分配公司
     *
     * 公司弹框，不支持账龄批量选中分配公司，如果之前已经选中了该公司，再次选择过滤选择过的公司。已分配过账龄的公司不允许重复匹配
     *
     * @param saveParam 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void addOu(AgingRangeOuMainSaveParam saveParam){
        if (Objects.isNull(saveParam.getMasId())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "主表ID为空!");
        }
        if (CollectionUtil.isEmpty(saveParam.getOuSaveParamList())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分配公司数据为空!");
        }
        Long masId = saveParam.getMasId();
        List<AgingRangeOuSaveParam> ouSaveParamList = saveParam.getOuSaveParamList();
        List<Long> ouIdList = ouSaveParamList.stream().map(AgingRangeOuSaveParam::getOuId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtil.isEmpty(ouIdList)){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "分配公司的ID为空!");
        }
        AgingRangeOuPageParam agingRangeOuPageParam = new AgingRangeOuPageParam();
        //agingRangeOuPageParam.setMasId(masId);
        agingRangeOuPageParam.setOuIdList(ouIdList);
        List<AgingRangeOuVO> oldAgingRangeOuVOList = agingRangeOuDomainService.selectByParam(agingRangeOuPageParam);


        //已分配过账龄的公司不允许重复匹配
        if (CollectionUtil.isNotEmpty(oldAgingRangeOuVOList)){
            List<AgingRangeOuVO> checkAgingRangeOuVOList = oldAgingRangeOuVOList.stream().filter(agingRangeOuVO -> !Objects.equals(agingRangeOuVO.getMasId(),masId)).distinct().collect(Collectors.toList());

            if (CollectionUtil.isNotEmpty(checkAgingRangeOuVOList)){
                List<Long> checkMasIdList = checkAgingRangeOuVOList.stream().map(AgingRangeOuVO::getMasId).distinct().collect(Collectors.toList());
                AgingRangePageParam agingRangePageParam = new AgingRangePageParam();
                agingRangePageParam.setIdList(checkMasIdList);
                List<AgingRangeVO> checkAgingRangeVOList = domainService.selectByParam(agingRangePageParam);
                //Map<Long, List<AgingRangeVO>> apOrderToPayMap = checkAgingRangeVOList.stream().collect(Collectors.groupingBy(AgingRangeVO::getId));
                Map<Long, AgingRangeVO> checkAgingRangeMap = checkAgingRangeVOList.stream().collect(Collectors.toMap(AgingRangeVO::getId, v -> v, (v1, v2) -> v2));

                List<AgingRangeOuHandleDTO> checkAgingRangeOuHandleList = checkAgingRangeOuVOList.stream().map(agingRangeOuVO -> {
                    AgingRangeOuHandleDTO agingRangeOuHandleDTO = new AgingRangeOuHandleDTO();
                    agingRangeOuHandleDTO.setAgingRangeOuId(agingRangeOuVO.getId());
                    agingRangeOuHandleDTO.setOuId(agingRangeOuVO.getOuId());
                    agingRangeOuHandleDTO.setOuCode(agingRangeOuVO.getOuCode());
                    agingRangeOuHandleDTO.setOuName(agingRangeOuVO.getOuName());
                    AgingRangeVO agingRangeVO = checkAgingRangeMap.get(agingRangeOuVO.getMasId());
                    if (Objects.isNull(agingRangeVO)){
                        throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该分配公司数据("+agingRangeOuVO.getId()+")的主表不存在!");
                    }
                    agingRangeOuHandleDTO.setAgingRangeId(agingRangeVO.getId());
                    agingRangeOuHandleDTO.setAgingRangeCode(agingRangeVO.getAgingRangeCode());
                    agingRangeOuHandleDTO.setAgingRangeName(agingRangeVO.getAgingRangeName());
                    return agingRangeOuHandleDTO;
                }).collect(Collectors.toList());

                String checkResult = checkAgingRangeOuHandleList.stream().map(checkVO ->
                        "公司(" + checkVO.getOuCode() + ")已分配过账龄(" + checkVO.getAgingRangeCode() +")"
                ).collect(Collectors.joining(";", "不允许重复匹配.[", "],请检查!"));
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, checkResult);

            }
        }

        List<AgingRangeOuSaveParam> newOuSaveParamList = new ArrayList<>();
        for (AgingRangeOuSaveParam rangeOuSaveParam : ouSaveParamList) {
            rangeOuSaveParam.setMasId(masId);
            Boolean checkExistFlag = oldAgingRangeOuVOList.stream().anyMatch(agingRangeOuVO ->
                    Objects.equals(rangeOuSaveParam.getMasId(),agingRangeOuVO.getMasId()) &&
                    Objects.equals(agingRangeOuVO.getOuId(),rangeOuSaveParam.getOuId()));
            if (!checkExistFlag){
                newOuSaveParamList.add(rangeOuSaveParam);
            }

        }

        if (CollectionUtil.isNotEmpty(newOuSaveParamList)){
            agingRangeOuDomainService.save(newOuSaveParamList);
        }

    }

    /**
     * 账龄区间取消分配公司
     *
     * 公司弹框，不支持账龄批量选中取消分配。展示分配过的公司，取消勾选点击确定后取消分配，
     * @param ids 入参
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void cancelOu(List<Long> ids){
        agingRangeOuDomainService.cancelOu(ids);
    }

}
