package com.elitesland.fin.application.service.report;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.aporder.ApOrderDtlConvert;
import com.elitesland.fin.application.facade.vo.report.IpvRespVO;
import com.elitesland.fin.domain.entity.report.IpvInvoiceDO;
import com.elitesland.fin.domain.param.report.IpvInvoiceComputeParam;
import com.elitesland.fin.domain.param.report.IpvInvoicePageParam;
import com.elitesland.fin.domain.param.report.IpvInvoiceParam;
import com.elitesland.fin.domain.service.report.IpvCommonDomainService;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.dto.aporder.ApOrderDtlDTO;
import com.elitesland.fin.infr.repo.aporder.ApOrderDtlRepo;
import com.elitesland.fin.infr.repo.aporder.ApOrderDtlRepoProc;
import com.elitesland.fin.infr.repo.aporder.ApOrderRepoProc;
import com.elitesland.fin.infr.repo.report.IpvInvoiceRepoProc;
import com.elitesland.fin.repo.accountingengine.InvSobAccountPeriodRepoProc;
import com.elitesland.fin.rpc.inv.RmiInvRpcService;
import com.elitesland.inv.dto.invIo.InvIoParamRpcDTO;
import com.elitesland.inv.dto.invIo.InvIoRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import javax.persistence.EntityManager;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class IpvCommonServiceImpl implements IpvCommonService {

    private final IpvCommonDomainService ipvCommonDomainService;
    private final InvSobAccountPeriodRepoProc invSobAccountPeriodRepoProc;
    private final ApOrderDtlRepoProc apOrderDtlRepoProc;
    private final ApOrderRepoProc apOrderRepoProc;
    private final ApOrderDtlRepo apOrderDtlRepo;
    private final IpvInvoiceRepoProc ipvInvoiceRepoProc;
    private final RmiInvRpcService rmiInvRpcService;
    private final TaskExecutor taskExecutor;
    private final RedisTemplate redisTemplate;
    private final EntityManager entityManager;
    private final TransactionTemplate transactionTemplate;

    @Override
    public PagingVO<IpvRespVO> page(IpvInvoicePageParam paramVO) {
        PagingVO<IpvRespVO> page = ipvCommonDomainService.page(paramVO);
        var sumRespVO = ipvCommonDomainService.sumIpv(paramVO);
        return page.setAggregatedData(BeanUtil.beanToMap(sumRespVO));
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void compute(IpvInvoiceComputeParam param) {
        long count = invSobAccountPeriodRepoProc.countAccountPeriodByParam(param.getBuDate(), param.getOuCode());
        if (count > 0) {
            throw new BusinessException("存在未关闭的库存会计期间,请检查！");
        }
        String buDateFromStr = "2024-10-01 00:00:00";
        LocalDateTime buDateFrom = LocalDateTimeUtil.parse(buDateFromStr, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        String buDateToStr = param.getBuDate() + "-31 23:59:59";
        LocalDateTime buDateTo = LocalDateTimeUtil.parse(buDateToStr, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        //查询满足条件的应付单数据
        var unComputeArOrderDtlList = apOrderDtlRepo.findUnComputeArOrderDtl(buDateFrom, buDateTo, param.getOuCode()).stream()
                .map(ApOrderDtlConvert.INSTANCE::doToDto).collect(Collectors.toList());
        if (CollUtil.isEmpty(unComputeArOrderDtlList)) {
            throw new BusinessException("没有找到满足条件的应付单数据");
        }
        String redisKey = "IPV-COMPUTE@".concat(param.getBuDate()).concat(param.getOuCode());
        if (redisTemplate.hasKey(redisKey)) {
            throw new BusinessException("正在计算中，请耐心等待！");
        }
        // 设置防重复操作限时标记(前置通知)
        redisTemplate.opsForValue().set(redisKey, "PV-COMPUTE", 300, TimeUnit.SECONDS);
        try {
            //异步
            CompletableFuture.runAsync(() -> {
                computeAndSaveIpvInvoice(unComputeArOrderDtlList, buDateFrom, buDateTo, redisKey, param.getBuDate());
            }, taskExecutor);
        } catch (Throwable throwable) {
            log.info("IPV账单:{},计算出错:{}", param.getBuDate(), throwable);
            redisTemplate.delete(redisKey);
        }
    }

    private void computeAndSaveIpvInvoice(List<ApOrderDtlDTO> unComputeArOrderDtlList, LocalDateTime buDateFrom, LocalDateTime buDateTo,
                                          String redisKey, String accountPeriod) {
        List<Long> masIds = unComputeArOrderDtlList.stream().map(ApOrderDtlDTO::getMasId).distinct().collect(Collectors.toList());
        List<ApOrderDTO> apOrderDTOS = apOrderRepoProc.queryByIds(masIds);
        Map<Long, ApOrderDTO> apOrderMap = apOrderDTOS.stream().collect(Collectors.toMap(ApOrderDTO::getId, Function.identity()));

        // 批量查询 invIoRpcDTO
        Map<String, List<InvIoRpcDTO>> invIoBySourceNoMap = getInvIoBySourceNoMap(unComputeArOrderDtlList);
        Map<String, List<InvIoRpcDTO>> invIoByIdMap = getInvIoByIdMap(unComputeArOrderDtlList);
        List<IpvInvoiceDO> ipvInvoiceDOS = new ArrayList<>();
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionTemplate.execute(transactionStatus -> {
            try {
                for (ApOrderDtlDTO apOrderDtlDTO : unComputeArOrderDtlList) {
                    String lotNo = "";
                    IpvInvoiceDO ipvInvoiceDO = new IpvInvoiceDO();
                    ApOrderDTO apOrderDTO = apOrderMap.get(apOrderDtlDTO.getMasId());
                    if (apOrderDTO == null) {
                        log.error("未查询到应收单：{}", apOrderDtlDTO.getMasId());
                        continue;
                    }
                    if (apOrderDtlDTO.getInvIoId() == null) {
                        List<InvIoRpcDTO> invIoRpcDTOs = invIoBySourceNoMap.get(apOrderDtlDTO.getSourceNo() + "@" + apOrderDtlDTO.getSourceNoDid());
                        if (CollUtil.isEmpty(invIoRpcDTOs)) {
                            log.info("未查询到库存流水信息[1],应收单行ID:{}", apOrderDtlDTO.getId());
                            continue;
                        }
                        InvIoRpcDTO invIoRpcDTO = invIoRpcDTOs.get(0);
                        ipvInvoiceDO.setWhId(invIoRpcDTO.getWhId());
                        ipvInvoiceDO.setWhCode(invIoRpcDTO.getWhCode());
                        ipvInvoiceDO.setWhName(invIoRpcDTO.getWhName());
                        lotNo = invIoRpcDTO.getLotNo();
                    } else {
                        List<InvIoRpcDTO> invIoRpcDTOS = invIoByIdMap.get(apOrderDtlDTO.getSourceNo() + "@" + apOrderDtlDTO.getSourceNoDid());
                        if (CollUtil.isEmpty(invIoRpcDTOS)) {
                            log.info("未查询到库存流水信息[2],应收单行ID:{}", apOrderDtlDTO.getId());
                            continue;
                        }
                        InvIoRpcDTO invIoRpcDTO = invIoRpcDTOS.get(0);
                        ipvInvoiceDO.setWhId(invIoRpcDTO.getWhId());
                        ipvInvoiceDO.setWhCode(invIoRpcDTO.getWhCode());
                        ipvInvoiceDO.setWhName(invIoRpcDTO.getWhName());
                        lotNo = invIoRpcDTO.getLotNo();
                    }
                    ipvInvoiceDO.setApOrderNo(apOrderDTO.getApOrderNo());
                    ipvInvoiceDO.setApOrderDid(apOrderDtlDTO.getId());
                    ipvInvoiceDO.setSourceDocNo(apOrderDtlDTO.getSourceNo());
                    ipvInvoiceDO.setSourceLineNo(apOrderDtlDTO.getSourceLine());
                    ipvInvoiceDO.setItemId(apOrderDtlDTO.getItemId());
                    ipvInvoiceDO.setItemCode(apOrderDtlDTO.getItemCode());
                    ipvInvoiceDO.setItemName(apOrderDtlDTO.getItemName());
                    ipvInvoiceDO.setItemType(apOrderDtlDTO.getItemType());
                    ipvInvoiceDO.setOuId(apOrderDTO.getOuId());
                    ipvInvoiceDO.setOuCode(apOrderDTO.getOuCode());
                    ipvInvoiceDO.setOuName(apOrderDTO.getOuName());
                    ipvInvoiceDO.setSuppId(apOrderDTO.getSuppId());
                    ipvInvoiceDO.setSuppCode(apOrderDTO.getSuppCode());
                    ipvInvoiceDO.setSuppName(apOrderDTO.getSuppName());
                    ipvInvoiceDO.setSmallCateCode(apOrderDtlDTO.getSmallCateCode());
                    ipvInvoiceDO.setSmallCateName(apOrderDtlDTO.getSmallCateName());
                    BigDecimal ipv = apOrderDtlDTO.getInvoicePriceVariance();
                    //记账日期
                    ipvInvoiceDO.setFinDate(apOrderDTO.getBuDate());
                    ipvInvoiceDO.setApQty(apOrderDtlDTO.getQty());
                    ipvInvoiceDO.setUom(apOrderDtlDTO.getUom());
                    ipvInvoiceDO.setAccountPeriod(accountPeriod);
                    ipvInvoiceDO.setIpv(ipv);
                    Map<String, Boolean> ipvFlagMap = new HashMap<>();
                    ipvFlagMap.put("ipvFlag", Boolean.FALSE);
                    Boolean flag = buildIpvAttr(buDateFrom, buDateTo, apOrderDtlDTO, ipvInvoiceDO, apOrderDTO, lotNo, ipvFlagMap);
                    if (!flag) {
                        continue;
                    }
                    ipvInvoiceDOS.add(ipvInvoiceDO);
                    Boolean ipvFlag = ipvFlagMap.get("ipvFlag");
                    //更新IPV账单计算标识
                    apOrderDtlRepoProc.updateIpvFlagById(apOrderDtlDTO.getId(), ipvFlag);
                    log.info("删除历史数据:{}", apOrderDtlDTO.getId());
                    IpvInvoiceParam ipvInvoiceParam = new IpvInvoiceParam();
                    ipvInvoiceParam.setAccountPeriod(accountPeriod);
                    ipvInvoiceParam.setApOrderNo(ipvInvoiceDO.getApOrderNo());
                    ipvInvoiceParam.setSourceDocNo(ipvInvoiceDO.getSourceDocNo());
                    ipvInvoiceParam.setSourceLineNo(ipvInvoiceDO.getSourceLineNo());
                    ipvInvoiceRepoProc.deleteByParam(ipvInvoiceParam);
                }
                //保存IPV账单
                batchInsert(ipvInvoiceDOS);
            } catch (Exception e) {
                log.error("【soAutoAlloc】批量自动配货异常：{}", e.getMessage());
                transactionStatus.setRollbackOnly();
            }
            return "ok";
        });
        redisTemplate.delete(redisKey);
    }

    private Map<String, List<InvIoRpcDTO>> getInvIoBySourceNoMap(List<ApOrderDtlDTO> unComputeArOrderDtlList) {
        Set<String> sourceNos = unComputeArOrderDtlList.stream().filter(row -> row.getInvIoId() == null).map(ApOrderDtlDTO::getSourceNo).distinct().collect(Collectors.toSet());
        if (CollUtil.isEmpty(sourceNos)) {
            return new HashMap<>();
        }
        List<InvIoRpcDTO> invIoRpcDTOS = rmiInvRpcService.findWhBySourceNos(new ArrayList<>(sourceNos));
        if (CollUtil.isEmpty(invIoRpcDTOS)) {
            return new HashMap<>();
        }
        return invIoRpcDTOS.stream().collect(Collectors.groupingBy(row -> row.getSourceNo() + "@" + row.getSrcDocDid()));
    }

    private Map<String, List<InvIoRpcDTO>> getInvIoByIdMap(List<ApOrderDtlDTO> unComputeArOrderDtlList) {
        Set<Long> ioIds = unComputeArOrderDtlList.stream().filter(row -> row.getInvIoId() != null).map(ApOrderDtlDTO::getInvIoId).distinct().collect(Collectors.toSet());
        if (CollUtil.isEmpty(ioIds)) {
            return new HashMap<>();
        }
        List<InvIoRpcDTO> invIoRpcDTOS = rmiInvRpcService.findInvIoByIds(new ArrayList<>(ioIds));
        if (CollUtil.isEmpty(invIoRpcDTOS)) {
            return new HashMap<>();
        }
        return invIoRpcDTOS.stream().collect(Collectors.groupingBy(row -> row.getSourceNo() + "@" + row.getSrcDocDid()));
    }

    private Boolean buildIpvAttr(LocalDateTime buDateFrom, LocalDateTime buDateTo, ApOrderDtlDTO apOrderDtlDTO,
                                 IpvInvoiceDO ipvInvoiceDO, ApOrderDTO apOrderDTO, String lotNo, Map<String, Boolean> ipvFlagMap) {
        BigDecimal ipv = ipvInvoiceDO.getIpv();
        if (ipvInvoiceDO.getApQty().compareTo(BigDecimal.ZERO) < 0) {
            ipvInvoiceDO.setRatioAmt(ipvInvoiceDO.getIpv());
            ipvFlagMap.put("ipvFlag", Boolean.TRUE);
            return Boolean.TRUE;
        }
        if (ipv.abs().compareTo(BigDecimal.ONE) <= 0) {
            ipvInvoiceDO.setRatioAmt(ipvInvoiceDO.getIpv());
            ipvFlagMap.put("ipvFlag", Boolean.TRUE);
            return Boolean.TRUE;
        }
        IpvInvoiceParam ipvInvoiceParam = new IpvInvoiceParam();
        ipvInvoiceParam.setApOrderNo(apOrderDTO.getApOrderNo());
        ipvInvoiceParam.setSourceDocNo(apOrderDtlDTO.getSourceNo());
        ipvInvoiceParam.setSourceLineNo(apOrderDtlDTO.getSourceLine());
        IpvRespVO ipvRespVO = ipvInvoiceRepoProc.findIpvByParam(ipvInvoiceParam);
        BigDecimal saledQty = ipvRespVO.getSaleQty() == null ? BigDecimal.ZERO : ipvRespVO.getSaleQty();
        BigDecimal saleRatio = ipvRespVO.getSaleRatio() == null ? BigDecimal.ZERO : ipvRespVO.getSaleRatio();
        BigDecimal ratioAmt = ipvRespVO.getRatioAmt() == null ? BigDecimal.ZERO : ipvRespVO.getRatioAmt();
        //获取销售出库数量
        InvIoParamRpcDTO invIoParamRpcDTO = new InvIoParamRpcDTO();
        invIoParamRpcDTO.setWhId(ipvInvoiceDO.getWhId());
        invIoParamRpcDTO.setOuId(apOrderDTO.getOuId());
        invIoParamRpcDTO.setLotNo(lotNo);
        invIoParamRpcDTO.setItemId(apOrderDtlDTO.getItemId());
        invIoParamRpcDTO.setIoDateFrom(buDateFrom);
        invIoParamRpcDTO.setIoDateTo(buDateTo);
        invIoParamRpcDTO.setToUom(apOrderDtlDTO.getUom());
        List<InvIoRpcDTO> saleOutList = rmiInvRpcService.findSaleOutByParam(invIoParamRpcDTO);
        if (CollUtil.isEmpty(saleOutList)) {
            log.info("查询销售出库数据为空:{}", apOrderDtlDTO.getId());
            return Boolean.FALSE;
        }
        InvIoRpcDTO salOutInfo = saleOutList.get(0);
        //应付单数量-[应付单、来源单、来源单行号已存在的当期销售出库数量
        BigDecimal subtractQty = apOrderDtlDTO.getQty().subtract(saledQty);
        if (salOutInfo.getConvertQty().compareTo(subtractQty) >= 0) {
            ipvInvoiceDO.setSaleQty(subtractQty);
            ipvInvoiceDO.setSaleRatio(BigDecimal.ONE.subtract(saleRatio));
            ipvInvoiceDO.setRatioAmt(ipv.subtract(ratioAmt));
            ipvFlagMap.put("ipvFlag", Boolean.TRUE);
        } else {
            ipvInvoiceDO.setSaleQty(salOutInfo.getConvertQty());
            ipvInvoiceDO.setSaleRatio(ipvInvoiceDO.getSaleQty().divide(apOrderDtlDTO.getQty(), 4, RoundingMode.HALF_UP));
            ipvInvoiceDO.setRatioAmt(ipvInvoiceDO.getSaleRatio().multiply(ipv));
            ipvFlagMap.put("ipvFlag", Boolean.FALSE);
        }
        return Boolean.TRUE;
    }

    /**
     * 批量保存IPV账单
     *
     * @param dataList
     */
    public void batchInsert(List<IpvInvoiceDO> dataList) {
        int index = 0;
        int batchSize = 500;
        for (IpvInvoiceDO data : dataList) {
            entityManager.persist(data);
            if (batchSize > 1) {
                // 开启批量
                index++;
                if (index % batchSize == 0) {
                    entityManager.flush();
                    entityManager.clear();
                }
            }
        }
        if (!dataList.isEmpty()) {
            entityManager.flush();
        }
    }
}
