package com.elitesland.fin.application.service.aptype;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.fin.application.convert.aptype.ApTypeConvert;
import com.elitesland.fin.application.facade.param.aptype.ApTypeParam;
import com.elitesland.fin.application.facade.vo.aptype.ApTypeVO;
import com.elitesland.fin.domain.entity.aptype.ApType;
import com.elitesland.fin.domain.param.aptype.ApTypePageParam;
import com.elitesland.fin.domain.service.aporder.ApOrderDomainService;
import com.elitesland.fin.domain.service.aptype.ApTypeDomainService;
import com.elitesland.fin.domain.service.aptype.ApTypeOuDomainService;
import com.elitesland.fin.infr.dto.aptype.ApTypeDTO;
import com.elitesland.fin.infr.dto.aptype.ApTypeOuDTO;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.application.service.aptype
 * @date 2022/3/11 15:36
 */
@Service
@RequiredArgsConstructor
public class ApTypeServiceImpl implements ApTypeService {

    private final ApTypeDomainService apTypeDomainService;
    private final ApTypeOuDomainService apTypeOuDomainService;

    private final ApOrderDomainService apOrderDomainService;


    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> save(ApTypeParam apTypeParam) {

        ApType apType = ApTypeConvert.INSTANCE.convert(apTypeParam);
        Long id = apTypeDomainService.save(apType);
        return ApiResult.ok(id);
    }


    @SysCodeProc
    @Override
    public PagingVO<ApTypeVO> page(ApTypePageParam apTypeParam) {

        PagingVO<ApTypeDTO> page = apTypeDomainService.page(apTypeParam);
        // DTO 转 VO
        PagingVO<ApTypeVO> res = ApTypeConvert.INSTANCE.convertPage(page);

        return res;
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Boolean> updateEnableFlag(List<Long> ids, Boolean enable) {

        List<ApTypeDTO> apTypeDTOS = apTypeDomainService.findByIds(ids);
        if (enable) {
            checkEnableRule(apTypeDTOS);
        }
        if (!enable) {
            checkStopRule(apTypeDTOS);
        }
        Boolean res = apTypeDomainService.updateEnableFlag(ids, enable);
        return ApiResult.ok(res);
    }


    /**
     * 停用状态校验
     */
    private void checkStopRule(List<ApTypeDTO> apTypeDTOS) {
        for (ApTypeDTO apTypeDTO : apTypeDTOS) {
            // 停用需是启用状态
            if (!apTypeDTO.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已停用状态不能再停用");
            }

        }
    }

    /**
     * 启用状态校验
     *
     * @param apTypeDTOS
     */
    private void checkEnableRule(List<ApTypeDTO> apTypeDTOS) {

        for (ApTypeDTO apTypeDTO : apTypeDTOS) {
            // 启用需是停用状态
            if (apTypeDTO.getEnableFlag()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "已启用状态不能再启动");
            }
            // 启用需校验已分配公司
            List<ApTypeOuDTO> apTypeOuDTOS = apTypeOuDomainService.queryByApTypeId(apTypeDTO.getId());
            if (CollUtil.isEmpty(apTypeOuDTOS)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "类型代码"+apTypeDTO.getApTypeCode()+"启用前需要分配公司");
            }

        }

    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> updateDef(Long id) {
        Long res = apTypeDomainService.updateDef(id);
        return ApiResult.ok(res);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public ApiResult<Long> del(Long id) {
        // 查询是否已被公司使用
        List<ApTypeOuDTO> apTypeOuDTOS = apTypeOuDomainService.queryByApTypeId(id);
        if (CollUtil.isNotEmpty(apTypeOuDTOS)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已分配公司!");
        }
        Boolean aBoolean = apOrderDomainService.queryByApTypeId(id);
        if(aBoolean){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该数据已关联应付单!");
        }

        Long res = apTypeDomainService.del(id);
        return ApiResult.ok(res);
    }


    @Override
    public ApiResult<ApTypeVO> getDef() {
        ApTypeDTO apTypeDTO = apTypeDomainService.getDef();
        ApTypeVO res = ApTypeConvert.INSTANCE.convert(apTypeDTO);
        if (res == null) {
            return ApiResult.ok(new ApTypeVO());
        }
        return ApiResult.ok(res);
    }

    @Override
    public ApiResult<List<ApTypeVO>> getList() {
        List<ApTypeDTO> apTypeDTOList = apTypeDomainService.getList();
        List<ApTypeVO> res = ApTypeConvert.INSTANCE.convertList(apTypeDTOList);
        return ApiResult.ok(res);
    }


}
