package com.elitesland.fin.infinity.utils;


import cn.hutool.extra.spring.SpringUtil;
import com.el.coordinator.core.common.utils.jwt.SpringUtils;
import com.elitesland.fin.infinity.vo.SshVo;
import lombok.extern.slf4j.Slf4j;
import net.schmizz.sshj.SSHClient;
import net.schmizz.sshj.connection.channel.direct.Session;
import net.schmizz.sshj.sftp.SFTPClient;
import net.schmizz.sshj.transport.verification.PromiscuousVerifier;

import java.io.IOException;
import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;

/**
 * @Description:
 * @author: huruizhong
 * @date: 2025/7/28 15:09
 */
@Slf4j
public class SshConnectionManager {
    private static SSHClient ssh;
    private static SshVo sshVo;
    private static Timer keepAliveTimer;
    private static int keepAliveInterval = 60; // 默认60秒



    // 初始化连接
    public static synchronized void init() throws IOException {
        if (ssh == null || !ssh.isConnected()) {
            String ip = SpringUtil.getProperty("TC.StatementAccount.ip");
            String port = SpringUtil.getProperty("TC.StatementAccount.port");

            String user = SpringUtil.getProperty("TC.StatementAccount.user");
            String password = SpringUtil.getProperty("TC.StatementAccount.password");

            log.info("尝试连接 SSH 服务器: {}", Thread.currentThread().getName());
            SSHClient sshClient = new SSHClient();
            sshClient.addHostKeyVerifier(new PromiscuousVerifier()); // 生产环境建议使用 KeyStoreVerifier
            sshClient.connect(ip, Integer.valueOf(port));
            sshClient.authPassword(user, password);
            ssh=sshClient;
            startKeepAlive(keepAliveInterval);
            log.info("连接 SSH 服务器完成: {}", Thread.currentThread().getName());

        }
    }

//

    // 启动保活机制
    public static void startKeepAlive(int intervalSeconds) {
        keepAliveInterval = intervalSeconds;
        if (keepAliveTimer != null) {
            keepAliveTimer.cancel();
        }

        keepAliveTimer = new Timer(true);
        keepAliveTimer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                try {
                    validateConnection();
                } catch (Exception e) {
                    log.warn("SSH 保活失败，尝试重连: {}", e.getMessage());
                    try {
                        reconnect();
                    } catch (IOException ex) {
                        log.error("SSH 重连失败: {}", ex.getMessage());
                    }
                }
            }
        }, intervalSeconds * 1000, intervalSeconds * 1000);
    }

    // 验证连接有效性
    public static boolean validateConnection() {
        if (ssh == null || !ssh.isConnected()) {
            return false;
        }

        try (Session session = ssh.startSession()) {
            Session.Command cmd = session.exec("exit 0");
            cmd.join(5, TimeUnit.SECONDS); // 设置超时时间
            return cmd.getExitStatus() == 0;
        } catch (Exception e) {
            log.warn("SSH 连接验证失败: {}", e.getMessage());
            return false;
        }
    }

    // 重新连接
    public static synchronized void reconnect() throws IOException {
        log.info("进入重连");
        if (validateConnection()){
            return;
        }
        if (ssh != null) {
            try {
                ssh.disconnect();
            } catch (Exception e) {
                // 忽略
            }
            log.info("ssh断开重新连接：{}", Thread.currentThread().getName());
        }
        init();
        sshVo = new SshVo(ssh, ssh.newSFTPClient());
        log.info("重连完成");
    }

    // 获取 SFTP 客户端
//    public SFTPClient getSftpClient() throws IOException {
//        if (!validateConnection()) {
//            reconnect();
//        }
//        return ssh.newSFTPClient();
//    }


    public static SshVo getSshVo() throws IOException {
        if (!validateConnection()) {
            reconnect();
            return sshVo;
        } else {
            while (Objects.isNull(sshVo)) {
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                log.info("等待连接对象初始化完成");
            }
//            log.info("连接已建立直接返回");
            return sshVo;
        }
    }

    // 关闭资源
    public void close() {
        if (keepAliveTimer != null) {
            keepAliveTimer.cancel();
        }

        if (ssh != null) {
            try {
                ssh.disconnect();
            } catch (Exception e) {
                log.warn("关闭 SSH 连接失败: {}", e.getMessage());
            }
        }
    }
}

