package com.elitesland.fin.utils;

import com.elitescloud.cloudt.common.base.QBaseModel;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.fin.application.facade.param.common.FinQueryParam;
import com.querydsl.core.types.Order;
import com.querydsl.core.types.OrderSpecifier;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQuery;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.elitesland.fin.utils.SqlUtil.SqlLikeOption.*;

/**
 * @author ryan.xu
 * @since 2022/12/07
 */
public abstract class SqlUtil {

    /**
     * SQL中的空集合
     */
    public static final String SQL_EMPTY_SET = "(NULL)";

    /**
     * 模糊查询选项
     */
    public enum SqlLikeOption {
        BOTH_SIDES, LEFT_SIDE, RIGHT_SIDE
    }

    /**
     * @param val 文本值
     * @return SQL文本值
     */
    public static String toSqlString(CharSequence val) {
        if (StringUtils.isEmpty(val)) {
            throw new IllegalArgumentException("[EDP-SQL] Parameter(val) is null or empty");
        }
        return escapeString(new StringBuilder(), val).toString();
    }

    private static StringBuilder escapeString(StringBuilder sb, CharSequence input) {
        sb.append('\'');
        for (int i = 0, n = input.length(); i < n; i++) {
            char ch = input.charAt(i);
            sb = ch == '\'' ? sb.append("''") : sb.append(ch);
        }
        sb.append('\'');
        return sb;
    }

    /**
     * @param val    匹配值
     * @param option 匹配选项
     * @return SQL匹配值
     */
    public static String toSqlLikeString(CharSequence val, SqlLikeOption option) {
        if (StringUtils.isEmpty(val)) {
            throw new IllegalArgumentException("[EDP-SQL] Parameter(val) is null or empty");
        }
        return escapeLikeString(new StringBuilder(), val, option).toString();
    }

    private static StringBuilder escapeLikeString(StringBuilder sb, CharSequence input, SqlLikeOption option) {
        Map<Character, String> escapeTable = escapeTable();
        sb.append(option != RIGHT_SIDE ? "%" : "");
        for (int i = 0, n = input.length(); i < n; i++) {
            char ch = input.charAt(i);
            String escaped = escapeTable.get(ch);
            sb = escaped != null ? sb.append(escaped) : sb.append(ch);
        }
        sb.append(option != LEFT_SIDE ? "%" : "");
//        sb.append(escapeClause());
        return sb;
    }

    /**
     * @param val 匹配值
     * @return SQL匹配值（左右全匹配场合）
     */
    public static String toSqlLikeString(String val) {
        return toSqlLikeString(val, BOTH_SIDES);
    }

    /**
     * @param vals 文本值集
     * @return SQL文本值集（用于`IN`操作）
     */
    public static String toSqlStringSet(Collection<? extends CharSequence> vals) {
        List<? extends CharSequence> nonNullVals = vals.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return nonNullVals.isEmpty() ? SQL_EMPTY_SET : nonNullVals.stream()
                .map(SqlUtil::toSqlString)
                .collect(Collectors.joining(",", "(", ")"));
    }

    /**
     * @param vals 数值集
     * @return SQL数值集（用于`IN`操作）
     */
    public static String toSqlNumberSet(Collection<? extends Number> vals) {
        List<? extends Number> nonNullVals = vals.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return nonNullVals.isEmpty() ? SQL_EMPTY_SET : nonNullVals.stream()
                .map(String::valueOf)
                .collect(Collectors.joining(",", "(", ")"));
    }


    /**
     * @param sqls 多个SQL查询语句
     * @return SQL联合查询（UNION ALL）
     */
 /*   public static String unionAll(SQL... sqls) {
        return Stream.of(sqls)
                .map(SQL::toString)
                .collect(Collectors.joining(" UNION ALL "));
    }*/

    private static final Pattern RE_WORD = Pattern.compile("^\\w+$");

    /**
     * 对SQL词汇进行输入检查
     *
     * @param input 输入的词汇
     * @return 检查后的SQL词汇
     * @throws IllegalArgumentException 检查通不过时抛出
     */
    public static String toSqlWord(String input) {
        if (!RE_WORD.matcher(input).matches()) {
            throw new IllegalArgumentException("[CORE-JDBC] Illegal inputted sql word: " + input);
        }
        return input;
    }

    /**
     * 转义字符集
     */
    private static final Map<Character, String> SQL_ESC = new HashMap<>();

    static {
        SQL_ESC.put('\'', "''");
        SQL_ESC.put('%', "\\%");
        SQL_ESC.put('_', "\\_");
        SQL_ESC.put('&', "\\&");
    }

    public static Map<Character, String> escapeTable() {
        return SQL_ESC;
    }

    public static String escapeClause() {
        return " escape '\\'";
    }

    /**
     * 常用基础查询条件拼装
     *
     * @param jpaQuery   jpaQuery对象
     * @param qBaseModel qBaseModel
     * @param query      查询条件
     * @param <T>        范型对象
     */
    public static <T> void handleCommonJpaQuery(JPAQuery<T> jpaQuery, QBaseModel qBaseModel, FinQueryParam query) {
        jpaQuery.where(qBaseModel.deleteFlag.eq(0));
        if (!ObjectUtils.isEmpty(query.getId())) {
            jpaQuery.where(qBaseModel.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            jpaQuery.where(qBaseModel.createUserId.eq(query.getCreateUserId()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateTimeStart())) {
            jpaQuery.where(qBaseModel.createTime.goe(query.getCreateTimeStart()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateTimeEnd())) {
            jpaQuery.where(qBaseModel.createTime.loe(query.getCreateTimeEnd()));
        }
        if (!ObjectUtils.isEmpty(query.getRemark())) {
            jpaQuery.where(qBaseModel.remark.like(SqlUtil.toSqlLikeString(query.getRemark())));
        }
    }

    /**
     * 自定义分页时的排序字段
     *
     * @param pathBase Q对象
     * @param orders   排序数据
     * @return 排序order
     */
    public static <T> OrderSpecifier<?>[] getSortedColumn(EntityPathBase<T> pathBase, List<OrderItem> orders) {
        if (orders == null || orders.size() == 0) {
            Path<String> fieldPath = Expressions.path(String.class, pathBase, "createTime");
            return new OrderSpecifier[]{new OrderSpecifier<>(Order.DESC, fieldPath)};
        } else {
            OrderSpecifier<String>[] orderSpecifiers = new OrderSpecifier[orders.size()];
            for (int i = 0; i < orders.size(); i++) {
                OrderItem orderItem = orders.get(i);
                Order order = orderItem.isAsc() ? Order.ASC : Order.DESC;
                Path<String> fieldPath = Expressions.path(String.class, pathBase, orderItem.getColumn());
                orderSpecifiers[i] = new OrderSpecifier<>(order, fieldPath);
            }
            return orderSpecifiers;
        }
    }

}
