package com.elitesland.fin.application.service.flow;

import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.fin.application.facade.param.flow.AccountFlowAccAmtParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowParam;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowAccAmtVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:统筹账户流水
 * </p>
 *
 * @Author Darren
 * @Date 2023/02/25
 * @Version 1.0
 * @Content:
 */
@Service
public class AccountFlowTcServiceImpl implements AccountFlowTcService {

    @Autowired
    private AccountFlowCommonService accountFlowCommonService;

    /**
     * 统筹账户类型：账户流水保存业务逻辑处理
     *
     * @param paramList 账户流水入参
     * @return 账户流水处理后的入参
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public List<AccountFlowParam> saveTc(List<AccountFlowParam> paramList) {
        //TODO 先不返回最上层保存了，一直向下每个逻辑块单独直接保存
        distinguishDataSource(paramList);
        return paramList;
    }

    /**
     * 按照数据来源区分：按照正常逻辑处理
     * 正常业务应该是每次只有一种数据来源
     * 账户类型+数据来源+交易类型必须是唯一的
     *
     * @param paramList 入参
     * @return
     */
    private void distinguishDataSource(List<AccountFlowParam> paramList) {

        val paramOptional = paramList.stream().map(AccountFlowParam::getDataSource).filter(Objects::nonNull).distinct().findFirst();
        if (paramOptional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据来源为空!");
        }
        String dataSource = paramOptional.get();
        if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TZD03.getValueCode())) {
            handleDataSource01(paramList);
        } else if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TZD04.getValueCode())) {
            handleDataSource02(paramList);
        } else if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TJD03.getValueCode())) {
            handleDataSource03(paramList);
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未匹配到数据来源!");
        }

        /* else if (Objects.equals(dataSource, UdcEnum.FIN_DATA_SOURCE_TYPE_TJD04.getValueCode())) {
            handleDataSource04(paramList);
        }*/


    }

    /***************************************************提供财务域接口开始************************************************/

    /**
     * 调整类型=调出的调整单审核通过：处理数据业务逻辑
     * <p>
     * 账户类型：统筹账户
     * 数据来源：调整类型=调出的调整单审核通过
     * 交易类型：调整单上的调整原因-调出
     *
     * @param paramList 入参
     * @return
     */
    private void handleDataSource01(List<AccountFlowParam> paramList) {
        paramList.forEach(param -> {
            //查询上一条的流水数据
            AccountFlowAccAmtVO previousData = accountFlowCommonService.selectPreviousData(param.getAccCode());
            //本条账户金额、账户占用金额业务处理计算逻辑
            AccountFlowAccAmtParam accountFlowAccAmtParam = AccountFlowAccAmtParam.builder()
                    .amount(param.getAmount())
                    .accAmt(previousData.getAccAmt())
                    .accOccAmt(previousData.getAccOccAmt())
                    .accHandleType(FinConstant.ACC_HANDLE_TYPE_SUB)
                    .accOccHandleType(FinConstant.ACC_HANDLE_TYPE_EQU).build();
            //计算后本条使用的数据
            AccountFlowAccAmtVO calculateData = accountFlowCommonService.handleAccAmt(accountFlowAccAmtParam);

            param.setAccAmt(calculateData.getAccAmt());
            param.setAccOccAmt(calculateData.getAccOccAmt());

            //默认赋值
            accountFlowCommonService.defaultAssignment(param);
            //账户流水保存并更新账户金额信息
            accountFlowCommonService.saveAccountAndFlow(param);
        });

        //流水数据保存
        // accountFlowCommonService.saveParamBatch(paramList);
    }


    /**
     * 调整类型=调进的调整单审核通过：处理数据业务逻辑
     * <p>
     * 账户类型：统筹账户
     * 数据来源：调整类型=调进的调整单审核通过
     * 交易类型：调整单上的调整原因-调进
     *
     * @param paramList 入参
     * @return
     */
    private void handleDataSource02(List<AccountFlowParam> paramList) {
        paramList.forEach(param -> {
            //查询上一条的流水数据
            AccountFlowAccAmtVO previousData = accountFlowCommonService.selectPreviousData(param.getAccCode());
            //本条账户金额、账户占用金额业务处理计算逻辑
            AccountFlowAccAmtParam accountFlowAccAmtParam = AccountFlowAccAmtParam.builder()
                    .amount(param.getAmount())
                    .accAmt(previousData.getAccAmt())
                    .accOccAmt(previousData.getAccOccAmt())
                    .accHandleType(FinConstant.ACC_HANDLE_TYPE_ADD)
                    .accOccHandleType(FinConstant.ACC_HANDLE_TYPE_EQU).build();
            //计算后本条使用的数据
            AccountFlowAccAmtVO calculateData = accountFlowCommonService.handleAccAmt(accountFlowAccAmtParam);

            param.setAccAmt(calculateData.getAccAmt());
            param.setAccOccAmt(calculateData.getAccOccAmt());

            //默认赋值
            accountFlowCommonService.defaultAssignment(param);
            //账户流水保存并更新账户金额信息
            accountFlowCommonService.saveAccountAndFlow(param);
        });

        //流水数据保存
        // accountFlowCommonService.saveParamBatch(paramList);
    }

    /**
     * 调剂单审核通过：处理数据业务逻辑
     * <p>
     * 账户类型：统筹账户
     * 数据来源：调剂单审核通过
     * 交易类型：调剂单的调整原因—调出/调剂单的调整原因—调进
     *
     * @param paramList 入参
     * @return
     */
    private void handleDataSource03(List<AccountFlowParam> paramList) {
        //调剂单审核通过时对入参的拆分类型做判断：用于区分发生金额是加还是减
        accountFlowCommonService.checkAdjustApprove(paramList);
        //按照拆分类型进行分组，从账户的(无需拆)，至账户(无需拆)
        Map<String, List<AccountFlowParam>> accountFlowListMap = paramList.stream().collect(Collectors.groupingBy(i -> i.getSplitType()));
        //从账户的调出无需拆
        val accountFlowTcTypeList = accountFlowListMap.get(FinConstant.SPLIT_TYPE_ADJUST_APPROVE_FROM);
        //至账户调进无需拆
        val accountFlowTjTypeList = accountFlowListMap.get(FinConstant.SPLIT_TYPE_ADJUST_APPROVE_TO);
        if (!CollectionUtils.isEmpty(accountFlowTcTypeList)) {
            //1.先调剂单的调整原因—调出
            accountFlowTcTypeList.forEach(param -> {
                //查询上一条的流水数据
                AccountFlowAccAmtVO previousData = accountFlowCommonService.selectPreviousData(param.getAccCode());
                //本条账户金额、账户占用金额业务处理计算逻辑
                AccountFlowAccAmtParam accountFlowAccAmtParam = AccountFlowAccAmtParam.builder()
                        .amount(param.getAmount())
                        .accAmt(previousData.getAccAmt())
                        .accOccAmt(previousData.getAccOccAmt())
                        .accHandleType(FinConstant.ACC_HANDLE_TYPE_SUB)
                        .accOccHandleType(FinConstant.ACC_HANDLE_TYPE_EQU).build();
                //计算后本条使用的数据
                AccountFlowAccAmtVO calculateData = accountFlowCommonService.handleAccAmt(accountFlowAccAmtParam);

                param.setAccAmt(calculateData.getAccAmt());
                param.setAccOccAmt(calculateData.getAccOccAmt());

                //默认赋值
                accountFlowCommonService.defaultAssignment(param);
                //账户流水保存并更新账户金额信息
                accountFlowCommonService.saveAccountAndFlow(param);
            });
        }
        //流水数据保存
        //accountFlowCommonService.saveParamBatch(accountFlowTcTypeList);

        //2.再调剂单的调整原因—调进
        if (!CollectionUtils.isEmpty(accountFlowTjTypeList)) {
            handleDataSource04(accountFlowTjTypeList);
        }
    }

    /**
     * 调整类型=调进的调剂单审核通过：处理数据业务逻辑
     * <p>
     * 账户类型：统筹账户
     * 数据来源：调整类型=调进的调剂单审核通过
     * 交易类型：调剂单的调整原因—调进
     *
     * @param paramList 入参
     * @return
     */
    private void handleDataSource04(List<AccountFlowParam> paramList) {
        paramList.forEach(param -> {
            //查询上一条的流水数据
            AccountFlowAccAmtVO previousData = accountFlowCommonService.selectPreviousData(param.getAccCode());
            //本条账户金额、账户占用金额业务处理计算逻辑
            AccountFlowAccAmtParam accountFlowAccAmtParam = AccountFlowAccAmtParam.builder()
                    .amount(param.getAmount())
                    .accAmt(previousData.getAccAmt())
                    .accOccAmt(previousData.getAccOccAmt())
                    .accHandleType(FinConstant.ACC_HANDLE_TYPE_ADD)
                    .accOccHandleType(FinConstant.ACC_HANDLE_TYPE_EQU).build();
            //计算后本条使用的数据
            AccountFlowAccAmtVO calculateData = accountFlowCommonService.handleAccAmt(accountFlowAccAmtParam);

            param.setAccAmt(calculateData.getAccAmt());
            param.setAccOccAmt(calculateData.getAccOccAmt());

            //默认赋值
            accountFlowCommonService.defaultAssignment(param);
            //账户流水保存并更新账户金额信息
            accountFlowCommonService.saveAccountAndFlow(param);
        });

        //流水数据保存
        //accountFlowCommonService.saveParamBatch(paramList);
    }


    /***************************************************提供财务域接口结束************************************************/

}
