package com.elitesland.fin.rpc.system;


import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.core.base.SeqNumProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.dto.SysUdcDTO;
import com.elitescloud.cloudt.system.dto.resp.SysCurrencyRespDTO;
import com.elitescloud.cloudt.system.provider.SysUdcRpcService;
import com.elitescloud.cloudt.system.provider.SysUserRpcService;
import com.elitescloud.cloudt.system.provider.extend.SysCurrencyRpcService;
import com.elitescloud.cloudt.system.service.SysSettingService;
import com.elitescloud.cloudt.system.vo.SysSettingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author zhiyu.he
 * @date 2022/4/26 10:21
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class SystemRpcServiceImpl implements SystemRpcService {

    private final SysUserRpcService sysUserRpcService;

    private final SeqNumProvider sysNumberRuleService;

    private final SysCurrencyRpcService sysCurrencyRpcService;

    private final SysSettingService sysSettingService;

    private final SysUdcRpcService sysUdcRpcService;

    @Override
    public SysUserDTO getUserByName(String userName) {
        ApiResult<SysUserDTO> user = sysUserRpcService.getUserByUsername(userName);
        if (user != null) {
            if (user.getData() != null) {
                return user.getData();
            } else {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "用户名称" + userName + "获取信息为空!");
            }
        } else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "用户名称 " + userName + "获取信息为空!");
        }
    }

    /**
     * 发号器发号
     *
     * @param code
     * @param runtimeValues
     * @return
     */
    @Override
    public String sysNumberRuleGenerateCode(String appCode, String code, List<String> runtimeValues) {
        log.info("系统域发号器调用：时间：{}，参数{},{}", LocalDateTime.now(), code, runtimeValues);
        List<String> list = new ArrayList<>();
        list.addAll(runtimeValues);
        try {
            return sysNumberRuleService.generateCode(appCode, code, list);
        } catch (Exception e) {
            log.error("sysNumberRuleGenerateCode error:", e);
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "发号器发号失败");
        }
    }

    /**
     * 查询币种
     *
     * @param currCode 币种编码
     * @return 币种
     */
    @Override
    public SysCurrencyRespDTO findCurrByCode(String currCode) {
        var apiResult = sysCurrencyRpcService.getByCode(currCode);
        Assert.notNull(apiResult, "远程调用查询币种信息失败");
        Assert.isTrue(apiResult.isSuccess(), "查询币种信息失败");
        return apiResult.getData();
    }

    /**
     * 查询币种
     *
     * @param currCodes 币种编码
     * @return 币种
     */
    @Override
    public List<SysCurrencyRespDTO> findCurrByCodes(Set<String> currCodes) {
        var apiResult = sysCurrencyRpcService.listByCodes(currCodes);
        Assert.notNull(apiResult, "远程调用查询币种信息失败");
        Assert.isTrue(apiResult.isSuccess(), "查询币种信息失败");
        return apiResult.getData();
    }

    @Override
    public SysSettingVO findSysSettingByNo(String no) {
        return sysSettingService.oneByNo(no);
    }

    @Override
    public List<SysUdcDTO> listByUdcCode(String appCode, String udcCode) {
        Set<String> udcCodes = new HashSet<>();
        udcCodes.add(udcCode);
        var apiResult = sysUdcRpcService.listByUdcCode(appCode, udcCodes);
        Assert.notNull(apiResult, "远程调用查询UDC信息失败");
        Assert.isTrue(apiResult.isSuccess(), "远程调用查询UDC信息失败");
        return apiResult.getData();
    }

}
