package com.elitesland.fin.application.service.account;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.account.AccountConvert;
import com.elitesland.fin.application.facade.dto.account.AccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccount.CreditAccountDTO;
import com.elitesland.fin.application.facade.dto.creditaccountflow.CreditAccountFlowDTO;
import com.elitesland.fin.application.facade.excel.account.AccountImportEntity;
import com.elitesland.fin.application.facade.param.account.*;
import com.elitesland.fin.application.facade.param.creditaccountflow.CreditAccountFlowPageParam;
import com.elitesland.fin.application.facade.param.flow.AccountFlowPageParam;
import com.elitesland.fin.application.facade.vo.account.*;
import com.elitesland.fin.application.facade.vo.app.StoreAccountCustAppVO;
import com.elitesland.fin.application.facade.vo.creditaccountflow.CreditAccountFlowVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowAppVO;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.app.AppAccountService;
import com.elitesland.fin.application.service.creditaccountflow.CreditAccountFlowService;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.common.ExcelEntityDataListener;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.account.AccountDO;
import com.elitesland.fin.entity.account.AccountSnapshotDO;
import com.elitesland.fin.repo.account.AccountRepo;
import com.elitesland.fin.repo.account.AccountRepoProc;
import com.elitesland.fin.repo.account.AccountSnapshotRepo;
import com.elitesland.fin.repo.account.AccountSnapshotRepoProc;
import com.elitesland.fin.repo.creditaccount.CreditAccountRepoProc;
import com.elitesland.fin.rpc.sale.RmiSaleRpcService;
import com.elitesland.fin.rpc.system.SystemRpcService;
import com.elitesland.fin.rpc.ystsupp.RmiOrgOuRpcServiceService;
import com.elitesland.sale.api.vo.resp.sal.RmiOrgBankAccRpcVO;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.application.service.account
 * @date 2023/2/16 09:02
 */
@Service
@RequiredArgsConstructor
public class AccountServiceImpl implements AccountService {

    private final AccountRepoProc accountRepoProc;
    private final AccountRepo accountRepo;
    private final RmiOrgOuRpcServiceService rmiOrgOuRpcServiceService;

    private final SystemRpcService systemRpcService;

    private final UdcProvider udcProvider;

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";
    private final RmiSaleRpcService rmiSaleRpcService;

    private final AccountSnapshotRepoProc accountSnapshotRepoProc;
    private final AccountSnapshotRepo accountSnapshotRepo;
    private final FlexFieldUtilService flexFieldUtilService;
    private final CreditAccountRepoProc creditAccountRepoProc;

    private AccountFlowService accountFlowService;
    @Autowired
    @Lazy
    public void setAccountFlowService(AccountFlowService accountFlowService) {
        this.accountFlowService = accountFlowService;
    }
    private CreditAccountFlowService creditAccountFlowService;
    @Autowired
    @Lazy
    public void setCreditAccountFlowService(CreditAccountFlowService creditAccountFlowService) {
        this.creditAccountFlowService = creditAccountFlowService;
    }

    private final AppAccountService appAccountService;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(AccountSaveParam accountSaveParam) {
        checkAccountSaveParam(accountSaveParam);

//        checkAccountExist(accountParam);
        // 新增
        if (ObjectUtil.isNull(accountSaveParam.getId())) {

            String accountCode = systemRpcService.sysNumberRuleGenerateCode("yst-fin", "ZH", new ArrayList<>());
            accountCode = accountSaveParam.getAccountHolderCode() + "-" + accountCode;
            accountSaveParam.setAccountCode(accountCode);
            // 账户编码唯一
            if (accountRepo.existsByAccountCode(accountCode)) {
                throw new BusinessException("账户编码已存在");
            }
            // 一家公司的客户只能创建一种类型的账户
            if (accountRepo.existsBySecOuCodeAndAccountHolderCodeAndAccountType(accountSaveParam.getSecOuCode(), accountSaveParam.getAccountHolderCode(), accountSaveParam.getAccountType())) {
                throw new BusinessException("该客户已经创建账户");
            }

            accountSaveParam.setState(UdcEnum.FIN_ACTIVE_STATUS_ACTIVE.getValueCode());
        }
        AccountDO accountDO = AccountConvert.INSTANCE.saveParam2DO(accountSaveParam);
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.ACCOUNT, accountDO);
        accountRepo.save(accountDO);
    }

    private void checkAccountSaveParam(AccountSaveParam accountSaveParam) {
        Assert.notEmpty(accountSaveParam.getSecOuCode(), "归属公司必填");
        Assert.notEmpty(accountSaveParam.getAccountHolderName(), "开户主体名称必填");
        Assert.notEmpty(accountSaveParam.getAccountType(), "账户类型必填");
        Assert.notEmpty(accountSaveParam.getAccountHolderCode(), "开户主体编码必填");

        Assert.notNull(accountSaveParam.getAccountAmount(), "账户金额必填");
        Assert.notNull(accountSaveParam.getAccountOccupancyAmount(), "账户占用金额必填");
        Assert.notNull(accountSaveParam.getAccountAvailableAmount(), "账户可用金额必填");

        Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);
        if (!pattern.matcher(accountSaveParam.getAccountAmount().toString()).matches()) {
            Assert.notNull(accountSaveParam.getAccountAmount(), "账户金额格式不正确,最多两位小数");
        }

        if (!pattern.matcher(accountSaveParam.getAccountOccupancyAmount().toString()).matches()) {
            Assert.notNull(accountSaveParam.getAccountAmount(), "账户占用金额格式不正确,最多两位小数");
        }

        if (!pattern.matcher(accountSaveParam.getAccountAvailableAmount().toString()).matches()) {
            Assert.notNull(accountSaveParam.getAccountAmount(), "账户可用金额格式不正确,最多两位小数");
        }

        Assert.isTrue(accountSaveParam.getAccountAmount().compareTo(accountSaveParam.getAccountOccupancyAmount().add(accountSaveParam.getAccountAvailableAmount())) == 0,
                "账户金额不等于账户占用金额+账户可用金额");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void accountImport(MultipartFile file) throws IOException {
        List<String> errorList = new ArrayList<>();

        List<AccountImportEntity> accountImportEntityList;
        ExcelEntityDataListener<AccountImportEntity> dataInternalListener = new ExcelEntityDataListener<>();
        EasyExcelFactory.read(file.getInputStream(), AccountImportEntity.class, dataInternalListener).sheet(0).headRowNumber(1).doRead();
        accountImportEntityList = dataInternalListener.getDatas();

        Assert.notEmpty(accountImportEntityList, "excel为空");

        accountImportEntityList.stream().forEach(item -> {
            if (FinConstant.YES.equals(item.getDefaultAccount())) {
                item.setDefaultAccount(FinConstant.TRUE);
            }

            if (FinConstant.NO.equals(item.getDefaultAccount())) {
                item.setDefaultAccount(FinConstant.FALSE);
            }
        });

        List<AccountDO> accountDOList = AccountConvert.INSTANCE.accountImportEntityList2AccountDOList(accountImportEntityList);

        buildErrorMsg(accountDOList, errorList);

        if (CollectionUtils.isNotEmpty(errorList)) {
            throw new BusinessException(StringUtils.join(errorList, FinConstant.WRAP));
        }

        //构建默认值
        buildDefaultValue(accountDOList);

        accountRepo.saveAll(accountDOList);
    }


    public void buildDefaultValue(List<AccountDO> accountDOList) {
        accountDOList.stream().forEach(item -> {
            //开票主体类型
            if (StringUtils.isNotEmpty(item.getAccountHolderType())) {
                item.setAccountHolderType(UdcEnum.PRINCIPAL_TYPE_INVOICE.getValueCode());
            }
        });
    }

    public void buildErrorMsg(List<AccountDO> accountDOList, List<String> errorList) {
        //必填校验
        checkImportMandatoryField(accountDOList, errorList);

        //重复账户编码校验
        checkImportRepeatAccountCode(accountDOList, errorList);

        //数据有效性校验
        checkDataValid(accountDOList, errorList);


    }

    public void checkDataValid(List<AccountDO> accountDOList, List<String> errorList) {

        //归属公司
        List<String> secOuCodeList = accountDOList.stream().map(AccountDO::getSecOuCode).collect(Collectors.toList());
        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuCodes(secOuCodeList);
        List<OrgOuRpcDTO> orgOuRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(orgOuRpcDtoParam);

        Set<String> ouCodeSet = orgOuRpcDTOList.stream().map(OrgOuRpcDTO::getOuCode).collect(Collectors.toSet());

        //归属加盟商
        List<String> secFranchiseeCodeList = accountDOList.stream().map(AccountDO::getSecFranchiseeCode).collect(Collectors.toList());


        //开户主体类型
        Map<String, String> principalTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.PRINCIPAL_TYPE_INVOICE.getModel(), UdcEnum.PRINCIPAL_TYPE_INVOICE.getCode());
        Collection<String> principalTypeList = principalTypeUdc.values();

        //开户主体名称
        //账户编号
        //账户名称

        //账户类型
        Map<String, String> finAccountTypeUdc = udcProvider.getValueMapByUdcCode(UdcEnum.ACCOUNT_TYPE_STORE.getModel(), UdcEnum.ACCOUNT_TYPE_STORE.getCode());
        Collection<String> finAccountTypeList = finAccountTypeUdc.values();

        for (int i = 0; i < accountDOList.size(); i++) {
            //归属公司
            if (StringUtils.isNotEmpty(accountDOList.get(i).getSecOuCode())) {
                if (!ouCodeSet.contains(accountDOList.get(i).getSecOuCode())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "公司编码", "公司编码在系统不存在"));
                }
            }

            //开户主体类型
            if (StringUtils.isNotEmpty(accountDOList.get(i).getAccountHolderType())) {
                if (!principalTypeList.contains(accountDOList.get(i).getAccountHolderType())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "开户主体类型", "开户主体类型在系统不存在"));
                }
            }

            //账户类型
            if (StringUtils.isNotEmpty(accountDOList.get(i).getAccountType())) {
                if (!finAccountTypeList.contains(accountDOList.get(i).getAccountType())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户类型", "账户类型在系统不存在"));
                }
            }

            Pattern pattern = Pattern.compile(FinConstant.TWO_DECIMAL_REGULAR);

            //账户金额
            if (accountDOList.get(i).getAccountAmount() != null) {
                if (!pattern.matcher(accountDOList.get(i).getAccountAmount().toString()).matches()) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户金额", "账户金额格式有误"));
                }
            }

            //账户占用金额
            if (accountDOList.get(i).getAccountOccupancyAmount() != null) {
                if (!pattern.matcher(accountDOList.get(i).getAccountOccupancyAmount().toString()).matches()) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户占用金额", "账户占用金额格式有误"));
                }
            }

            //账户可用金额
            if (accountDOList.get(i).getAccountAvailableAmount() != null) {
                if (!pattern.matcher(accountDOList.get(i).getAccountAvailableAmount().toString()).matches()) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户可用金额", "账户可用金额格式有误"));
                }
            }

            //计算金额是否正确
            if (accountDOList.get(i).getAccountAmount() != null &&
                    accountDOList.get(i).getAccountOccupancyAmount() != null &&
                    accountDOList.get(i).getAccountAvailableAmount() != null) {
                if (accountDOList.get(i).getAccountAmount().compareTo(accountDOList.get(i).getAccountOccupancyAmount().add(accountDOList.get(i).getAccountAvailableAmount())) != 0) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "金额", "账户金额不等于账户占用金额+账户可用金额"));
                }
            }
        }
    }

    private void checkImportRepeatAccountCode(List<AccountDO> accountDOList, List<String> errorList) {
        List<AccountDO> allAccountDOList = accountRepo.findAll();
        Set<String> accountCodeSet = allAccountDOList.stream().map(AccountDO::getAccountCode).collect(Collectors.toSet());

        for (int i = 0; i < accountDOList.size(); i++) {
            if (accountCodeSet.contains(accountDOList.get(i).getAccountCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户编码", "账户编码在系统已经存在"));
            }
        }

        Set<String> excelAccountCodeSet = new HashSet<>();
        for (int i = 0; i < accountDOList.size(); i++) {
            if (StringUtils.isNotEmpty(accountDOList.get(i).getAccountCode())) {
                if (excelAccountCodeSet.contains(accountDOList.get(i).getAccountCode())) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户编码", "账户编码重复"));
                }
                excelAccountCodeSet.add(accountDOList.get(i).getAccountCode());
            }
        }
    }

    private void checkImportMandatoryField(List<AccountDO> accountDOList, List<String> errorList) {
        for (int i = 0; i < accountDOList.size(); i++) {
            if (StringUtils.isEmpty(accountDOList.get(i).getSecOuCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "归属公司编码", "归属公司编码必填"));
            }

            if (StringUtils.isEmpty(accountDOList.get(i).getSecFranchiseeCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "归属加盟商编码", "归属加盟商编码编码必填"));
            }

            if (StringUtils.isEmpty(accountDOList.get(i).getAccountHolderName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "开户主体名称", "开户主体名称必填"));
            }

            if (StringUtils.isEmpty(accountDOList.get(i).getAccountCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户编码", "账户编码必填"));
            }

            if (StringUtils.isEmpty(accountDOList.get(i).getAccountName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户名称", "账户名称必填"));
            }

            if (StringUtils.isEmpty(accountDOList.get(i).getAccountType())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户类型", "账户类型必填"));
            }

            if (accountDOList.get(i).getAccountAmount() == null) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户金额", "账户金额必填"));
            }

            if (accountDOList.get(i).getAccountAvailableAmount() == null) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户可用金额", "账户可用金额必填"));
            }
            if (accountDOList.get(i).getAccountOccupancyAmount() == null) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, i + 2, "账户占用金额", "账户占用金额必填"));
            }
        }
    }

    @SysCodeProc
    @Override
    public AccountVO get(Long id) {
        AccountVO res = accountRepoProc.get(id);
        return res;
    }

    @SysCodeProc
    @Override
    public PagingVO<AccountVO> page(AccountPageParam accountPageParam) {

        PagingVO<AccountVO> res = accountRepoProc.page(accountPageParam);
        if (CollectionUtils.isEmpty(res.getRecords())) {
            return PagingVO.<AccountFlowVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        //根据公司编码查询公司名称
        //归属加盟商编码查询名称
        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        List<String> ouCodes = new ArrayList<>();
//        List<String> franchiseeCodes = new ArrayList<>();
        res.getRecords().stream().forEach(item -> {
            ouCodes.add(item.getSecOuCode());
//            franchiseeCodes.add(item.getSecFranchiseeCode());
        });
        orgOuRpcDtoParam.setOuCodes(ouCodes);

        List<OrgOuRpcDTO> orgOuRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(orgOuRpcDtoParam);


        res.getRecords().stream().forEach(item -> {
            OrgOuRpcDTO orgOuResult = orgOuRpcDTOList.stream()
                    .filter(orgOuRpcDTO -> StrUtil.equals(item.getSecOuCode(), orgOuRpcDTO.getOuCode()))
                    .findFirst()
                    .orElse(null);
            if (orgOuResult != null) {
                item.setSecOuName(orgOuResult.getOuName());
            }
        });
        flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.ACCOUNT,res.getRecords());
        return res;
    }

    @Override
    @SysCodeProc
    public List<AccountDTO> getAccountByAccountParam(AccountParam param) {
        return accountRepoProc.getAccountByAccountParam(param);
    }


    @Override
    @Transactional
    public Long updateState(AccountParam accountParam) {
        if (null == accountParam.getIds()) {
            throw new BusinessException("id不能为空");
        }
        Long res = accountRepoProc.updateState(accountParam);
        return res;
    }

    @Override
    public List<AccountVO> queryAccount(List<String> accountCodes, String type, String state) {
        return accountRepoProc.queryByAccounts(accountCodes, type, state);
    }

    @Override
    public AccountVO getByCode(String accCode) {
        AccountVO res = accountRepoProc.getByCode(accCode);
        return res;
    }

    @Override
    @Transactional
    public Boolean updateAmtByCode(String accCode, BigDecimal accAmt, BigDecimal accOccAmt) {
        Assert.notEmpty(accCode, "accCode不能为空");
        AccountParam accountParam = new AccountParam();
        accountParam.setAccCode(accCode);
        accountParam.setAccAmt(accAmt);
        accountParam.setAccOccAmt(accOccAmt);
        if (null != accAmt && null != accOccAmt) {
            return accountRepoProc.updateAmtByCode(accountParam);
        }
        if (null != accAmt) {
            return accountRepoProc.updateAccAmtByCode(accountParam);
        }
        if (null != accOccAmt) {
            return accountRepoProc.updateOccAmtByCode(accountParam);
        }
        return true;

    }


    @Override
    public List<AccountBankInfoVo> getBankByCustCode(String custCode) {
        List<RmiOrgBankAccRpcVO> bankAccRpcVOS = rmiSaleRpcService.findBankAccByCustCode(custCode);
        if (CollectionUtil.isEmpty(bankAccRpcVOS)) {
            throw new BusinessException("查询客户银行信息为空，请配置");
        }
        // 查询付款银行账号信息
        List<RmiOrgBankAccRpcVO> bankVos = bankAccRpcVOS.stream().filter(item -> StrUtil.equals(item.getAccType(), "OUT")).collect(Collectors.toList());
        return AccountConvert.INSTANCE.rpcBankVos2Vos(bankVos);
    }

    @Override
    @SysCodeProc
    public PagingVO<AccountSnapshotVo> querySnapshot(AccountSnapshotParam param) {

        PagingVO<AccountSnapshotVo> pagingVO = accountSnapshotRepoProc.querySnapshot(param);
        //根据公司编码查询公司名称
        //归属加盟商编码查询名称
        OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        List<String> ouCodes = new ArrayList<>();
        pagingVO.getRecords().stream().forEach(item -> {
            ouCodes.add(item.getSecOuCode());
        });
        orgOuRpcDtoParam.setOuCodes(ouCodes);

        List<OrgOuRpcDTO> orgOuRpcDTOList = rmiOrgOuRpcServiceService.findOuDtoByParam(orgOuRpcDtoParam);

        pagingVO.getRecords().stream().forEach(item -> {
            OrgOuRpcDTO orgOuResult = orgOuRpcDTOList.stream()
                    .filter(orgOuRpcDTO -> StrUtil.equals(item.getSecOuCode(), orgOuRpcDTO.getOuCode()))
                    .findFirst()
                    .orElse(null);
            if (orgOuResult != null) {
                item.setSecOuName(orgOuResult.getOuName());
            }
        });
        if(CollectionUtils.isNotEmpty(pagingVO.getRecords())){
            flexFieldUtilService.handleFlexFieldShowNameForVO(FinFlexFieldCodeConstant.ACCOUNT,pagingVO.getRecords());
        }
        return pagingVO;

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void accountSnapshot(String param) {
        LocalDateTime dateTime = LocalDateTime.now();
        List<AccountDO> accountDOList = accountRepo.findAll();
        if (CollectionUtil.isEmpty(accountDOList)) {
            return;
        }
        List<AccountSnapshotDO> snapshotDOS = AccountConvert.INSTANCE.accountDos2Snapshots(accountDOList);
        snapshotDOS.stream().forEach(snapshot -> {
            snapshot.setSnapshotTime(dateTime);
        });
        accountSnapshotRepo.saveAll(snapshotDOS);

    }

    @Override
    @SysCodeProc
    public List<AccountSnapshotDO> selectAccountSnapshotByParam(AccountSnapshotParam queryParam) {
        List<AccountSnapshotDO> accountSnapshotDOList = accountSnapshotRepoProc.selectAccountSnapshotByParam(queryParam);
        if (CollectionUtils.isEmpty(accountSnapshotDOList)) {
            return Collections.EMPTY_LIST;
        }
        return accountSnapshotDOList;
    }


    @Override
    @SysCodeProc
    public List<AccountOuVO> selectOuByParam(AccountAppParam accountAppParam){
        selectCustByStore(accountAppParam);
        if(StringUtils.isBlank(accountAppParam.getCustCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户编码为空!");
        }

        List<AccountOuVO> accountOuAllList = new ArrayList<>();
        //List<String> accountHolderCodes = accountRepoProc.selectSecOuCodeByAccountHolderCode(accountAppParam.getCustCode());
        //List<String> objectCodes = creditAccountRepoProc.selectOuCodeByObjectCode(accountAppParam.getCustCode());
       // List<String> ouCodeAllList = Stream.of(accountHolderCodes, objectCodes).flatMap(list -> list.stream()).filter(Objects::nonNull).distinct().collect(Collectors.toList());

        List<AccountOuVO> accountOuList = accountRepoProc.selectAccountOuByAccountHolderCode(accountAppParam.getCustCode());
        if (CollectionUtil.isNotEmpty(accountOuList)){
            accountOuAllList.addAll(accountOuList);
        }
        List<AccountOuVO> creditAccountOuList = creditAccountRepoProc.selectAccountOuByObjectCode(accountAppParam.getCustCode());
        if (CollectionUtil.isNotEmpty(creditAccountOuList)){
            accountOuAllList.addAll(creditAccountOuList);
        }
        if (CollectionUtil.isEmpty(accountOuAllList)){
            return Collections.EMPTY_LIST;
        }

        List<String> ouCodeAllList = accountOuAllList.stream().map(AccountOuVO::getOuCode).distinct().filter(Objects::nonNull).collect(Collectors.toList());
        Map<String, OrgOuRpcSimpleDTO> ouMap = rmiOrgOuRpcServiceService.findBaseOuMapByCodes(ouCodeAllList);

       /* accountOuAllList = accountOuAllList.stream().collect(Collectors.collectingAndThen(
                    Collectors.toCollection(() -> new TreeSet<AccountOuVO>(Comparator.comparing(AccountOuVO::getOuCode))), ArrayList::new)
            );*/
        accountOuAllList = accountOuAllList.stream().distinct().collect(Collectors.collectingAndThen(
                        Collectors.toCollection(() -> new TreeSet<AccountOuVO>(Comparator.comparing(AccountOuVO::getOuCode))), ArrayList::new));


        accountOuAllList = accountOuAllList.stream().sorted(Comparator.comparing(AccountOuVO::getCreateTime).reversed()).collect(Collectors.toList());

       /* List<AccountOuVO> accountOuVOList = accountOuAllList.stream().map(s -> {
            AccountOuVO accountOuVO = new AccountOuVO();
            accountOuVO.setOuCode(s);
            OrgOuRpcSimpleDTO ouRpcSimpleDTO = ouMap.get(s);
            if (Objects.nonNull(ouRpcSimpleDTO)){
                accountOuVO.setOuId(ouRpcSimpleDTO.getId());
                accountOuVO.setOuName(ouRpcSimpleDTO.getOuName());
            }

            return accountOuVO;
        }).collect(Collectors.toList());*/

        accountOuAllList.forEach(accountOuVO -> {
            OrgOuRpcSimpleDTO ouRpcSimpleDTO = ouMap.get(accountOuVO.getOuCode());
            if (Objects.nonNull(ouRpcSimpleDTO)){
                accountOuVO.setOuId(ouRpcSimpleDTO.getId());
                accountOuVO.setOuName(ouRpcSimpleDTO.getOuName());
            }
        });

        return accountOuAllList;
    }

    @Override
    @SysCodeProc
    public AccountAmtSumVO selectAmtSumByParam(AccountAppParam accountAppParam){
        selectCustByStore(accountAppParam);

        if(StringUtils.isBlank(accountAppParam.getCustCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户编码为空!");
        }
        if(StringUtils.isBlank(accountAppParam.getOuCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "公司编号为空!");
        }
        AccountAmtSumVO amtSumVO = new AccountAmtSumVO();

        AccountAmtSumVO accountAmtSumVO = accountRepoProc.selectAmtSumByParam(accountAppParam.getOuCode(),accountAppParam.getCustCode());
        AccountAmtSumVO creditAccountAmtSumVO = creditAccountRepoProc.selectAmtSumByParam(accountAppParam.getOuCode(),accountAppParam.getCustCode());
        if (Objects.isNull(accountAmtSumVO) && Objects.isNull(creditAccountAmtSumVO)){
            return amtSumVO;
        }
        BigDecimal accountAmountSum = BigDecimal.ZERO;
        BigDecimal accountOccupancyAmountSum = BigDecimal.ZERO;
        BigDecimal accountAvailableAmountSum = BigDecimal.ZERO;

        if (Objects.nonNull(accountAmtSumVO)){
            if(Objects.nonNull(accountAmtSumVO.getAccountAmountSum())){
                accountAmountSum = accountAmountSum.add(accountAmtSumVO.getAccountAmountSum());
            }
            if(Objects.nonNull(accountAmtSumVO.getAccountOccupancyAmountSum())){
                accountOccupancyAmountSum = accountOccupancyAmountSum.add(accountAmtSumVO.getAccountOccupancyAmountSum());
            }
            if(Objects.nonNull(accountAmtSumVO.getAccountAvailableAmountSum())){
                accountAvailableAmountSum = accountAvailableAmountSum.add(accountAmtSumVO.getAccountAvailableAmountSum());
            }
        }

        if (Objects.nonNull(creditAccountAmtSumVO)){
            if(Objects.nonNull(creditAccountAmtSumVO.getAccountAmountSum())){
                accountAmountSum = accountAmountSum.add(creditAccountAmtSumVO.getAccountAmountSum());
            }
            //信用账户存在使用金额，账户汇总需要减去使用金额
            if (Objects.nonNull(creditAccountAmtSumVO.getCreditAccountUsedAmountSum())) {
                accountAmountSum = accountAmountSum.subtract(creditAccountAmtSumVO.getCreditAccountUsedAmountSum());
            }
            if(Objects.nonNull(creditAccountAmtSumVO.getAccountOccupancyAmountSum())){
                accountOccupancyAmountSum = accountOccupancyAmountSum.add(creditAccountAmtSumVO.getAccountOccupancyAmountSum());
            }
            if(Objects.nonNull(creditAccountAmtSumVO.getAccountAvailableAmountSum())){
                accountAvailableAmountSum = accountAvailableAmountSum.add(creditAccountAmtSumVO.getAccountAvailableAmountSum());
            }
        }

        amtSumVO.setAccountAmountSum(accountAmountSum);
        amtSumVO.setAccountAvailableAmountSum(accountAvailableAmountSum);
        amtSumVO.setAccountOccupancyAmountSum(accountOccupancyAmountSum);

        return amtSumVO;
    }

    @Override
    @SysCodeProc
    public AccountAmtFlowVO selectAccountAmtAndFlow(AccountAppParam accountAppParam){
        selectCustByStore(accountAppParam);

        if(StringUtils.isBlank(accountAppParam.getCustCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户编码为空!");
        }
        if(StringUtils.isBlank(accountAppParam.getOuCode())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "公司编号为空!");
        }
        if(StringUtils.isBlank(accountAppParam.getAccountType())){
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户类型为空!");
        }

        if(Objects.nonNull(accountAppParam.getPreviousMonth())){
            LocalDate localDate = LocalDate.now();
            LocalDate previousLocalDate = localDate.minus(accountAppParam.getPreviousMonth(), ChronoUnit.MONTHS);
            LocalDateTime transactionTimeS = LocalDateTime.of(previousLocalDate, LocalTime.MIN);
            LocalDateTime transactionTimeE = LocalDateTime.of(localDate, FinConstant.LOCAL_TIME_MAX);

            accountAppParam.setTransactionTimeS(transactionTimeS);
            accountAppParam.setTransactionTimeE(transactionTimeE);
        }



        AccountAmtFlowVO accountAmtFlowVO = new AccountAmtFlowVO();
        accountAmtFlowVO.setFlowPagingVO(PagingVO.<AccountFlowAppVO>builder().total(0L).records(Collections.EMPTY_LIST).build());

        if (Objects.equals(accountAppParam.getAccountType(),UdcEnum.ACCOUNT_TYPE_STORE.getValueCode()) ||
                Objects.equals(accountAppParam.getAccountType(),UdcEnum.ACCOUNT_TYPE_FLZH.getValueCode())){
            List<AccountVO> accountVOList = accountRepoProc.selectAccountAmtByParam(accountAppParam.getOuCode(),accountAppParam.getCustCode(),accountAppParam.getAccountType());
            if (CollectionUtil.isEmpty(accountVOList)){
                return accountAmtFlowVO;
            }
            if (accountVOList.size()>1){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户+公司+账户类型查询到多条账户信息!");
            }
            AccountVO accountVO = accountVOList.get(0);
            accountAmtFlowVO.setAccountCode(accountVO.getAccountCode());
            accountAmtFlowVO.setAccountName(accountVO.getAccountName());
            accountAmtFlowVO.setAccountType(accountVO.getAccountType());
            BigDecimal accountAvailableAmount = Objects.isNull(accountVO.getAccountAvailableAmount()) ? BigDecimal.ZERO : accountVO.getAccountAvailableAmount();
            BigDecimal accountOccupancyAmount = Objects.isNull(accountVO.getAccountOccupancyAmount()) ? BigDecimal.ZERO : accountVO.getAccountOccupancyAmount();
            BigDecimal accountAmount = accountOccupancyAmount.add(accountAvailableAmount);
            accountAmtFlowVO.setAccountAmount(accountAmount);
            //accountAmtFlowVO.setAccountAmount(accountVO.getAccountAmount());
            accountAmtFlowVO.setAccountAvailableAmount(accountVO.getAccountAvailableAmount());
            accountAmtFlowVO.setAccountOccupancyAmount(accountVO.getAccountOccupancyAmount());

            selectAccountFlow(accountAmtFlowVO,accountAppParam);

        }else if (Objects.equals(accountAppParam.getAccountType(),UdcEnum.CREDIT_ACCOUNT_TYPE_CREDIT.getValueCode())){
            List<CreditAccountDTO> creditAccountDTOList = creditAccountRepoProc.selectCreditAccountAmtByParam(accountAppParam.getOuCode(),accountAppParam.getCustCode(),accountAppParam.getAccountType());
            if (CollectionUtil.isEmpty(creditAccountDTOList)){
                return accountAmtFlowVO;
            }
            if (creditAccountDTOList.size()>1){
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "客户+公司+账户类型查询到多条信用账户信息!");
            }
            CreditAccountDTO creditAccountDTO = creditAccountDTOList.get(0);
            accountAmtFlowVO.setAccountCode(creditAccountDTO.getCreditAccountCode());
            accountAmtFlowVO.setAccountName(creditAccountDTO.getCreditAccountName());
            accountAmtFlowVO.setAccountType(creditAccountDTO.getCreditAccountType());
            BigDecimal creditAccountAvailableLimit = Objects.isNull(creditAccountDTO.getCreditAccountAvailableLimit()) ? BigDecimal.ZERO : creditAccountDTO.getCreditAccountAvailableLimit();
            BigDecimal creditAccountOccupancyLimit = Objects.isNull(creditAccountDTO.getCreditAccountOccupancyLimit()) ? BigDecimal.ZERO : creditAccountDTO.getCreditAccountOccupancyLimit();
            BigDecimal accountAmount = creditAccountOccupancyLimit.add(creditAccountAvailableLimit);

            accountAmtFlowVO.setAccountAmount(accountAmount);
            //accountAmtFlowVO.setAccountAmount(creditAccountDTO.getCreditAccountLimit());
            accountAmtFlowVO.setAccountAvailableAmount(creditAccountDTO.getCreditAccountAvailableLimit());
            accountAmtFlowVO.setAccountOccupancyAmount(creditAccountDTO.getCreditAccountOccupancyLimit());

            selectCreditAccountFlow(accountAmtFlowVO,accountAppParam);

        }else {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "账户类型未匹配到!");
        }


        return accountAmtFlowVO;
    }

    public void selectAccountFlow(AccountAmtFlowVO accountAmtFlowVO,AccountAppParam accountAppParam){
        AccountFlowPageParam flowPageParam = new AccountFlowPageParam();
        flowPageParam.setCurrent(accountAppParam.getCurrent() + 1);
        flowPageParam.setSize(accountAppParam.getSize());
        flowPageParam.setKeyword(accountAppParam.getKeyword());
        flowPageParam.setAccountCode(accountAmtFlowVO.getAccountCode());
        flowPageParam.setAccountType(accountAppParam.getAccountType());
        flowPageParam.setFlowNo(accountAppParam.getFlowNo());
        flowPageParam.setTransactionTimeS(accountAppParam.getTransactionTimeS());
        flowPageParam.setTransactionTimeE(accountAppParam.getTransactionTimeE());
        flowPageParam.setAmountFrom(accountAppParam.getAmountFrom());
        flowPageParam.setAmountTo(accountAppParam.getAmountTo());
        flowPageParam.setTransactionTypeList(accountAppParam.getTransactionTypeList());
        flowPageParam.setSourceDocList(accountAppParam.getSourceDocList());
        flowPageParam.setSourceNo(accountAppParam.getSourceNo());
        flowPageParam.setOrders(accountAppParam.getOrders());
        flowPageParam.setCustomParam(accountAppParam.getCustomParam());
        flowPageParam.setConditions(accountAppParam.getConditions());
        flowPageParam.setBasicModuleCode(accountAppParam.getBasicModuleCode());
        flowPageParam.setBusinessObjectCode(accountAppParam.getBusinessObjectCode());
        flowPageParam.setAuditDateS(accountAppParam.getAuditDateS());
        flowPageParam.setAuditDateE(accountAppParam.getAuditDateE());
        PagingVO<AccountFlowVO> pagingVO = accountFlowService.appPage(flowPageParam);

        List<AccountFlowVO> flowVOList = pagingVO.getRecords();
        if (CollectionUtils.isEmpty(flowVOList)) {
            accountAmtFlowVO.setFlowPagingVO(PagingVO.<AccountFlowAppVO>builder().total(0L).records(Collections.EMPTY_LIST).build());
        }else {
            List<AccountFlowAppVO> accountFlowAppVOList = flowVOList.stream().map(accountFlowVO -> {
                AccountFlowAppVO accountFlowAppVO = new AccountFlowAppVO();
                accountFlowAppVO.setId(accountFlowVO.getId());
                accountFlowAppVO.setFlowNo(accountFlowVO.getFlowNo());
                accountFlowAppVO.setAmount(accountFlowVO.getAmount());
                accountFlowAppVO.setTransactionType(accountFlowVO.getTransactionType());
                accountFlowAppVO.setTransactionTypeName(accountFlowVO.getTransactionTypeName());
                accountFlowAppVO.setTransactionTime(accountFlowVO.getTransactionTime());
                accountFlowAppVO.setSourceDoc(accountFlowVO.getSourceDoc());
                accountFlowAppVO.setSourceDocName(accountFlowVO.getSourceDocName());
                accountFlowAppVO.setSourceNo(accountFlowVO.getSourceNo());
                accountFlowAppVO.setAccountType(accountFlowVO.getAccountType());
                accountFlowAppVO.setAccountTypeName(accountFlowVO.getAccountTypeName());
                accountFlowAppVO.setAccountCode(accountFlowVO.getAccountCode());
                accountFlowAppVO.setAccountName(accountFlowVO.getAccountName());
                accountFlowAppVO.setAuditDate(accountFlowVO.getAuditDate());
                return accountFlowAppVO;
            }).collect(Collectors.toList());


            accountAmtFlowVO.setFlowPagingVO(PagingVO.<AccountFlowAppVO>builder()
                    .total(pagingVO.getTotal())
                    .records(accountFlowAppVOList)
                    .build());
        }

    }

    public void selectCreditAccountFlow(AccountAmtFlowVO accountAmtFlowVO,AccountAppParam accountAppParam){
        CreditAccountFlowPageParam flowPageParam = new CreditAccountFlowPageParam();
        flowPageParam.setCurrent(accountAppParam.getCurrent() + 1);
        flowPageParam.setSize(accountAppParam.getSize());
        flowPageParam.setKeyword(accountAppParam.getKeyword());
        flowPageParam.setCreditAccountCode(accountAmtFlowVO.getAccountCode());
        flowPageParam.setCreditAccountType(accountAppParam.getAccountType());
        flowPageParam.setFlowNo(accountAppParam.getFlowNo());
        flowPageParam.setTransactionTimeS(accountAppParam.getTransactionTimeS());
        flowPageParam.setTransactionTimeE(accountAppParam.getTransactionTimeE());
        flowPageParam.setAmountFrom(accountAppParam.getAmountFrom());
        flowPageParam.setAmountTo(accountAppParam.getAmountTo());
        flowPageParam.setTransactionTypeList(accountAppParam.getTransactionTypeList());
        flowPageParam.setSourceDocList(accountAppParam.getSourceDocList());
        flowPageParam.setSourceNo(accountAppParam.getSourceNo());
        flowPageParam.setOrders(accountAppParam.getOrders());
        flowPageParam.setCustomParam(accountAppParam.getCustomParam());
        flowPageParam.setConditions(accountAppParam.getConditions());
        flowPageParam.setBasicModuleCode(accountAppParam.getBasicModuleCode());
        flowPageParam.setBusinessObjectCode(accountAppParam.getBusinessObjectCode());
        flowPageParam.setAuditDateS(accountAppParam.getAuditDateS());
        flowPageParam.setAuditDateE(accountAppParam.getAuditDateE());
        PagingVO<CreditAccountFlowDTO> pagingVO = creditAccountFlowService.appPage(flowPageParam);

        List<CreditAccountFlowDTO> flowDTOList = pagingVO.getRecords();
        if (CollectionUtils.isEmpty(flowDTOList)) {
            accountAmtFlowVO.setFlowPagingVO(PagingVO.<AccountFlowAppVO>builder().total(0L).records(Collections.EMPTY_LIST).build());
        }else {
            List<AccountFlowAppVO> accountFlowAppVOList = flowDTOList.stream().map(accountFlowVO -> {
                AccountFlowAppVO accountFlowAppVO = new AccountFlowAppVO();
                accountFlowAppVO.setId(accountFlowVO.getId());
                accountFlowAppVO.setFlowNo(accountFlowVO.getFlowNo());
                accountFlowAppVO.setAmount(accountFlowVO.getAmount());
                accountFlowAppVO.setTransactionType(accountFlowVO.getTransactionType());
                accountFlowAppVO.setTransactionTypeName(accountFlowVO.getTransactionTypeName());
                accountFlowAppVO.setTransactionTime(accountFlowVO.getTransactionTime());
                accountFlowAppVO.setSourceDoc(accountFlowVO.getSourceDoc());
                accountFlowAppVO.setSourceDocName(accountFlowVO.getSourceDocName());
                accountFlowAppVO.setSourceNo(accountFlowVO.getSourceNo());
                accountFlowAppVO.setAccountType(accountFlowVO.getCreditAccountType());
                accountFlowAppVO.setAccountTypeName(accountFlowVO.getCreditAccountTypeName());
                accountFlowAppVO.setAccountCode(accountFlowVO.getCreditAccountCode());
                accountFlowAppVO.setAccountName(accountFlowVO.getCreditAccountName());
                accountFlowAppVO.setAuditDate(accountFlowVO.getAuditDate());

                return accountFlowAppVO;
            }).collect(Collectors.toList());


            accountAmtFlowVO.setFlowPagingVO(PagingVO.<AccountFlowAppVO>builder()
                    .total(pagingVO.getTotal())
                    .records(accountFlowAppVOList)
                    .build());
        }

    }


    private void selectCustByStore(AccountAppParam accountAppParam){

        if (StringUtils.isBlank(accountAppParam.getCustCode())){
            //AccountAppParam accountAppParam = new AccountAppParam();
            accountAppParam.setStoreCode(accountAppParam.getStoreCode());
            StoreAccountCustAppVO accountCustAppVO = appAccountService.selectCheckCustByStore(accountAppParam);

            accountAppParam.setCustCode(accountCustAppVO.getStoreCustCode());
        }

        //return accountCustAppVO;
    }

}
