package com.elitesland.fin.domain.service.paymentperiod;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.fin.application.convert.paymentperiod.AgingRangeConvert;
import com.elitesland.fin.application.facade.param.paymentperiod.AgingRangePageParam;
import com.elitesland.fin.application.facade.param.paymentperiod.AgingRangeSaveParam;
import com.elitesland.fin.application.facade.vo.paymentperiod.AgingRangePagingVO;
import com.elitesland.fin.application.facade.vo.paymentperiod.AgingRangeVO;
import com.elitesland.fin.domain.entity.paymentperiod.AgingRangeDO;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeDtlRepo;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeDtlRepoProc;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeRepo;
import com.elitesland.fin.infr.repo.paymentperiod.AgingRangeRepoProc;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/04/16
 * @Version 1.0
 * @Content:
 */
@Service
@RequiredArgsConstructor
public class AgingRangeDomainServiceImpl implements AgingRangeDomainService{

    private final AgingRangeRepo agingRangeRepo;
    private final AgingRangeRepoProc agingRangeRepoProc;
    private final AgingRangeDtlRepo agingRangeDtlRepo;
    private final AgingRangeDtlRepoProc agingRangeDtlRepoProc;


    /**
     * 账龄区间保存
     *
     * @param saveParam 入参
     * @return账龄区间ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public Long save(AgingRangeSaveParam saveParam) {
        AgingRangeDO AgingRangeDO = AgingRangeConvert.INSTANCE.saveVoToDo(saveParam);
        Long id = agingRangeRepo.save(AgingRangeDO).getId();
        return id;
    }


    /**
     * 账龄区间分页查询
     *
     * @param pageParam 入参
     * @return 账龄区间信息集合
     */
    @Override
    @SysCodeProc
    public PagingVO<AgingRangePagingVO> page(AgingRangePageParam pageParam) {

        PagingVO<AgingRangePagingVO> pagingVO = agingRangeRepoProc.page(pageParam);
        if (CollectionUtils.isEmpty(pagingVO.getRecords())) {
            return PagingVO.<AgingRangePagingVO>builder().total(0L).records(Collections.EMPTY_LIST).build();
        }

        return pagingVO;
    }

    /**
     * 根据ID查询主表数据
     *
     * @param id 账龄区间ID
     * @return 账龄区间主表数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO findById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "ID不能为空!");
        }
        Optional<AgingRangeDO> optional = agingRangeRepo.findById(id);
        if (optional.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        return AgingRangeConvert.INSTANCE.doToRespVo(optional.get());
    }

    /**
     * 根据账龄区间ID查询账龄区间主表数据
     *
     * @param id 账龄区间ID
     * @return 账龄区间主表数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO selectById(Long id) {
        if (Objects.isNull(id)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "ID不能为空!");
        }
        AgingRangePageParam queryVO = new AgingRangePageParam();
        queryVO.setId(id);
        List<AgingRangeVO> respVOList = agingRangeRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        AgingRangeVO agreementVO = respVOList.get(0);

        return agreementVO;
    }

    /**
     * 根据账龄区间编码查询账龄区间主表数据
     *
     * @param code 根据账龄区间编码
     * @return 账龄区间主表数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO selectByCode(String code) {
        if (StringUtils.isBlank(code)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "编码不能为空!");
        }
        AgingRangePageParam queryVO = new AgingRangePageParam();
        queryVO.setAgingRangeCode(code);
        List<AgingRangeVO> respVOList = agingRangeRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(respVOList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "未查询到数据信息!");
        }
        if (respVOList.size() > 1) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "查询到多条数据信息!");
        }
        AgingRangeVO agreementVO = respVOList.get(0);

        return agreementVO;
    }

    /**
     * 根据入参查询账龄区间数据
     *
     * @param queryVO 入参
     * @return 账龄区间主表数据
     */
    @Override
    @SysCodeProc
    public List<AgingRangeVO> selectByParam(AgingRangePageParam queryVO) {
        List<AgingRangeVO> agreementVOList = agingRangeRepoProc.selectListByParam(queryVO);
        if (CollectionUtils.isEmpty(agreementVOList)) {
            return Collections.EMPTY_LIST;
        }
        return agreementVOList;
    }

    /**
     * 校验查询编号/名称是否已经存在
     *
     * @param queryVO 入参
     * @return 账龄区间主表数据
     */
    @Override
    @SysCodeProc
    public AgingRangeVO checkSelectByParam(AgingRangePageParam queryVO) {
        AgingRangeVO agreementVO = agingRangeRepoProc.checkSelectByParam(queryVO);
        return agreementVO;
    }

    /**
     * 根据ID批量更新状态
     *
     * @param status 状态
     * @param ids    ID集合
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStatusBatch(String status, List<Long> ids) {
        agingRangeRepoProc.updateStatusBatch(status, ids);
    }

    /**
     * 根据ID更新状态
     *
     * @param status 状态
     * @param id     ID
     * @return
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateStatusById(String status, Long id) {
        agingRangeRepoProc.updateStatusById(status, id);
    }


    /**
     * 根据ID批量更新deleteFlag
     *
     * @param deleteFlag 删除标记
     * @param ids        主表ID
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDeleteFlagBatch(Integer deleteFlag, List<Long> ids) {
        agingRangeRepoProc.updateDeleteFlagBatch(deleteFlag, ids);
    }
}
