package com.elitesland.fin.provider.accountingengine;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.accountingengine.FinEventTableConvert;
import com.elitesland.fin.application.convert.accountingengine.FinJournalConvert;
import com.elitesland.fin.application.convert.accountingengine.FinJournalGenerateConvert;
import com.elitesland.fin.application.facade.dto.accountingengine.FinEventTableDTO;
import com.elitesland.fin.application.facade.dto.accountingengine.FinJournalDTO;
import com.elitesland.fin.application.facade.param.accountingengine.ManualProposedParam;
import com.elitesland.fin.application.service.accountingengine.FinAccountEngineJdbcService;
import com.elitesland.fin.application.service.accountingengine.JournalGenerateService;
import com.elitesland.fin.domain.entity.accountingengine.FinFlexibleValueDO;
import com.elitesland.fin.domain.entity.accountingengine.FinJournalDO;
import com.elitesland.fin.domain.entity.accountingengine.FinSetOfBookDO;
import com.elitesland.fin.domain.entity.accountingengine.FinSetOfBookLineDO;
import com.elitesland.fin.dto.accountingengine.FinFlexibleValueCheckItemRpcDTO;
import com.elitesland.fin.dto.accountingengine.FinJournalRpcDTO;
import com.elitesland.fin.param.accountingengine.FinJournalRpcParam;
import com.elitesland.fin.param.accountingengine.FinJournalSyncRpcParam;
import com.elitesland.fin.param.accountingengine.ManualProposedRpcParam;
import com.elitesland.fin.repo.accountingengine.*;
import com.elitesland.fin.service.accountingengine.JournalGenerateRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明:
 * </p>
 *
 * @Author Darren
 * @Date 2024/03/19
 * @Version 1.0
 * @Content:
 */
@RequiredArgsConstructor
@RestController
@RequestMapping(Application.URI_PREFIX + JournalGenerateRpcService.PATH)
@Slf4j
public class JournalGenerateRpcServiceImpl implements JournalGenerateRpcService {

    private final JournalGenerateService journalGenerateService;
    private final FinJournalRepoProc finJournalRepoProc;
    private final FinFlexibleValueRepo finFlexibleValueRepo;
    private final FinSetOfBookRepo finSetOfBookRepo;
    private final FinSetOfBookLineRepo finSetOfBookLineRepo;
    private final FinJournalRepo finJournalRepo;
    private final FinEventTableRepoProc finEventTableRepoProc;
    private final FinAccountEngineJdbcService finAccountEngineJdbcService;
    private final FinAccountEngineDetailsRepoProc finAccountEngineDetailsRepoProc;

    @Override
    public ApiResult<Void> manualProposed(ManualProposedRpcParam rpcParam) {
        log.info("手动拟定的Rpc入参:{}", JSON.toJSONString(rpcParam));

        if (Objects.isNull(rpcParam)){
            return ApiResult.ok();
        }

        ManualProposedParam manualProposedParam = FinJournalGenerateConvert.INSTANCE.rpcParamToParam(rpcParam);
        journalGenerateService.manualProposed(manualProposedParam);

        return ApiResult.ok();
    }

    @Override
    @SysCodeProc
    public ApiResult<List<FinJournalRpcDTO>> findWaitSyncJournal(FinJournalRpcParam param) {
        List<FinJournalDTO> journalDTOList = finJournalRepoProc.findWaitSyncJournal(param);
        if(CollectionUtils.isEmpty(journalDTOList)){
            return ApiResult.ok(new ArrayList<>());
        }
        List<FinJournalRpcDTO> rpcDTOList = FinJournalConvert.INSTANCE.DTOToRpc(journalDTOList);
        // 查询科目
        List<String> segment2List = rpcDTOList.stream().map(FinJournalRpcDTO::getSegment2).distinct().collect(Collectors.toList());
        if(CollectionUtils.isNotEmpty(segment2List)){
            Map<String, FinFlexibleValueDO> flexibleMap = finFlexibleValueRepo.findByFlexibleValueCodeIn(segment2List).stream()
                    .collect(Collectors.toMap(FinFlexibleValueDO::getFlexibleValueCode, t -> t));
            // 查询核算维度
            List<FinSetOfBookDO> bookDOList = finSetOfBookRepo.findAllBySobCode("NXG01");
            if(CollectionUtils.isEmpty(bookDOList)){
                return ApiResult.fail("账簿NXG01不存在");
            }
            Map<String, FinSetOfBookLineDO> lineMap = finSetOfBookLineRepo.findAllByMasId(bookDOList.get(0).getId()).stream()
                    .collect(Collectors.toMap(FinSetOfBookLineDO::getAccountDimenCode, t -> t));
            for (FinJournalRpcDTO rpcDTO : rpcDTOList) {
                if(StringUtils.isNotEmpty(rpcDTO.getSegment2())){
                    Map<String, Object> map = BeanUtil.beanToMap(rpcDTO);
                    if(!flexibleMap.containsKey(rpcDTO.getSegment2())){
                        return ApiResult.fail("科目" + rpcDTO.getSegment2() + "不存在");
                    }
                    FinFlexibleValueDO valueDO = flexibleMap.get(rpcDTO.getSegment2());
                    if(StringUtils.isNotEmpty(valueDO.getCheckitementry())){
                        List<FinFlexibleValueCheckItemRpcDTO> checkItemRpc = JSON.parseArray(valueDO.getCheckitementry(), FinFlexibleValueCheckItemRpcDTO.class);
                        Map<String, Object> assgrp = new HashMap<>();
                        for (FinFlexibleValueCheckItemRpcDTO checkItemRpcDTO : checkItemRpc) {
                            if(!lineMap.containsKey(checkItemRpcDTO.getAsstactitemNumber())){
                                return ApiResult.fail("核算维度" + checkItemRpcDTO.getAsstactitemNumber() + "不存在");
                            }
                            FinSetOfBookLineDO bookLineDO = lineMap.get(checkItemRpcDTO.getAsstactitemNumber());
                            Map<String, Object> number = new HashMap<>();
                            number.put("number", map.get(bookLineDO.getColumnName()) != null ? map.get(bookLineDO.getColumnName()) : "");
                            assgrp.put(bookLineDO.getAccountDimenName(), number);
                        }
                        rpcDTO.setAssgrp(assgrp);
                    }
                }
            }
        }
        return ApiResult.ok(rpcDTOList);
    }

    @Transactional
    @Override
    public ApiResult<Void> updateSyncStatus(List<FinJournalSyncRpcParam> param) {
        log.info("凭证同步状态更新, 时间：{}，入参：{}", LocalDateTime.now(), JSONObject.toJSONString(param));
        // 更新业务单据状态，根据凭证的业务单号（docNum1）和会计引擎ID（accountEngineId）查询
        List<Long> idList = param.stream().map(FinJournalSyncRpcParam::getId).collect(Collectors.toList());
        List<FinJournalDO> journalDOS = finJournalRepo.findByIdIn(idList);
        List<Long> engineIdList = journalDOS.stream().map(FinJournalDO::getAccountEngineId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if(CollectionUtils.isEmpty(engineIdList)){
            return ApiResult.fail("会计引擎ID不存在");
        }
        Map<Long, FinJournalDO> engineMap = journalDOS.stream().collect(Collectors.toMap(FinJournalDO::getId, t -> t));
        List<FinEventTableDTO> tableDOS = finEventTableRepoProc.findByEngineId(engineIdList);
        Map<Long, FinEventTableDTO> tableMap = tableDOS.stream().collect(Collectors.toMap(FinEventTableDTO::getEngineId, t -> t));

        //获取事件表单数据库链接
        Connection connection = null;
        try {
            connection = finAccountEngineJdbcService.getEventTableConnection(FinEventTableConvert.INSTANCE.DTOToDO(tableDOS.get(0)));
        } catch (Exception e) {
            log.error("获取事件表单数据库连接异常,事件表单:{},会计引擎ID：{}", tableDOS.get(0), e);
            return ApiResult.fail("获取事件表单数据库连接异常");
        }
        try {
            for (FinJournalSyncRpcParam paramVO : param) {
                if(engineMap.containsKey(paramVO.getId())){
                    FinJournalDO finJournalDO = engineMap.get(paramVO.getId());
                    FinEventTableDTO tableDTO = tableMap.get(finJournalDO.getAccountEngineId());
                    if(tableDTO != null){
                        try {
                            String updateProposedStatusSql = "update " + tableDTO.getDatabaseName() + "." + tableDTO.getMasTable()
                                    + " set transfer_status = '" + paramVO.getSyncStatus()
                                    + "' where " + tableDTO.getColumnDocNum() + " = '" + finJournalDO.getDocNum() + "'";
                            PreparedStatement preparedStatement = connection.prepareStatement(updateProposedStatusSql);
                            log.info("更新传帐状态sql:\n{}", updateProposedStatusSql);
                            int res = preparedStatement.executeUpdate();
                            preparedStatement.close();
                        } catch (SQLException e) {
                            log.error("更新传帐状态异常, {}", e);
                            throw new BusinessException("更新传帐状态异常异常：" + e.getMessage());
                        }
                    }
                }
                finJournalRepoProc.updateSyncStatus(paramVO);
            }
        } catch (Exception e) {
            log.error("更新凭证同步状态异常,{}", e);
            throw new BusinessException("更新传帐状态异常异常：" + e.getMessage());
        } finally {
            //释放数据库连接
            finAccountEngineJdbcService.closeConnection(connection);
        }
        return ApiResult.ok();
    }

    @Override
    public ApiResult<Void> updateSyncStatusBatch(FinJournalSyncRpcParam param) {
        log.info("凭证同步状态回写, 时间：{}，入参：{}", LocalDateTime.now(), JSONObject.toJSONString(param));
        if(CollectionUtils.isEmpty(param.getIdList())){
            return ApiResult.fail("ID列表为空");
        }
        finJournalRepoProc.updateSyncStatusBatch(param);
        return ApiResult.ok();
    }

    /**
     * 删除逻辑：
     *
     * 金蝶未处理凭证，中台发出【删除】指令，金蝶删除凭证，回传删除成功，中台删除成功，视为删除成功，
     * 删除成功后，业务单据的拟定状态重置为草稿，传账状态重置为待传账，
     * 会计引擎的事件分类明细里面，拟定类型改为手动，传账类型改为手动，如果金蝶删除不成功，中台凭证也不能删除，后续只能进行【红冲】
     *
     * 红冲逻辑：
     *
     * 金蝶已处理凭证，中台发出【红冲】指令，金蝶红冲凭证，回传红冲成功，中台删除成功，视为红冲成功，
     * 红冲成功后，业务单据的拟定状态重置为草稿，传账状态重置为待传账，
     * 会计引擎的事件分类明细里面，拟定类型改为手动，传账类型改为手动，视为红冲成功，否则不成功
     * @param param
     * @return
     */
    @Override
    @Transactional
    public ApiResult<Void> deleteByBillId(@RequestBody FinJournalSyncRpcParam param) {
        log.info("凭证删除红冲回写, 时间：{}，入参：{}", LocalDateTime.now(), JSONObject.toJSONString(param));
        if(StringUtils.isEmpty(param.getBillId())){
            return ApiResult.fail("凭证ID为空");
        }
        FinJournalRpcParam rpcParam = new FinJournalRpcParam();
        rpcParam.setBillId(param.getBillId());
        List<FinJournalDTO> journalDTOList = finJournalRepoProc.findWaitSyncJournal(rpcParam);
        // 删除凭证信息
        finJournalRepoProc.deleteByBillId(param.getBillId());
        // 更新业务单据拟定状态草稿，传帐状态待传帐
        List<Long> engineIdList = journalDTOList.stream().map(FinJournalDTO::getAccountEngineId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if(CollectionUtils.isEmpty(engineIdList)){
            return ApiResult.fail("会计引擎ID不存在");
        }
        Map<Long, FinJournalDTO> engineMap = journalDTOList.stream().collect(Collectors.toMap(FinJournalDTO::getId, t -> t));
        List<FinEventTableDTO> tableDOS = finEventTableRepoProc.findByEngineId(engineIdList);
        Map<Long, FinEventTableDTO> tableMap = tableDOS.stream().collect(Collectors.toMap(FinEventTableDTO::getEngineId, t -> t));

        //获取事件表单数据库链接
        Connection connection = null;
        try {
            connection = finAccountEngineJdbcService.getEventTableConnection(FinEventTableConvert.INSTANCE.DTOToDO(tableDOS.get(0)));
        } catch (Exception e) {
            log.error("获取事件表单数据库连接异常,事件表单:{},会计引擎ID：{}", tableDOS.get(0), e);
            return ApiResult.fail("获取事件表单数据库连接异常");
        }
        try {
            for(FinJournalDTO journalDTO : journalDTOList){
                FinEventTableDTO tableDTO = tableMap.get(journalDTO.getAccountEngineId());
                if(tableDTO != null){
                    try {
                        String updateProposedStatusSql = "update " + tableDTO.getDatabaseName() + "." + tableDTO.getMasTable()
                                + " set transfer_status = '" + "WAIT" + "', "
                                + " set proposed_status = '" + "DRAFT" + "'"
                                + " where " + tableDTO.getColumnDocNum() + " = '" + journalDTO.getDocNum() + "'";
                        PreparedStatement preparedStatement = connection.prepareStatement(updateProposedStatusSql);
                        log.info("更新传帐状态,拟定状态sql:\n{}", updateProposedStatusSql);
                        int res = preparedStatement.executeUpdate();
                        preparedStatement.close();
                    } catch (SQLException e) {
                        log.error("更新传帐状态,拟定状态异常, {}", e);
                        throw new BusinessException("更新传帐状态,拟定状态异常异常：" + e.getMessage());
                    }
                }
                // 会计引擎事件分类拟定类型，传帐类型改为手动
                if(journalDTO.getAccountEngineDetailsId() != null){
                    finAccountEngineDetailsRepoProc.updateProposedType(journalDTO.getAccountEngineDetailsId());
                }
            }
        } catch (Exception e) {
            log.error("更新传帐状态,拟定状态异常,{}", e);
            throw new BusinessException("更新传帐状态,拟定状态异常：" + e.getMessage());
        } finally {
            //释放数据库连接
            finAccountEngineJdbcService.closeConnection(connection);
        }
        return ApiResult.ok();
    }

}
