package com.elitesland.fin.application.service.excel.imp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.facade.excel.flowrepair.AccountFlowRepairImportEntity;
import com.elitesland.fin.application.facade.param.flow.AccountFlowQueryParam;
import com.elitesland.fin.application.facade.vo.flow.AccountFlowVO;
import com.elitesland.fin.application.service.flow.AccountFlowService;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.flowrepair.AccountFlowRepairDO;
import com.elitesland.fin.repo.flowrepair.AccountFlowRepairRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明: 账户流水修复重算
 * </p>
 *
 * @Author Darren
 * @Date 2023/10/26
 * @Version 1.0
 * @Content:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class AccountFlowRepairImportServiceImpl implements DataImport<AccountFlowRepairImportEntity> {
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final AccountFlowService accountFlowService;
    private final AccountFlowRepairRepo accountFlowRepairRepo;

    @Override
    public String getTmplCode() {
        return "account_flow_repair_import";

    }

    @Override
    public List<String> executeImport(List<AccountFlowRepairImportEntity> dataList, int rowId) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        //导入结果集
        List<AccountFlowRepairImportEntity> flowRepairImportList = new ArrayList<>();

        List<String> repairFlowNoList = dataList.stream().map(AccountFlowRepairImportEntity::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());

        List<AccountFlowVO> flowVOList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(repairFlowNoList)){
            AccountFlowQueryParam flowQueryParam = new AccountFlowQueryParam();
            flowQueryParam.setFlowNoList(repairFlowNoList);
            flowVOList = accountFlowService.selectListByQueryParam(flowQueryParam);
        }

        List<String> flowNoList = flowVOList.stream().map(AccountFlowVO::getFlowNo).filter(Objects::nonNull).distinct().collect(Collectors.toList());


        List<String> errorResult = new ArrayList<>();

        for (AccountFlowRepairImportEntity data : dataList) {
            List<String> errorList = new ArrayList<>();

            //导入数据是否存在重复
            int finalRowId = rowId;
            data.setRowId(rowId);

            String flowNo = data.getFlowNo();
            if (StringUtils.isBlank(flowNo)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "流水号", "不能为空"));
            } else {
                // 如果不包含流水号， 抛出错误
                if (!flowNoList.contains(flowNo)) {
                    errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "流水号", "流水号不存在，请输入正确流水号"));
                }
            }

            // 获取实际金额
            BigDecimal actualAmount = data.getActualAmount();
            if (Objects.isNull(actualAmount)) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowId, "实际金额", "不能为空"));
            }

            if (StringUtils.isNotBlank(flowNo) && Objects.nonNull(actualAmount)){
                flowRepairImportList.stream().filter(v -> Objects.equals(flowNo, v.getFlowNo())
                        && (actualAmount.compareTo(v.getActualAmount()) == 0) ).findFirst().ifPresent(flowExcel ->
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, finalRowId, "整行", "导入重复数据,与第" + flowExcel.getRowId() + "行重复"))
                );
            }

            // 如果没有错误信息 则将数据加入到结果集合中
            if (CollectionUtils.isEmpty(errorList)) {
                flowRepairImportList.add(data);
                errorResult.add(null);
            } else {
                errorResult.add(StringUtils.join(errorList, FinConstant.WRAP));
            }
            rowId++;

        }


        if (CollectionUtils.isNotEmpty(flowRepairImportList)) {

            saveData(flowRepairImportList, flowVOList);

            return CollectionUtil.isNotEmpty(errorResult) ? errorResult : null;
        } else {
            return errorResult;
        }

    }

    private void saveData(List<AccountFlowRepairImportEntity> saveList, List<AccountFlowVO> flowVOList) {
        SysUserDTO sysUserDTO = getUser();

        List<AccountFlowRepairDO> flowRepairDOList = new ArrayList<>();
        Map<String, List<AccountFlowRepairImportEntity>> saveListMap = saveList.stream().collect(Collectors.groupingBy(AccountFlowRepairImportEntity::getFlowNo));
        for (String flowNo : saveListMap.keySet()) {
            List<AccountFlowRepairImportEntity> repairFlowList = saveListMap.get(flowNo);
            AccountFlowVO accountFlowVO = flowVOList.stream().filter(flowVO -> Objects.equals(flowVO.getFlowNo(), flowNo)).findFirst().get();

            List<AccountFlowRepairDO> accountFlowRepairDOList = repairFlowList.stream().map(flowExcel -> {
                AccountFlowRepairDO flowRepairDO = new AccountFlowRepairDO();
                flowRepairDO.setFlowNo(flowExcel.getFlowNo());
                flowRepairDO.setActualAmount(flowExcel.getActualAmount());
                flowRepairDO.setAmount(accountFlowVO.getAmount());
                flowRepairDO.setOpenAccountEntityCode(null);
                flowRepairDO.setOpenAccountEntityName(accountFlowVO.getAccountHolderName());
                flowRepairDO.setAccountType(accountFlowVO.getAccountType());
                flowRepairDO.setAccountCode(accountFlowVO.getAccountCode());
                flowRepairDO.setAccountName(accountFlowVO.getAccountName());
                flowRepairDO.setSourceNo(accountFlowVO.getSourceNo());
                flowRepairDO.setRepairStatus(UdcEnum.FLOW_REPAIR_STATUS_DRAFT.getValueCode());
                flowRepairDO.setRecalculationReason(UdcEnum.FLOW_RECALCUL_REASON_10.getValueCode());
                flowRepairDO.setRecalculationUserId(sysUserDTO.getId());
                flowRepairDO.setRecalculationUser(sysUserDTO.getUsername());
                return flowRepairDO;
            }).collect(Collectors.toList());

            flowRepairDOList.addAll(accountFlowRepairDOList);
        }

        // 保存数据
        accountFlowRepairRepo.saveAll(flowRepairDOList);

    }

    private SysUserDTO getUser() {
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        if (userDetails == null) {
            throw new BusinessException("当前登陆人信息获取为空!");
        }
        if (Objects.isNull(userDetails.getUser())){
            throw new BusinessException("当前登陆人信息为空!");
        }
        return userDetails.getUser();
    }
}
