package com.elitesland.fin.provider.invperiod;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.facade.dto.accountingengine.FinAccountPeriodDTO;
import com.elitesland.fin.application.facade.dto.accountingengine.InvSobAccountPeriodDTO;
import com.elitesland.fin.application.facade.param.accountingengine.FinAccountPeriodParam;
import com.elitesland.fin.application.facade.param.accountingengine.InvSobAccountPeriodParam;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.param.invperiod.InvPeriodControlRpcParam;
import com.elitesland.fin.repo.accountingengine.FinAccountPeriodRepoProc;
import com.elitesland.fin.repo.accountingengine.InvSobAccountPeriodRepoProc;
import com.elitesland.fin.service.invperiod.InvPeriodRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


@Service
@RequiredArgsConstructor
@Slf4j
@RestController
@RequestMapping(Application.URI_PREFIX + InvPeriodRpcService.PATH)
public class InvPeriodRpcServiceImpl implements InvPeriodRpcService {

    private final InvSobAccountPeriodRepoProc invSobAccountPeriodRepoProc;
    private final FinAccountPeriodRepoProc finAccountPeriodRepoProc;

    @Override
    public ApiResult<LocalDateTime> queryInvPeriodControl(InvPeriodControlRpcParam param) {
        Assert.notNull(param, "参数不能为空");
        Assert.notEmpty(param.getOuCode(), "公司编码不能为空");
        Assert.notNull(param.getTransactionDate(), "交易日期不能为空");
        log.info("queryInvPeriodControl 查询参数{}", param);
        InvSobAccountPeriodParam queryParam = new InvSobAccountPeriodParam();
        queryParam.setOuCode(param.getOuCode());
        queryParam.setTransactionDate(param.getTransactionDate());
        List<InvSobAccountPeriodDTO> periodDTOList = invSobAccountPeriodRepoProc.query(queryParam);

        Assert.notEmpty(periodDTOList, "该公司 {} 还未设置该交易期间 {} 的库存会计期间控制", param.getOuCode(),param.getTransactionDate().toLocalDate());

        List<InvSobAccountPeriodDTO> openList = periodDTOList.stream().filter(item -> UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_OPEN.getValueCode().equals(item.getStatus())).collect(Collectors.toList());
        List<InvSobAccountPeriodDTO> closedList = periodDTOList.stream().filter(item -> UdcEnum.ACCOUNT_PERIOD_CONTROL_STATUS_CLOSED.getValueCode().equals(item.getStatus())).collect(Collectors.toList());

        // 公司和交易日期，判断交易日期所属会计期间是否关闭，未关闭，记账日期=交易日期，已关闭，记账日期=下月1号
        if (CollectionUtil.isNotEmpty(openList)) {
            return ApiResult.ok(param.getTransactionDate());
        } else {
            return ApiResult.ok(beginDayOfNextMonth(LocalDateTime.now()));
        }

    }

    // 查询会计期间设置
    private Map<String, FinAccountPeriodDTO> queryAccountPeriod(InvPeriodControlRpcParam param, List<InvSobAccountPeriodDTO> periodDTOList) {
        List<String> periodStyleList = periodDTOList.stream().map(InvSobAccountPeriodDTO::getPeriodStyle).distinct().collect(Collectors.toList());
        List<String> accountPeriodCodeList = periodDTOList.stream().map(InvSobAccountPeriodDTO::getAccountPeriodCode).distinct().collect(Collectors.toList());
        FinAccountPeriodParam periodParam = new FinAccountPeriodParam();
        periodParam.setTransactionDate(param.getTransactionDate());
        periodParam.setPeriodStyleList(periodStyleList);
        periodParam.setAccountPeriodCodeList(accountPeriodCodeList);
        List<FinAccountPeriodDTO> accountPeriodDTOList = finAccountPeriodRepoProc.findByParam(periodParam);
        Map<String, FinAccountPeriodDTO> periodDTOMap = accountPeriodDTOList.stream().collect(Collectors.toMap((item -> item.getAccountPeriodCode() + item.getPeriodStyle()), t -> t, (t1, t2) -> t1));
        return periodDTOMap;

    }

    private LocalDateTime beginDayOfNextMonth(LocalDateTime dateTime) {
        int currentYear = dateTime.getYear();
        int currentMonth = dateTime.getMonthValue();
        int nextYear, nextMonth;
        if (currentMonth == 12) {
            nextYear = currentYear + 1;
            nextMonth = 1;
        } else {
            nextYear = currentYear;
            nextMonth = currentMonth + 1;
        }
        LocalDate nextMonthFirstDay = LocalDate.of(nextYear, nextMonth, 1);
        LocalDateTime beginDayOfNextMonth = LocalDateTime.of(nextMonthFirstDay, dateTime.toLocalTime());
        return beginDayOfNextMonth;
    }
}
