package com.elitesland.fin.domain.service.saleinv;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitesland.fin.application.convert.saleinv.SaleInvConvert;
import com.elitesland.fin.application.facade.param.saleinv.SaleInvParam;
import com.elitesland.fin.application.facade.vo.saleinv.SaleInvVO;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.saleinv.*;
import com.elitesland.fin.domain.param.saleinv.SaleInvPageParam;
import com.elitesland.fin.infr.dto.saleinv.SaleInvDTO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvDtlDTO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvSumDTO;
import com.elitesland.fin.infr.dto.saleinv.SaleInvdDtlDTO;
import com.elitesland.fin.infr.factory.saleinv.SaleInvFactory;
import com.elitesland.fin.infr.repo.saleinv.*;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.domain.service.saleinv
 * @date 2022/5/6 17:00
 */
@Service
@RequiredArgsConstructor
public class SaleInvDomainServiceImpl implements SaleInvDomainService {

    public final SaleInvRepoProc saleInvRepoProc;
    public final SaleInvRepo saleInvRepo;
    public final SaleInvDtlRepoProc saleInvDtlRepoProc;
    public final SaleInvDtlRepo saleInvDtlRepo;
    public final SaleInvdDtlRepoProc saleInvdDtlRepoProc;
    public final SaleInvdDtlRepo saleInvdDtlRepo;
    public final SaleInvFactory saleInvFactory;
    public final FlexFieldUtilService flexFieldUtilService;


    private final SeqNumProvider sysNumberRuleService;

    @Override
    public PagingVO<SaleInvDTO> page(SaleInvPageParam saleInvPageParam) {
        PagingVO<SaleInvDTO> res = saleInvFactory.page(saleInvPageParam);
        return res;
    }

    @Override
    public SaleInvSumDTO selectListSum(SaleInvPageParam param){
        SaleInvSumDTO saleInvSumDTO = saleInvFactory.selectListSum(param);
        return saleInvSumDTO;
    }


    @Override
    public SaleInvDTO get(Long id) {
        SaleInvDTO res = saleInvRepoProc.get(id);
        List<SaleInvDtlDTO> saleInvDtlDTOList = saleInvDtlRepoProc.queryByMasId(id);
        res.setSaleInvDtlDTOList(saleInvDtlDTOList);
        return res;
    }

    @Override
    @Transactional
    public Long save(SaleInv saleInv) {
        if (!UdcEnum.APPLY_STATUS_DRAFT.getValueCode().equals(saleInv.getOrderState())) {
            throw new BusinessException("非草稿状态不能新增");
        }
        // 销售发票表新增
        SaleInvDO saleInvDO = saveOrUpdate(saleInv);

        return saleInvDO.getId();
    }

    @Override
    @Transactional
    public Long del(List<Long> ids) {
        ids.forEach(x -> {
            SaleInvDTO saleInvDTO = saleInvRepoProc.get(x);
            // 校验草稿状态
            if (!saleInvDTO.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException("非草稿状态不能删除操作");
            }
        });
        Long res = saleInvRepoProc.del(ids);
        // 删除明细数据
        saleInvDtlRepoProc.delByMasId(ids);
        return res;
    }

    @Override
    @Transactional
    public Long update(SaleInv saleInv) {
        // 草稿状态才能编辑
        if (!UdcEnum.APPLY_STATUS_DRAFT.getValueCode().equals(saleInv.getOrderState())) {
            throw new BusinessException("非草稿状态不能编辑");
        }
        // 销售发票表新增
        SaleInvDO saleInvDO = saveOrUpdate(saleInv);

        return saleInvDO.getId();
    }

    @Override
    @Transactional
    public Long submit(SaleInv saleInv) {
        saleInv.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        // 校验明细金额
        saleInv.checkAmt();
        // 校验单头与明细金额
        saleInv.checkAmtSum();
        SaleInvDO saleInvDO = saveOrUpdate(saleInv);
        return saleInvDO.getId();
    }

    @Override
    public List<SaleInvdDtlDTO> getInvdLists(Long masId) {
        List<SaleInvdDtlDTO> list = saleInvdDtlRepoProc.getList(masId);
        return list;
    }

    @Override
    @Transactional
    public Long updateInvInfo(SaleInvParam saleInvParam) {
        Long res = saleInvRepoProc.updateInvInfo(saleInvParam);
        return res;
    }

    @Override
    @Transactional
    public void updateWorkInfo(ProcessInfo processInfo, Long resId) {
        saleInvRepoProc.updateWorkInfo(processInfo, resId);
    }


    @Transactional
    public SaleInvDO saveOrUpdate(SaleInv saleInv) {
        // 校验单头
        saleInv.check();
        // 校验明细
        saleInv.checkDtl();
        if (CharSequenceUtil.isBlank(saleInv.getApplyNo())) {
            // 发号器-生成申请单号
            String applyNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.XXFP, null);
            saleInv.setApplyNo(applyNo);
        }
        SaleInvDO convert = SaleInvConvert.INSTANCE.convert(saleInv);
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.SALE_INV, convert);
        SaleInvDO saleInvDO = saleInvRepo.save(convert);
        // 新增应付单明细单
        // 根据应付单id删除明细行
        saleInvDtlRepoProc.delByMasId(Lists.newArrayList(saleInvDO.getId()));
        List<SaleInvDtl> saleInvDtls = saleInv.getSaleInvDtls();
        List<SaleInvDtlDO> saleInvDtlDOS = SaleInvConvert.INSTANCE.convertListDO(saleInvDtls);
        saleInvDtlDOS.stream().forEach(x -> {
            x.setMasId(saleInvDO.getId());
            saleInvDtlRepo.save(x);
        });
        // 新增已开发票的明细
        if (CollUtil.isNotEmpty(saleInv.getSaleInvdDtls())) {
            List<SaleInvdDtlDO> saleInvdDtlDOS = SaleInvConvert.INSTANCE.convertInvd(saleInv.getSaleInvdDtls());
            saleInvdDtlDOS.stream().forEach(x -> {
                x.setMasId(saleInvDO.getId());
                saleInvdDtlRepo.save(x);
            });
        }
        return saleInvDO;
    }

    @Override
    public SaleInvVO queryAmt() {
       return saleInvRepoProc.queryAmt();
    }

    /**
     * 更新是否生成应收单标记
     * @param applyNo 开票申请单号
     * @param arFlag 是否生成应收单标记
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void updateArFlagByApplyNo(String applyNo, String arFlag) {

        if (Objects.equals(arFlag,UdcEnum.AR_FLAG_TYPE_1.getValueCode())){
            saleInvRepoProc.updateArFlagByApplyNo(applyNo,true);
        }else {
            saleInvRepoProc.updateArFlagByApplyNo(applyNo,false);
        }

    }
}
