package com.elitesland.fin.infr.repo.aporder;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.support.customfield.service.impl.CustomFieldJpaServiceUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.facade.param.arorder.ApOrderRefreshParam;
import com.elitesland.fin.application.service.excel.entity.ApOrderExportEntity;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.common.WriteoffUdcEnum;
import com.elitesland.fin.domain.entity.aporder.ApOrderDO;
import com.elitesland.fin.domain.entity.aporder.QApOrderDO;
import com.elitesland.fin.domain.entity.aporder.QApOrderDtlDO;
import com.elitesland.fin.domain.entity.paymentperiod.QReceiptPaymentAgreementDO;
import com.elitesland.fin.domain.param.aporder.ApOrderPageParam;
import com.elitesland.fin.domain.param.aporder.ApOrderParam;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.dto.common.ApVerDTO;
import com.elitesland.fin.param.aporder.ApOrderSyncResultRpcParam;
import com.elitesland.fin.param.aporder.ApOrderSyncRpcParam;
import com.elitesland.fin.utils.BusinessSecurityUtil;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.*;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.infr.repo.aporder
 * @date 2022/3/16 10:38
 */
@Component
@RequiredArgsConstructor
public class ApOrderRepoProc {


    private final JPAQueryFactory jpaQueryFactory;
    private final QApOrderDO qApOrderDO = QApOrderDO.apOrderDO;


    public PagingVO<ApOrderDTO> page(ApOrderPageParam apOrderPageParam, List<Long> ids) {
        List<Predicate> predicates = new ArrayList<>();
        checkAndAddPredicates(apOrderPageParam, predicates);
        pageWhere(apOrderPageParam, predicates);
        // 来源单号明细搜索
        if (StringUtils.isNotBlank(apOrderPageParam.getSourceNoDtl()) || CollectionUtil.isNotEmpty(apOrderPageParam.getDtlSourceNoList())) {
            if (CollectionUtil.isNotEmpty(ids)) {
                predicates.add(qApOrderDO.id.in(ids));
            } else {
                return new PagingVO<>();
            }
        }
        if (CollectionUtil.isNotEmpty(apOrderPageParam.getInOutCustList())) {
            predicates.add(qApOrderDO.inOutCust.in(apOrderPageParam.getInOutCustList()));
        }

        JPAQuery<ApOrderDTO> query = select(ApOrderDTO.class)
                .where(ExpressionUtils.allOf(predicates));
        //数据权限
        BusinessSecurityUtil.where(query, ApOrderDO.class);
        apOrderPageParam.setPaging(query);
        apOrderPageParam.fillOrders(query, qApOrderDO);
        return PagingVO.<ApOrderDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    public PagingVO<ApOrderDTO> unverPage(ApOrderPageParam apOrderPageParam, List<Long> ids) {
        List<Predicate> predicates = new ArrayList<>();
        apOrderPageParam.setOrderState(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
        checkAndAddPredicates(apOrderPageParam, predicates);
        pageWhere(apOrderPageParam, predicates);
        // 来源单号明细搜索
        if (CollectionUtil.isNotEmpty(ids)) {
            predicates.add(qApOrderDO.id.in(ids));
        }
        if (CollectionUtil.isNotEmpty(apOrderPageParam.getInOutCustList())) {
            predicates.add(qApOrderDO.inOutCust.in(apOrderPageParam.getInOutCustList()));
        }
        predicates.add(qApOrderDO.orderState.ne(UdcEnum.APPLY_STATUS_VOID.getValueCode()));
        //predicates.add(qApOrderDO.orderState.ne(ids));
        JPAQuery<ApOrderDTO> query = select(ApOrderDTO.class)
                .where(ExpressionUtils.allOf(predicates));
        //数据权限
        BusinessSecurityUtil.where(query, ApOrderDO.class);
        apOrderPageParam.setPaging(query);
        apOrderPageParam.fillOrders(query, qApOrderDO);
        return PagingVO.<ApOrderDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    private void pageWhere(ApOrderPageParam apOrderPageParam, List<Predicate> predicates) {
        // 审核人
        if (StringUtils.isNotBlank(apOrderPageParam.getAuditUser())) {
            predicates.add(qApOrderDO.auditUser.like('%' + apOrderPageParam.getAuditUser() + '%'));
        }
        // 来源单据
        if (StringUtils.isNotBlank(apOrderPageParam.getCreateMode())) {
            predicates.add(qApOrderDO.createMode.eq(apOrderPageParam.getCreateMode()));
        }
        // 是否期初
        if (null != apOrderPageParam.getInitFlag()) {
            predicates.add(qApOrderDO.initFlag.eq(apOrderPageParam.getInitFlag()));
        }
        //增加扩展字段查询条件begin
        Predicate customFieldPredicate = CustomFieldJpaServiceUtil.getPredicate(apOrderPageParam.getConditions(), ApOrderDO.class);
        if (customFieldPredicate != null) {
            predicates.add(customFieldPredicate);
            //predicate= ExpressionUtils.and(predicate,customFieldPredicate);
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getTransferStatus())) {
            predicates.add(qApOrderDO.transferStatus.eq(apOrderPageParam.getTransferStatus()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getFixedAssetStatus())) {
            predicates.add(qApOrderDO.fixedAssetStatus.eq(apOrderPageParam.getFixedAssetStatus()));
        }
        //增加扩展字段查询条件end
        predicates.add(qApOrderDO.deleteFlag.eq(0));
    }

    private void checkAndAddPredicates(ApOrderPageParam apOrderPageParam, List<Predicate> predicates) {
        if (null != apOrderPageParam.getOuId()) {
            predicates.add(qApOrderDO.ouId.eq(apOrderPageParam.getOuId()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getOuCode())) {
            predicates.add(qApOrderDO.ouCode.eq(apOrderPageParam.getOuCode()));
        }
        // 币种
        if (StringUtils.isNotBlank(apOrderPageParam.getCurrCode())) {
            predicates.add(qApOrderDO.currCode.eq(apOrderPageParam.getCurrCode()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getCurrName())) {
            predicates.add(qApOrderDO.currName.eq(apOrderPageParam.getCurrName()));
        }
        // 业务日期
        if (null != apOrderPageParam.getBuDateStart() && null != apOrderPageParam.getBuDateEnd()) {
            predicates.add(qApOrderDO.buDate.between(apOrderPageParam.getBuDateStart(), apOrderPageParam.getBuDateEnd()));
        }
        // 供应商Id
        if (null != apOrderPageParam.getSuppId()) {
            predicates.add(qApOrderDO.suppId.eq(apOrderPageParam.getSuppId()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getSuppCode())) {
            predicates.add(qApOrderDO.suppCode.eq(apOrderPageParam.getSuppCode()));
        }
        // 来源系统单号
        if (StringUtils.isNotBlank(apOrderPageParam.getSourceNo())) {
            predicates.add(qApOrderDO.sourceNo.like('%' + apOrderPageParam.getSourceNo() + '%'));
        }
        if (null != apOrderPageParam.getApTypeId()) {
            predicates.add(qApOrderDO.apTypeId.eq(apOrderPageParam.getApTypeId()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getOrderState())) {
            predicates.add(qApOrderDO.orderState.eq(apOrderPageParam.getOrderState()));
        }
        // 核销状态
        if (StringUtils.isNotBlank(apOrderPageParam.getVerState())) {
            predicates.add(qApOrderDO.verState.eq(apOrderPageParam.getVerState()));
        }
        // 核销状态
        if (CollectionUtil.isNotEmpty(apOrderPageParam.getVerStateList())) {
            predicates.add(qApOrderDO.verState.in(apOrderPageParam.getVerStateList()));
        }
        if (Objects.nonNull(apOrderPageParam.getProposedStatus())) {
            predicates.add(qApOrderDO.proposedStatus.eq(apOrderPageParam.getProposedStatus()));
        }
        // 红冲状态
        if (Objects.nonNull(apOrderPageParam.getRedState())) {
            predicates.add(qApOrderDO.redState.eq(apOrderPageParam.getRedState()));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getRedSourceNo())) {
            predicates.add(qApOrderDO.redSourceNo.like('%' + apOrderPageParam.getRedSourceNo() + '%'));
        }
        // 应付单号
        if (StringUtils.isNotBlank(apOrderPageParam.getApOrderNo())) {
            predicates.add(qApOrderDO.apOrderNo.like('%' + apOrderPageParam.getApOrderNo() + '%'));
        }
        if (null != apOrderPageParam.getTotalAmtStart() && null != apOrderPageParam.getTotalAmtEnd()) {
            predicates.add(qApOrderDO.totalAmt.between(apOrderPageParam.getTotalAmtStart(), apOrderPageParam.getTotalAmtEnd()));
        }
        // 经办人
        if (StringUtils.isNotBlank(apOrderPageParam.getOperator())) {
            predicates.add(qApOrderDO.operator.like('%' + apOrderPageParam.getOperator() + '%'));
        }
        if (null != apOrderPageParam.getOperUserId()) {
            predicates.add(qApOrderDO.operUserId.eq(apOrderPageParam.getOperUserId()));
        }
        // 创建人
        if (StringUtils.isNotBlank(apOrderPageParam.getCreator())) {
            predicates.add(qApOrderDO.creator.like('%' + apOrderPageParam.getCreator() + '%'));
        }
        if (StringUtils.isNotBlank(apOrderPageParam.getProtocolCode())) {
            predicates.add(qApOrderDO.protocolCode.eq(apOrderPageParam.getProtocolCode()));
        }
        if (CollectionUtil.isNotEmpty(apOrderPageParam.getProtocolCodeList())) {
            predicates.add(qApOrderDO.protocolCode.in(apOrderPageParam.getProtocolCodeList()));
        }
        if (CollectionUtil.isNotEmpty(apOrderPageParam.getRelevanceOuCodeList())) {
            predicates.add(qApOrderDO.relevanceOuCode.in(apOrderPageParam.getRelevanceOuCodeList()));
        }
    }

    public Boolean queryByApTypeId(Long id) {
        return jpaQueryFactory.select(qApOrderDO.id)
                .from(qApOrderDO)
                .where(qApOrderDO.apTypeId.eq(id))
                .fetchCount() > 0;
    }

    public List<ApOrderDTO> queryByIds(List<Long> ids) {
        return select(ApOrderDTO.class)
                .where(qApOrderDO.id.in(ids))
                .fetch();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qApOrderDO.id,
                qApOrderDO.sourceNo,
                qApOrderDO.suppId,
                qApOrderDO.suppCode,
                qApOrderDO.suppName,
                qApOrderDO.ouCode,
                qApOrderDO.ouId,
                qApOrderDO.ouName,
                qApOrderDO.apTypeId,
                qApOrderDO.apTypeName,
                qApOrderDO.apTypeCode,
                qApOrderDO.currCode,
                qApOrderDO.currName,
                qApOrderDO.operator,
                qApOrderDO.operUserId,
                qApOrderDO.buType,
                qApOrderDO.totalAmt,
                qApOrderDO.totalCurAmt,
                qApOrderDO.buDate,
                qApOrderDO.auditUserId,
                qApOrderDO.auditUser,
                qApOrderDO.auditDate,
                qApOrderDO.orderState,
                qApOrderDO.exchangeRate,
                qApOrderDO.initFlag,
                qApOrderDO.taxFlag,
                qApOrderDO.buId,
                qApOrderDO.buCode,
                qApOrderDO.buName,
                qApOrderDO.payMentCode,
                qApOrderDO.payMentName,
                qApOrderDO.payMentId,
                qApOrderDO.createTime,
                qApOrderDO.updater,
                qApOrderDO.modifyTime,
                qApOrderDO.exclTaxAmt,
                qApOrderDO.taxAmt,
                qApOrderDO.taxCurAmt,
                qApOrderDO.apOrderNo,
                qApOrderDO.creator,
                qApOrderDO.createMode,
                qApOrderDO.remark,
                qApOrderDO.localCurrCode,
                qApOrderDO.localCurrName,
                qApOrderDO.exclTaxCurAmt,
                qApOrderDO.taxRate,
                qApOrderDO.verState,
                qApOrderDO.verAmt,
                qApOrderDO.auditDataVersion,
                qApOrderDO.procInstId,
                qApOrderDO.procInstStatus,
                qApOrderDO.approvedTime,
                qApOrderDO.submitTime,
                qApOrderDO.addrNo,
                qApOrderDO.suppAddrNo,
                qApOrderDO.redState,
                qApOrderDO.redSourceId,
                qApOrderDO.redSourceNo,
                qApOrderDO.proposedStatus,
                qApOrderDO.inOutCust,
                qApOrderDO.extensionInfo,
                qApOrderDO.protocolCode,
                qApOrderDO.relevanceOuCode,
                qApOrderDO.transferStatus,
                qApOrderDO.fixedAssetStatus,
                qApOrderDO.taxpayerType

        )).from(qApOrderDO);
    }

    public ApOrderDTO get(Long id) {
        ApOrderDTO apOrderDTO = select(ApOrderDTO.class)
                .where(qApOrderDO.id.eq(id))
                .fetchOne();
        return apOrderDTO;
    }

    public Long audit(List<Long> ids, String content, SysUserDTO user) {
        long res = jpaQueryFactory.update(qApOrderDO)
                .set(qApOrderDO.auditDate, LocalDateTime.now())
                .set(qApOrderDO.auditRejection, content)
                .set(qApOrderDO.auditUser, user.getUsername())
                .set(qApOrderDO.auditUserId, user.getId())
                .where(qApOrderDO.id.in(ids))
                .execute();
        return res;

    }

    public Long del(List<Long> ids) {
        long res = jpaQueryFactory.delete(qApOrderDO)
                .where(qApOrderDO.id.in(ids))
                .execute();
        return res;
    }

    public List<ApOrderDTO> getApOrderList(ApOrderParam apOrderParam) {
        List<Predicate> predicates = new ArrayList<>();
        // 公司
        if (null != apOrderParam.getOuId()) {
            predicates.add(qApOrderDO.ouId.eq(apOrderParam.getOuId()));
        }
        // 供应商
        if (null != apOrderParam.getSuppId()) {
            predicates.add(qApOrderDO.suppId.eq(apOrderParam.getSuppId()));
        }
        // 币种
        if (CharSequenceUtil.isNotBlank(apOrderParam.getCurrCode())) {
            predicates.add(qApOrderDO.currCode.eq(apOrderParam.getCurrCode()));
        }
        // 业务日期
        if (null != apOrderParam.getBuDateStart() && null != apOrderParam.getBuDateEnd()) {
            predicates.add(qApOrderDO.buDate.between(apOrderParam.getBuDateStart(), apOrderParam.getBuDateEnd()));
        }
        predicates.add(qApOrderDO.orderState.eq(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode()));
        List<ApOrderDTO> fetch = select(ApOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
        return fetch;
    }

    public List<ApOrderDTO> selectApOrderByParam(ApOrderParam apOrderParam) {
        List<Predicate> predicates = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(apOrderParam.getApOrderNoList())) {
            predicates.add(qApOrderDO.apOrderNo.in(apOrderParam.getApOrderNoList()));
        }

        List<ApOrderDTO> fetch = select(ApOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
        return fetch;
    }

    public ApVerDTO queryVerAmtById(Long id) {
        return jpaQueryFactory.select(Projections.bean(ApVerDTO.class,
                qApOrderDO.id,
                qApOrderDO.totalAmt,
                qApOrderDO.verAmt,
                qApOrderDO.auditDataVersion)).from(qApOrderDO).where(qApOrderDO.id.eq(id)).fetchOne();
    }

    public long updateVerAmt(ApVerDTO apVerDTO) {
        return jpaQueryFactory.update(qApOrderDO).set(qApOrderDO.verState, apVerDTO.getVerState())
                .set(qApOrderDO.verAmt, apVerDTO.getVerAmt())
                .set(qApOrderDO.auditDataVersion, apVerDTO.getAuditDataVersion() + 1)
                .where(qApOrderDO.id.eq(apVerDTO.getId())
                        .and(qApOrderDO.totalAmt.goe(apVerDTO.getVerAmt()))
                        .and(qApOrderDO.auditDataVersion.eq(apVerDTO.getAuditDataVersion())))
                .execute();
    }

    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qApOrderDO)
                .set(qApOrderDO.procInstStatus, ProcInstStatus.APPROVING)
                .set(qApOrderDO.procInstId, processInfo.getProcInstId())
                .set(qApOrderDO.submitTime, LocalDateTime.now())
                .where(qApOrderDO.id.eq(id));
        if (!Objects.equals(processInfo.getProcInstStatus(), ProcInstStatus.APPROVED)) {
            jpaUpdateClause.set(qApOrderDO.procInstStatus, ProcInstStatus.APPROVING);
        }
        jpaUpdateClause.execute();
    }

    public void updateSourceNo(String sourceNo, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qApOrderDO)
                .set(qApOrderDO.sourceNo, sourceNo)
                .where(qApOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public void setNullRedFlag(Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qApOrderDO).setNull(qApOrderDO.redState)
                .where(qApOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public void setRedFlag(Long id, Boolean flag) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qApOrderDO).set(qApOrderDO.redState, flag)
                .where(qApOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public PagingVO<ApOrderExportEntity> defaultExport(ApOrderPageParam param) {
        QReceiptPaymentAgreementDO receiptPaymentAgreementDO = QReceiptPaymentAgreementDO.receiptPaymentAgreementDO;
        List<Predicate> predicates = new ArrayList<>();
        checkAndAddPredicates(param, predicates);
        pageWhere(param, predicates);
        var apOrderDtlDO = QApOrderDtlDO.apOrderDtlDO;
        if (StringUtils.isNotBlank(param.getSourceNoDtl())) {
            predicates.add(qApOrderDO.id.in(
                    jpaQueryFactory
                            .select(apOrderDtlDO.masId)
                            .from(apOrderDtlDO)
                            .where(apOrderDtlDO.sourceNo.like("%" + param.getSourceNoDtl() + "%"))
                            .groupBy(apOrderDtlDO.masId))
            );
        }

        var query = jpaQueryFactory.select(Projections.bean(
                        ApOrderExportEntity.class,
                        qApOrderDO.sourceNo,
                        qApOrderDO.suppCode,
                        qApOrderDO.suppName,
                        qApOrderDO.ouCode,
                        qApOrderDO.ouName,
                        qApOrderDO.apTypeName,
                        qApOrderDO.apTypeCode,
                        qApOrderDO.currCode,
                        qApOrderDO.currName,
                        qApOrderDO.operator,
                        qApOrderDO.buType,
                        qApOrderDO.totalAmt,
                        qApOrderDO.totalCurAmt,
                        qApOrderDO.buDate,
                        qApOrderDO.auditUser,
                        qApOrderDO.auditDate,
                        qApOrderDO.orderState,
                        qApOrderDO.exchangeRate,
                        qApOrderDO.initFlag,
                        qApOrderDO.taxFlag,
                        qApOrderDO.buCode,
                        qApOrderDO.buName,
                        qApOrderDO.payMentCode,
                        qApOrderDO.payMentName,
                        qApOrderDO.createTime,
                        qApOrderDO.updater,
                        qApOrderDO.modifyTime,
                        qApOrderDO.exclTaxAmt,
                        qApOrderDO.taxAmt,
                        qApOrderDO.taxCurAmt,
                        qApOrderDO.apOrderNo,
                        qApOrderDO.creator,
                        qApOrderDO.createMode,
                        qApOrderDO.remark,
                        qApOrderDO.taxRate,
                        qApOrderDO.verState,
                        qApOrderDO.verAmt,
                        qApOrderDO.approvedTime,
                        qApOrderDO.submitTime,
                        qApOrderDO.addrNo,
                        qApOrderDO.suppAddrNo,
                        qApOrderDO.redState,
                        qApOrderDO.redSourceNo,
                        qApOrderDO.proposedStatus,
                        qApOrderDO.relevanceOuCode,
                        qApOrderDO.inOutCust,
                        apOrderDtlDO.itemId,
                        apOrderDtlDO.itemName,
                        apOrderDtlDO.itemCode,
                        apOrderDtlDO.itemType,
                        apOrderDtlDO.smallCateCode,
                        apOrderDtlDO.smallCateName,
                        apOrderDtlDO.uom,
                        apOrderDtlDO.uomName,
                        apOrderDtlDO.price,
                        apOrderDtlDO.exclTaxPrice,
                        apOrderDtlDO.totalAmt.as("detailTotalAmt"),
                        apOrderDtlDO.taxAmt.as("detailTaxAmt"),
                        apOrderDtlDO.exclTaxAmt.as("detailExclTaxAmt"),
                        apOrderDtlDO.taxCurAmt,
                        apOrderDtlDO.qty,
                        apOrderDtlDO.verAmt,
                        apOrderDtlDO.unVerAmt,
                        apOrderDtlDO.applyVerAmTing.as("applyVerAmt"),
                        apOrderDtlDO.sourceNo.as("detailSourceNo"),
                        apOrderDtlDO.sourceLine,
                        apOrderDtlDO.protocolCode,
                        apOrderDtlDO.invoicePriceVariance,
                        apOrderDtlDO.costPrice,
                        receiptPaymentAgreementDO.protocolName
                        ))
                .from(qApOrderDO)
                .leftJoin(apOrderDtlDO)
                .on(qApOrderDO.id.eq(apOrderDtlDO.masId))
                .leftJoin(receiptPaymentAgreementDO).on(apOrderDtlDO.protocolCode.eq(receiptPaymentAgreementDO.protocolCode)
                        .and(receiptPaymentAgreementDO.protocolType.eq(UdcEnum.AGREEMENT_PROTOCOL_TYPE_PAYMENT.getValueCode())))
                .where(ExpressionUtils.allOf(predicates));
        //数据权限
        BusinessSecurityUtil.where(query, ApOrderDO.class);
        param.fillOrders(query, qApOrderDO);
        param.setPaging(query);

        return new PagingVO<>(query.fetchCount(), query.fetch());
    }


    public List<ApOrderDTO> selectInOutCustByApOrderNo(List<String> apOrderNoList) {
        if (CollectionUtil.isEmpty(apOrderNoList)) {
            return Collections.EMPTY_LIST;
        }
        List<ApOrderDTO> apOrderDTOList = jpaQueryFactory.select(
                        Projections.bean(ApOrderDTO.class,
                                qApOrderDO.apOrderNo,
                                qApOrderDO.inOutCust,
                                qApOrderDO.relevanceOuCode))
                .from(qApOrderDO)
                .where(qApOrderDO.apOrderNo.in(apOrderNoList))
                .fetch();

        return apOrderDTOList;
    }


    public void updateVerState(String verState, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qApOrderDO)
                .set(qApOrderDO.verState, verState)
                .where(qApOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public List<ApOrderDTO> findUnVerIds(ApOrderRefreshParam param) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(ApOrderDTO.class,
                        qApOrderDO.id,
                        qApOrderDO.totalAmt,
                        qApOrderDO.verState
                ))
                .from(qApOrderDO);
        if (CollUtil.isNotEmpty(param.getDocNoList())){
            jpaQuery.where(qApOrderDO.apOrderNo.in(param.getDocNoList()));
        }
        List<String> verStates = Arrays.asList(WriteoffUdcEnum.FIN_VERIFY_STATUS_AWAIT.getValueCode(), WriteoffUdcEnum.FIN_VERIFY_STATUS_PART.getValueCode());
        jpaQuery.where(qApOrderDO.verState.in(verStates));
        return jpaQuery.fetch();
    }

    public List<ApOrderDO> queryBySourceNo(String sourceNo) {
        return jpaQueryFactory.selectFrom(qApOrderDO).where(qApOrderDO.sourceNo.eq(sourceNo)).fetch();
    }

    public List<ApOrderDO> queryBySourceNoList(List<String> sourceNoList) {
        return jpaQueryFactory.selectFrom(qApOrderDO).where(qApOrderDO.sourceNo.in(sourceNoList)).fetch();
    }

    public List<ApOrderDTO> findSyncApOrder(ApOrderSyncRpcParam apOrderParam) {
        List<Predicate> predicates = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(apOrderParam.getIdList())) {
            predicates.add(qApOrderDO.id.in(apOrderParam.getIdList()));
        }
        if (CollectionUtil.isNotEmpty(apOrderParam.getTransferStatusList())) {
            predicates.add(qApOrderDO.transferStatus.in(apOrderParam.getTransferStatusList()));
        }
        if (CollectionUtil.isNotEmpty(apOrderParam.getFixedAssetStatusList())) {
            predicates.add(qApOrderDO.fixedAssetStatus.in(apOrderParam.getFixedAssetStatusList()));
        }
        List<ApOrderDTO> fetch = select(ApOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
        return fetch;
    }

    public void updateSyncStatus(ApOrderSyncResultRpcParam param) {
        jpaQueryFactory.update(qApOrderDO)
                .set(qApOrderDO.fixedAssetStatus, param.getSyncStatus())
                .where(qApOrderDO.id.eq(param.getId())).execute();
    }
}
