package com.elitesland.fin.domain.service.payorder;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitescloud.cloudt.core.seq.SeqNumProvider;
import com.elitesland.fin.application.convert.payorder.PayOrderConvert;
import com.elitesland.fin.application.convert.payorder.PayOrderDtlConvert;
import com.elitesland.fin.application.service.excel.entity.PayOrderExportEntity;
import com.elitesland.fin.common.FinConstant;
import com.elitesland.fin.common.FinFlexFieldCodeConstant;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.aporder.ApOrder;
import com.elitesland.fin.domain.entity.payorder.PayOrder;
import com.elitesland.fin.domain.entity.payorder.PayOrderDO;
import com.elitesland.fin.domain.entity.payorder.PayOrderDtl;
import com.elitesland.fin.domain.entity.payorder.PayOrderDtlDO;
import com.elitesland.fin.domain.param.payorder.PayOrderPageParam;
import com.elitesland.fin.domain.param.payorder.PayOrderParam;
import com.elitesland.fin.infr.dto.arorder.ArOrderExcelDTO;
import com.elitesland.fin.infr.dto.common.ApVerDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDtlDTO;
import com.elitesland.fin.infr.factory.payorder.PayOrderFactory;
import com.elitesland.fin.infr.repo.payorder.PayOrderDtlRepo;
import com.elitesland.fin.infr.repo.payorder.PayOrderDtlRepoProc;
import com.elitesland.fin.infr.repo.payorder.PayOrderRepo;
import com.elitesland.fin.infr.repo.payorder.PayOrderRepoProc;
import com.elitesland.fin.rpc.pur.PurSuppOutService;
import com.elitesland.fin.utils.BeanUtils;
import com.elitesland.fin.utils.BigDecimalUtils;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcDTO;
import com.elitesland.support.provider.flexField.service.FlexFieldUtilService;
import com.elitesland.workflow.ProcessInfo;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author zhiyu.he
 * @date 2022/3/16 9:52
 */
@Service
@RequiredArgsConstructor
public class PayOrderDomainServiceImpl implements PayOrderDomainService {

    private final PayOrderRepo payOrderRepo;

    private final PayOrderRepoProc payOrderRepoProc;

    private final PayOrderFactory payOrderFactory;

    private final PayOrderDtlRepo payOrderDtlRepo;

    private final PayOrderDtlRepoProc payOrderDtlRepoProc;
    //发号器生成付款单号
    private final SeqNumProvider sysNumberRuleService;
    private final FlexFieldUtilService flexFieldUtilService;
    private final PurSuppOutService purSuppOutService;


    @Override
    public PagingVO<PayOrderDTO> page(PayOrderPageParam param) {
        if (param.getBuDateStart() == null || param.getBuDateEnd() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择业务日期!");
        }
        return payOrderFactory.payOrderPage(param);
    }

    @Override
    public PagingVO<PayOrderDTO> writeoffPage(PayOrderPageParam param) {
      /*  if (param.getBuDateStart() == null || param.getBuDateEnd() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "请选择业务日期!");
        }*/
        return payOrderFactory.writeoffPage(param);
    }

    @Override
    public PayOrderDTO queryById(Long id, Boolean flag) {
        if (flag) {
            PayOrderDTO payOrderDTO = payOrderRepoProc.queryById(id);
            List<PayOrderDtlDTO> payOrderDtlDTOS = payOrderDtlRepoProc.queryByMasId(id);
            payOrderDTO.setPayOrderDtlVOList(payOrderDtlDTOS);
            return payOrderDTO;
        } else {
            return payOrderRepoProc.queryById(id);
        }
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public List<Long> deleteByIds(List<Long> ids) {
        List<PayOrderDTO> payOrderDTOS = payOrderRepoProc.queryByIds(ids);
        payOrderDTOS.forEach(dto -> {
            if (!dto.getOrderState().equals(UdcEnum.APPLY_STATUS_DRAFT.getValueCode())) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除状态为草稿的付款单!");
            }
            //红冲原单处理
            if (Objects.nonNull(dto.getRedSourceNo())) {
                payOrderRepoProc.setRedFlag(dto.getRedSourceId(), false);
            }
        });
        payOrderRepoProc.deleteByIds(ids);
        payOrderDtlRepoProc.deleteByMasIds(ids);
        return ids;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public Long save(PayOrder payOrder) {
        // 保存逻辑
        // 无ID则为新增
        if (payOrder.getId() != null) {
            payOrder.checkOrderState();
        } else {
            //新增数据才会生成付款单号
            String payOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.FKD, null);
            payOrder.setPayOrderNo(payOrderNo);
        }
        // 校验非空
        payOrder.checkNotNull(payOrder.getCreateMode());
        // 非手工
        if (!payOrder.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_MANU.getValueCode())) {
            // 校验金额不能为0
            if (payOrder.getId() == null) {
                // 来源应付单计算统计金额
                if (payOrder.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_AP.getValueCode())) {
                    payOrder.checkDtl(false);
                    payOrder.countMoney();
                    payOrder.checkTotalMoney();
                    payOrder.setPayTypeCode("AP");
                    payOrder.setPayTypeName("标准付款");
                }
                // 来源采购订单
                if (payOrder.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_PO.getValueCode())) {
                    payOrder.setPoDef();
                    // 计算明细数据
                    payOrder.checkDtl(false);
                    // 计算单头数据
                    payOrder.countMoney();
                }
            }
        } else {
            // 校验明细是否为空  金额是否为0
            payOrder.checkTotalMoney();
        }
        //未设置状态的时候设置默认状态为草稿
        if (StrUtil.isBlank(payOrder.getOrderState())) {
            payOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        }
        // 如果来源是应付单走应付单逻辑
        if (payOrder.getVerAmt() == null) {
            // 默认核销状态
            payOrder.defaultVer();
        }
        handlePayOrder(payOrder);
        return updateOrSave(payOrder).getId();
    }

    private void handlePayOrder(PayOrder payOrder) {
        if (StringUtils.isBlank(payOrder.getInOutCust()) || StringUtils.isBlank(payOrder.getRelevanceOuCode())) {
            Map<String, PurSuppBaseRpcDTO> suppMap = purSuppOutService.findSimpleSuppMapByCodes(Collections.singletonList(payOrder.getSuppCode()));

            //供应商的内外部类型
            if (MapUtil.isNotEmpty(suppMap) && suppMap.containsKey(payOrder.getSuppCode())) {
                PurSuppBaseRpcDTO purSuppBaseRpcDTO = suppMap.get(payOrder.getSuppCode());
                if (Objects.nonNull(purSuppBaseRpcDTO)) {
                    if (StringUtils.isBlank(payOrder.getInOutCust()) && StringUtils.isNotBlank(purSuppBaseRpcDTO.getIoType())) {
                        payOrder.setInOutCust(purSuppBaseRpcDTO.getIoType());
                    }
                    //系统内供应商关联的团内公司，系统外供应商空着
                    if (Objects.equals(purSuppBaseRpcDTO.getSuppType2(), "INNER") && StringUtils.isBlank(payOrder.getRelevanceOuCode())) {
                        payOrder.setRelevanceOuCode(purSuppBaseRpcDTO.getOuCode2());
                    }
                    /*if (Objects.equals(purSuppBaseRpcDTO.getIoType(),"INSIDE") && StringUtils.isBlank(payOrder.getRelevanceOuCode())){
                        payOrder.setRelevanceOuCode(purSuppBaseRpcDTO.getOuCode2());
                    }*/
                }
            }
        }

    }

    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    @Override
    public Long submit(PayOrder payOrder) {
        //无ID则为新增  校验订单状态
        if (payOrder.getId() != null) {
            payOrder.checkOrderState();
        } else {
            //新增数据才会生成付款单号
            String payOrderNo = sysNumberRuleService.generateCode(FinConstant.FIN, FinConstant.FKD, null);
            payOrder.setPayOrderNo(payOrderNo);
        }
        if (payOrder.getVerAmt() == null) {
            //默认核销状态
            payOrder.defaultVer();
        }
        //校验表头非空项
        payOrder.checkNotNull(payOrder.getCreateMode());
        //无来源都要检验是否为0金额
        if (!payOrder.getCreateMode().equals(UdcEnum.FIN_PAY_DOC_CLS_MANU.getValueCode())) {
            //校验金额不能为0
            payOrder.checkTotalMoney();
        }
        //校验所有金额
        payOrder.checkMoney();
        //校验明细是否为空
        payOrder.checkDtl(true);
        payOrder.setOrderState(UdcEnum.APPLY_STATUS_DOING.getValueCode());
        handlePayOrder(payOrder);
        PayOrderDO payOrderDO = updateOrSave(payOrder);
        return payOrderDO.getId();
    }

    @Override
    public List<PayOrderDTO> queryByPayTypeId(List<Long> ids) {
        return payOrderRepoProc.queryByPayTypeId(ids);
    }

    @Override
    public List<PayOrderDO> queryByIds(List<Long> ids) {
        return payOrderRepo.findAllById(ids);
    }

    @Override
    public void saveAll(List<PayOrderDO> payOrderDOS) {
        payOrderRepo.saveAll(payOrderDOS);
    }

    @Override
    public List<PayOrderDTO> queryByParam(PayOrderParam payOrderParam) {
        return payOrderRepoProc.queryByParam(payOrderParam);
    }

    @Override
    public List<PayOrderDTO> selectPayOrderByParam(PayOrderParam payOrderParam) {
        return payOrderRepoProc.selectPayOrderByParam(payOrderParam);
    }

    @Override
    @Transactional(rollbackFor = {Exception.class})
    @Retryable(value = {BusinessException.class}, maxAttempts = 5, backoff = @Backoff(value = 500))
    public void updateVerAmt(Long id, BigDecimal amt) {
        ApVerDTO apVerDTO = payOrderRepoProc.queryVerAmtById(id);
        BigDecimal totalAmt = apVerDTO.getTotalAmt();
        if (apVerDTO.getVerAmt().add(amt).compareTo(totalAmt) > 0) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "当前核销金额大于付款单金额，请刷新页面重新输入");
        }
        apVerDTO.setAmtAndVerState(amt);
        long rows = payOrderRepoProc.updateVerAmt(apVerDTO);
        if (rows == 0) {
            throw new BusinessException("付款单更新已核销金额失败");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        payOrderRepoProc.updateWorkInfo(processInfo, id);
    }

    @Override
    public PagingVO<PayOrderExportEntity> exportPayOrder(PayOrderPageParam param) {
        Long count = payOrderRepoProc.countExportPayOrder(param);
        if (count > 0) {
            List<PayOrderExportEntity> payOrderDTOS = payOrderRepoProc.exportPayOrder(param);
            return new PagingVO<>(count, payOrderDTOS);
        }
        return new PagingVO<>();
    }

    private PayOrderDO updateOrSave(PayOrder payOrder) {
        //更新字段 主表信息
        PayOrderDO payOrderDO = PayOrderConvert.INSTANCE.convertToDo(payOrder);
        //先删除之前明细信息
        if (payOrder.getId() != null) {
            payOrderDtlRepoProc.deleteByMasIds(List.of(payOrder.getId()));
        } else {
            payOrderDO.setAuditDataVersion(0);
        }
        flexFieldUtilService.handFlexFieldValueFeference(FinFlexFieldCodeConstant.PAY_ORDER, payOrderDO);
        PayOrderDO save = payOrderRepo.save(payOrderDO);
        //细单信息
        List<PayOrderDtlDO> payOrderDtlDOS = PayOrderDtlConvert.INSTANCE.convertToDO(payOrder.getDtlList());

        //设置主键关联ID的值
        String sourceNo = payOrder.getSourceNo();
        payOrderDtlDOS.forEach(dtl -> {
            //获取新增后的主表ID
            dtl.setMasId(save.getId());
            if (Objects.isNull(dtl.getId())) {
                dtl.setAuditDataVersion(0);
            }
            dtl.setVerAmt(BigDecimal.ZERO);
            dtl.setUnVerAmt(dtl.getTotalAmt());
            dtl.setApplyVerAmTing(BigDecimal.ZERO);
            if (StrUtil.isNotBlank(sourceNo)) {
                dtl.setSourceNo(sourceNo);
            }
        });
        payOrderDtlRepo.saveAll(payOrderDtlDOS);
        return save;
    }

    @Override
    public PayOrder redPunchCreate(Long id) {
        PayOrderDO payOrderDO = payOrderRepo.getReferenceById(id);
        payOrderDO.setRedState(Boolean.TRUE);
        payOrderRepo.save(payOrderDO);
        PayOrder newPayOrder = BeanUtils.toBean(payOrderDO, PayOrder.class);

        newPayOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        newPayOrder.setProcInstId(null);
        newPayOrder.setProcInstStatus(null);
        newPayOrder.setRedState(null);
        newPayOrder.setRedSourceNo(payOrderDO.getPayOrderNo());
        newPayOrder.setRedSourceId(payOrderDO.getId());
        newPayOrder.setRealPayAmt(BigDecimalUtils.negate(newPayOrder.getRealPayAmt()));
        newPayOrder.setTotalAmt(BigDecimalUtils.negate(newPayOrder.getTotalAmt()));
        newPayOrder.setTotalCurAmt(BigDecimalUtils.negate(newPayOrder.getTotalCurAmt()));
        newPayOrder.setRealPayCurAmt(BigDecimalUtils.negate(newPayOrder.getRealPayCurAmt()));
        newPayOrder.setId(null);
        newPayOrder.setPayOrderNo(null);
        newPayOrder.setProposedStatus(null);
        newPayOrder.setAuditUser(null);
        newPayOrder.setApprovedTime(null);
        newPayOrder.setAuditUserId(null);
        newPayOrder.setRemark(null);

        List<PayOrderDtlDO> dtls = payOrderDtlRepo.findAllByMasId(id);
        List<PayOrderDtl> newDtls = dtls.stream().map(dtl -> {
            PayOrderDtl dtlDO = BeanUtils.toBean(dtl, PayOrderDtl.class);
            dtlDO.setId(null);
            dtlDO.setTotalAmt(BigDecimalUtils.negate(dtl.getTotalAmt()));
            dtlDO.setTotalCurAmt(BigDecimalUtils.negate(dtl.getTotalCurAmt()));
            dtlDO.setRealPayAmt(BigDecimalUtils.negate(dtl.getRealPayAmt()));
            dtlDO.setRealPayCurAmt(BigDecimalUtils.negate(dtl.getRealPayCurAmt()));
            return dtlDO;
        }).collect(Collectors.toList());
        newPayOrder.setDtlList(newDtls);
        newPayOrder.setCheck(false);
        return newPayOrder;
    }
}
