package com.elitesland.fin.utils.excel.convert;


import cn.hutool.core.collection.ConcurrentHashSet;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

public class ExcelConverterManager {
    // converter class and converter map
    private final Map<Class<? extends Converter>, Converter> converterMap = new ConcurrentHashMap<>();
    
    // The singleton object of ExcelConverterManager
    private static volatile ExcelConverterManager instance;
    
    // record which one converter cannot refresh
    private static final Set<Converter> notRequiredRefreshSet = new ConcurrentHashSet<>();
    
    // a volatile field to mark the converter whether refreshing
    private volatile long lastRefreshTime = 0L;
    
    private final static long refreshIntervalMills = 5 * 60 * 1000L;
    
    public static ExcelConverterManager instance() {
        return instance(true);
    }
    
    public static ExcelConverterManager instance(boolean refresh) {
        if(instance == null) {
            synchronized (ExcelConverterManager.class) {
                if(instance == null) {
                    instance = new ExcelConverterManager();
                }
            }
        }
        if (refresh) {
            instance.lastRefreshTime = System.currentTimeMillis();
            notRequiredRefreshSet.clear();
            return instance;
        }
        return instance;
    }
    
    public static void refresh() {
        if(instance == null) {
            return;
        }
        long currentTime;
        if((currentTime = System.currentTimeMillis()) >= instance.lastRefreshTime + refreshIntervalMills) {
            instance.lastRefreshTime = currentTime;
            notRequiredRefreshSet.clear();
        }
    }
    
    public Converter getConverter(Class<? extends Converter> clz) throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        if(converterMap.containsKey(clz)) {
            return refreshConverter(converterMap.get(clz));
        }
        
        Converter converter = initConverter(clz);
        converterMap.put(clz, converter);
        return converter;
    }
    
    private Converter initConverter(Class<? extends Converter> clz) throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        Converter converter = clz.getDeclaredConstructor().newInstance();
        converter.initBaseValue();
        notRequiredRefreshSet.add(converter);
        return converter;
    }
    
    private Converter refreshConverter(Converter converter) {
        if(notRequiredRefreshSet.contains(converter)) {
            return converter;
        }
        
        notRequiredRefreshSet.add(converter);
        converter.refreshBaseValue();
        
        return converter;
    }
}
