package com.elitesland.fin.utils.excel.convert;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;

/**
 * convert IN to OUT, when converting can use base value.
 */
public abstract class Converter<IN, OUT, BASE> extends JsonSerializer<IN> {
    private final static Logger log = LoggerFactory.getLogger(Converter.class);
    protected volatile BASE baseValue;
    /**
     * convert value(IN) to return value(OUT)
     */
    public abstract OUT convert(IN value);
    
    /**
     * initialize base value
     */
    public synchronized BASE initBaseValue() {
        return baseValue;
    }
    
    /**
     * get base value
     */
    public BASE baseValue() {
        return baseValue;
    }
    
    /**
     * refresh base value when it needs to refresh
     */
    public synchronized BASE refreshBaseValue() {
        return baseValue;
    }
    
    @Override
    public void serialize(IN value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
        ExcelConverterManager instance = ExcelConverterManager.instance(false);
        try {
            Converter converter = instance.getConverter(this.getClass());
            
            Object converted = converter.convert(value);
            if(converted instanceof String) {
                gen.writeString((String) converted);
            } else if(converted instanceof BigDecimal) {
                gen.writeString(((BigDecimal) converted).toPlainString());
            } else if (converted instanceof Boolean) {
                gen.writeString(((Boolean) converted) ? "是" : "否");
            } else {
                gen.writeString(converted.toString());
            }
        } catch (NoSuchMethodException | InvocationTargetException | InstantiationException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }
}
