package com.elitesland.fin.provider.payorder;

import cn.hutool.core.text.CharSequenceUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.exception.BusinessException;
import com.elitesland.fin.Application;
import com.elitesland.fin.application.convert.payorder.PayOrderConvert;
import com.elitesland.fin.application.convert.payorder.PayOrderDtlConvert;
import com.elitesland.fin.application.convert.payorder.SyncRecordConvert;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.payorder.PayOrder;
import com.elitesland.fin.domain.entity.payorder.PayOrderDO;
import com.elitesland.fin.domain.param.payorder.PayOrderPageParam;
import com.elitesland.fin.domain.param.payorder.SyncRecordParam;
import com.elitesland.fin.domain.service.payorder.PayOrderDomainService;
import com.elitesland.fin.domain.service.payorder.PayOrderDtlDomainService;
import com.elitesland.fin.domain.service.payorder.SyncRecordDomainService;
import com.elitesland.fin.dto.payorder.OrderAmtRpcDTO;
import com.elitesland.fin.dto.payorder.PayOrderDtlRpcDTO;
import com.elitesland.fin.dto.payorder.PayOrderRpcDTO;
import com.elitesland.fin.dto.payorder.SyncRecordRpcDTO;
import com.elitesland.fin.infr.dto.aporder.ApOrderDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDtlDTO;
import com.elitesland.fin.infr.dto.payorder.SyncRecordDTO;
import com.elitesland.fin.infr.repo.aporder.ApOrderRepoProc;
import com.elitesland.fin.infr.repo.payorder.PayOrderRepoProc;
import com.elitesland.fin.param.payorder.OrderAmtRpcParam;
import com.elitesland.fin.param.payorder.PayOrderPageRpcParam;
import com.elitesland.fin.param.payorder.PayOrderRpcParam;
import com.elitesland.fin.param.payorder.PayOrderSyncRpcParam;
import com.elitesland.fin.service.payorder.PayOrderRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.provider.payorder
 * @date 2022/5/18 13:16
 */
@RequiredArgsConstructor
@RestController
@RequestMapping(Application.URI_PREFIX + PayOrderRpcService.PATH)
@Slf4j
public class PayOrderRpcServiceImpl implements PayOrderRpcService {

    public final PayOrderDomainService payOrderDomainService;
    private final PayOrderDtlDomainService payOrderDtlDomainService;
    private final SyncRecordDomainService syncRecordDomainService;
    private final PayOrderRepoProc payOrderRepoProc;
    private final ApOrderRepoProc apOrderRepoProc;

    @Override
    public Long save(PayOrderRpcParam payOrderRpcParam) {
        if (CharSequenceUtil.isBlank(payOrderRpcParam.getCreateMode())) {
            throw new BusinessException("来源单据 createMode不能为空");
        }
        PayOrder payOrder = PayOrderConvert.INSTANCE.convertRpc(payOrderRpcParam);
        payOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        Long resId = payOrderDomainService.save(payOrder);
        return resId;
    }

    @Override
    public ApiResult<PayOrderRpcDTO> save2(PayOrderRpcParam payOrderRpcParam) {
        if (CharSequenceUtil.isBlank(payOrderRpcParam.getCreateMode())) {
            throw new BusinessException("来源单据 createMode不能为空");
        }
        PayOrder payOrder = PayOrderConvert.INSTANCE.convertRpc(payOrderRpcParam);
        payOrder.setOrderState(UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
        Long resId = payOrderDomainService.save(payOrder);

        PayOrderDTO payOrderDTO = payOrderDomainService.queryById(resId, false);
        PayOrderRpcDTO payOrderRpcDTO = PayOrderConvert.INSTANCE.convertDtoToRpc(payOrderDTO);
        return ApiResult.ok(payOrderRpcDTO);
    }

    @Override
    public ApiResult<PagingVO<PayOrderRpcDTO>> page(PayOrderPageRpcParam payOrderPageRpcParam) {
        PayOrderPageParam payOrderPageParam = PayOrderConvert.INSTANCE.convertPageParamRpc(payOrderPageRpcParam);
        PagingVO<PayOrderRpcDTO> res = PayOrderConvert.INSTANCE.convertRpcDto(payOrderDomainService.page(payOrderPageParam));
        return ApiResult.ok(res);
    }

    @Override
    public ApiResult<List<PayOrderRpcDTO>> payOrderDetails(@RequestBody PayOrderPageRpcParam payOrderPageRpcParam) {
        PayOrderPageParam payOrderPageParam = PayOrderConvert.INSTANCE.convertPageParamRpc(payOrderPageRpcParam);
        payOrderPageParam.setSize(100000);
        PagingVO<PayOrderDTO> page = payOrderDomainService.page(payOrderPageParam);
        if(page.getTotal() <= 0){
            return ApiResult.ok(new ArrayList<>());
        }
        PagingVO<PayOrderRpcDTO> res = PayOrderConvert.INSTANCE.convertRpcDto(page);
        List<PayOrderRpcDTO> records = res.getRecords();

        List<Long> idList = records.stream().map(PayOrderRpcDTO::getId).toList();
        Map<Long, List<PayOrderDtlDTO>> collect = payOrderDtlDomainService.queryByMasId(idList).stream().collect(Collectors.groupingBy(PayOrderDtlDTO::getMasId));

        SyncRecordParam recordParam = new SyncRecordParam();
        recordParam.setPayOrderIdList(idList);
        Map<Long, List<SyncRecordDTO>> recordMap = syncRecordDomainService.findSyncRecord(recordParam).stream().collect(Collectors.groupingBy(SyncRecordDTO::getPayOrderId));

        for(PayOrderRpcDTO order : records){
            List<PayOrderDtlDTO> payOrderDtlDTOS = collect.get(order.getId());
            if(CollectionUtils.isNotEmpty(payOrderDtlDTOS)){
                List<PayOrderDtlRpcDTO> dtlRpcDTOList = PayOrderDtlConvert.INSTANCE.convertRpcDto(payOrderDtlDTOS);
                order.setPayOrderDtlVOList(dtlRpcDTOList);
            }
            List<SyncRecordDTO> recordDTOList = recordMap.get(order.getId());
            if(CollectionUtils.isNotEmpty(recordDTOList)){
                List<SyncRecordRpcDTO> dtlRpcDTOList = SyncRecordConvert.INSTANCE.convertRpcDto(recordDTOList);
                order.setRecordList(dtlRpcDTOList);
            }
        }
        return ApiResult.ok(records);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Void> updateSyncStatus(PayOrderSyncRpcParam param) {
        log.info("司库同步更新付款单状态, 时间：{}，入参：{}", LocalDateTime.now(), JSONObject.toJSONString(param));
        List<PayOrderDO> payOrderDOS = payOrderDomainService.queryByIds(Collections.singletonList(param.getId()));
        if(CollectionUtils.isEmpty(payOrderDOS)){
            return ApiResult.fail("付款单不存在");
        }
        PayOrderDTO payOrderDTO = new PayOrderDTO();
        payOrderDTO.setId(param.getId());
        payOrderDTO.setSyncStatus(param.getSyncStatus());
        payOrderDTO.setSyncError(param.getSyncError());
        payOrderDTO.setOutSysKey(param.getOutSysKey());
        payOrderRepoProc.updateSyncStatus(payOrderDTO);
        if(UdcEnum.PAY_ORDER_SYNC_STATUS_SYNC_SUCCESS.getValueCode().equals(param.getSyncStatus())
            || UdcEnum.PAY_ORDER_SYNC_STATUS_SYNC_FAIL.getValueCode().equals(param.getSyncStatus())){
            // 同步成功或者失败需要记录一下同步的流水号
            if(param.getSyncRecord() == null){
                return ApiResult.fail("同步记录不存在");
            }
            SyncRecordRpcDTO syncRecord = param.getSyncRecord();
            syncRecordDomainService.save(SyncRecordConvert.INSTANCE.convert(syncRecord));
        }
        return ApiResult.ok();
    }

    @Override
    public ApiResult<OrderAmtRpcDTO> queryOrderAmtByOuIdAndSuppId(OrderAmtRpcParam param) {
        OrderAmtRpcDTO amtDTO = new OrderAmtRpcDTO();
        // 1.根据供应商和公司查询付款单，排除单据状态是【作废】的 计算总的支付金额；
        param.setFilterDocStatusList(Collections.singletonList("VOID"));
        List<PayOrderDTO> payOrderDTOS = payOrderRepoProc.queryOrderAmt(param);
        if(CollectionUtils.isEmpty(payOrderDTOS)){
            amtDTO.setPayOrderAmt(BigDecimal.ZERO);
        } else {
            BigDecimal reduce = payOrderDTOS.stream().map(PayOrderDTO::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            amtDTO.setPayOrderAmt(reduce);
        }
        // 2.根据供应商和公司查询应付单，排除单据状态是【作废】的 计算总的应付金额；
        param.setFilterDocStatusList(Collections.singletonList("VOID"));
        List<ApOrderDTO> apOrderDTOS = apOrderRepoProc.queryOrderAmt(param);
        if(CollectionUtils.isEmpty(apOrderDTOS)){
            amtDTO.setApOrderAmt(BigDecimal.ZERO);
        } else {
            BigDecimal reduce = apOrderDTOS.stream().map(ApOrderDTO::getTotalAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
            amtDTO.setApOrderAmt(reduce);
        }
        return ApiResult.ok(amtDTO);
    }
}
